#' SSP Plot: Visualization of MultSE and Sampling Effort
#'
#' Plots the relationship between MultSE and sampling effort using results from SSP simulations.
#'
#' @param xx A data frame generated by \code{\link{summary_ssp}}.
#' @param opt A vector or data matrix generated by \code{\link{ioptimum}}.
#' @param multi.site Logical. Indicates whether several sites were simulated.
#'
#' @details
#' This function visualizes the behavior of MultSE (pseudo-multivariate standard error)
#' as sampling effort increases. If simulations involve two sampling scales (e.g., sites and samples),
#' separate graphs are generated. Two shaded bands highlight sub-optimal (light grey)
#' and optimal (dark grey) improvements in precision. The graph also displays the relative gain in precision
#' (as cumulative percentage) for each level of sampling effort, compared to the lowest.
#'
#' This visualization helps identify when additional sampling effort results in diminishing returns.
#' The plot is generated using \code{\link[ggplot2:ggplot2-package]{ggplot2}} and can be further customized.
#'
#' @return A \code{\link[ggplot2:ggplot2-package]{ggplot2}} object.
#'
#' @note This is an exploratory plot and can be edited or extended using standard \code{ggplot2} functions.
#'
#' @references
#' Guerra-Castro, E.J., Cajas, J.C., Simões, N., Cruz-Motta, J.J., & Mascaró, M. (2021). SSP: an R package to estimate sampling effort in studies of ecological communities. Ecography 44(4), 561-573. doi:  \doi{10.1111/ecog.05284}
#'
#' Wickham, H. (2016). ggplot2: Elegant Graphics for Data Analysis. Springer.
#'
#' @seealso \code{\link[ggplot2:ggplot2-package]{ggplot2}}
#'
#' @examples
#' ## Single site: micromollusk from Cayo Nuevo (Yucatan, Mexico)
#' data(micromollusk)
#' par.mic <- assempar(data = micromollusk, type = "P/A", Sest.method = "average")
#' sim.mic <- simdata(par.mic, cases = 3, N = 20, sites = 1)
#' sam.mic <- sampsd(dat.sim = sim.mic, Par = par.mic, transformation = "P/A",
#'                   method = "jaccard", n = 10, m = 1, k = 3)
#' summ.mic <- summary_ssp(results = sam.mic, multi.site = FALSE)
#' opt.mic <- ioptimum(xx = summ.mic, multi.site = FALSE)
#' plot_ssp(xx = summ.mic, opt = opt.mic, multi.site = FALSE)
#'
#' ## Multiple sites: Sponges from Alacranes National Park (Yucatan, Mexico)
#' data(sponges)
#' par.spo <- assempar(data = sponges, type = "counts", Sest.method = "average")
#' sim.spo <- simdata(par.spo, cases = 3, N = 10, sites = 3)
#' sam.spo <- sampsd(dat.sim = sim.spo, Par = par.spo, transformation = "square root",
#'                   method = "bray", n = 10, m = 3, k = 3)
#' summ.spo <- summary_ssp(results = sam.spo, multi.site = TRUE)
#' opt.spo <- ioptimum(xx = summ.spo, multi.site = TRUE)
#' plot_ssp(xx = summ.spo, opt = opt.spo, multi.site = TRUE)
#'
#' @importFrom ggplot2 ggplot aes_ geom_point geom_errorbar theme_bw ylab xlab
#' @importFrom ggplot2 scale_y_continuous scale_x_continuous theme element_text element_blank
#' @importFrom ggplot2 element_rect element_line annotate facet_grid geom_rect geom_text
#' @export


plot_ssp <- function(xx, opt, multi.site) {
  if (multi.site == FALSE) {
    ul <- round(max(xx$upper) + 0.1 * max(xx$upper), 2)

    fig <- ggplot(xx, aes_(x = ~samples, y = ~mean)) +
      geom_point(size = 0.5) +
      geom_errorbar(aes_(ymin = ~lower, ymax = ~upper), size = 0.1, width = .2) +
      theme_bw(base_size = 16) +
      ylab("Multivariate pseudo SE") +
      xlab("Sampling effort (n)") +
      scale_y_continuous(breaks = seq(0.0, ul, 0.025)) +
      scale_x_continuous(breaks = seq(min(xx$samples), max(xx$samples), 1)) +
      theme(
        axis.text.x = element_text(colour = "black", size = 10),
        axis.text.y = element_text(colour = "black", size = 10),
        axis.title.x = element_text(colour = "black", size = 12),
        axis.title.y = element_text(colour = "black", size = 12),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_rect(size = 0.4),
        axis.ticks = element_line(size = 0.2)
      ) +
      annotate("rect", xmin = opt[2], xmax = opt[3], ymin = min(xx$lower),
               ymax = max(xx$upper), alpha = .5, fill = "grey10") +
      annotate("rect", xmin = opt[1], xmax = opt[2], ymin = min(xx$lower),
               ymax = max(xx$upper), alpha = .5, fill = "grey50") +
      geom_text(aes_(x = ~samples, y = ~upper + 0.01, label = ~cum), na.rm = TRUE)

    return(fig)

  } else {

    shade.opt <- data.frame(
      "xmin" = c(opt[1, 2], opt[2, 2]),
      "xmax" = c(opt[1, 3], opt[2, 3]),
      "ymin" = c(min(xx$lower), min(xx$lower)),
      "ymax" = c(max(xx$lower), max(xx$lower)),
      "sv" = c("sites", "samples")
    )

    shade.sub <- data.frame(
      "xmin" = c(opt[1, 1], opt[2, 1]),
      "xmax" = c(opt[1, 2], opt[2, 2]),
      "ymin" = c(min(xx$lower), min(xx$lower)),
      "ymax" = c(max(xx$lower), max(xx$lower)),
      "sv" = c("sites", "samples")
    )

    ul <- round(max(xx$upper) + 0.1 * max(xx$upper), 2)

    my_breaks <- function(x) {
      y <- seq(min(x), max(x), 1)
      round(y, 0)
    }

    fig <- ggplot(xx, aes_(x = ~samples, y = ~mean)) +
      geom_point() +
      geom_errorbar(aes_(ymin = ~lower, ymax = ~upper), width = .1) +
      facet_grid(. ~ sv, scales = "free_x") +
      theme_bw(base_size = 16) +
      ylab("Multivariate pseudo SE") +
      xlab("Sampling effort") +
      scale_y_continuous(breaks = seq(0.0, ul, 0.025)) +
      scale_x_continuous(breaks = my_breaks) +
      theme(
        axis.text.x = element_text(colour = "black", size = 10),
        axis.text.y = element_text(colour = "black", size = 10),
        axis.title.x = element_text(colour = "black", size = 12),
        axis.title.y = element_text(colour = "black", size = 12),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_rect(size = 0.4),
        axis.ticks = element_line(size = 0.2)
      ) +
      geom_rect(data = shade.opt, aes_(x = NULL, y = NULL,
                                       xmin = ~xmin, xmax = ~xmax,
                                       ymin = ~ymin, ymax = ~ymax),
                alpha = 0.5, fill = "grey10") +
      geom_rect(data = shade.sub, aes_(x = NULL, y = NULL,
                                       xmin = ~xmin, xmax = ~xmax,
                                       ymin = ~ymin, ymax = ~ymax),
                alpha = 0.5, fill = "grey50") +
      geom_text(aes_(x = ~samples, y = ~upper + 0.01, label = ~cum), na.rm = TRUE)

    return(fig)
  }
}
