% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/testEdges.R
\name{regressEdges}
\alias{regressEdges}
\title{Regression analysis of edges across ordered conditions}
\usage{
regressEdges(networksDF, orderedGroups, padjustMethod = "BH", minMeanEdge = 0)
}
\arguments{
\item{networksDF}{A data.frame output from \code{\link{runSCORPION}} containing
TF-target pairs as rows and network identifiers as columns.}

\item{orderedGroups}{A named list where each element is a character vector of
column names in \code{networksDF}. Names represent ordered conditions
(e.g., list(Normal = c("P31--N", "P32--N"), Border = c("P31--B", "P32--B"),
Tumor = c("P31--T", "P32--T"))). The order of list elements defines the
progression (first to last).}

\item{padjustMethod}{Character specifying the p-value adjustment method for multiple
testing correction. See \code{\link[stats]{p.adjust}} for options. Default "BH"
(Benjamini-Hochberg FDR).}

\item{minMeanEdge}{Numeric threshold for minimum mean absolute edge weight to
include in testing. Edges with mean absolute weight below this threshold are
excluded. Default 0 (no filtering).}
}
\value{
A data.frame containing:
  \itemize{
    \item{tf: Transcription factor}
    \item{target: Target gene}
    \item{slope: Regression slope (change in edge weight per condition step)}
    \item{intercept: Regression intercept}
    \item{rSquared: R-squared value (proportion of variance explained)}
    \item{fStatistic: F-statistic for the regression}
    \item{pValue: Raw p-value for the slope}
    \item{pAdj: Adjusted p-value}
    \item{meanEdge: Overall mean edge weight across all conditions}
    \item{One column per condition showing mean edge weight in that condition}
  }
}
\description{
Performs linear regression on network edges from runSCORPION output
to identify edges that show significant trends across ordered conditions (e.g.,
disease progression: Normal -> Border -> Tumor).
}
\details{
This function performs simple linear regression for each edge, modeling edge weight
as a function of an ordered categorical variable (coded as 0, 1, 2, ... for each
condition level).

The slope coefficient indicates the average change in edge weight per step along
the ordered progression. Positive slopes indicate increasing edge weights,
negative slopes indicate decreasing edge weights.

The function uses vectorized computations for efficiency with large datasets.
}
\examples{
\dontrun{
# Load test data and build networks by donor and region
# Note: T = Tumor, N = Normal, B = Border regions
data(scorpionTest)
nets <- runSCORPION(
  gexMatrix = scorpionTest$gex,
  tfMotifs = scorpionTest$tf,
  ppiNet = scorpionTest$ppi,
  cellsMetadata = scorpionTest$metadata,
  groupBy = c("donor", "region")
)

# Define ordered progression: Normal -> Border -> Tumor
normal_nets <- grep("--N$", colnames(nets), value = TRUE)
border_nets <- grep("--B$", colnames(nets), value = TRUE)
tumor_nets <- grep("--T$", colnames(nets), value = TRUE)

ordered_conditions <- list(
  Normal = normal_nets,
  Border = border_nets,
  Tumor = tumor_nets
)

# Perform regression analysis
results_regression <- regressEdges(
  networksDF = nets,
  orderedGroups = ordered_conditions
)

# View top edges with strongest trends
head(results_regression[order(results_regression$pAdj), ])

# Edges with positive slopes (increasing from N to T)
increasing <- results_regression[results_regression$pAdj < 0.05 & 
                                  results_regression$slope > 0, ]
print(paste("Edges increasing along N->B->T:", nrow(increasing)))

# Edges with negative slopes (decreasing from N to T)
decreasing <- results_regression[results_regression$pAdj < 0.05 & 
                                  results_regression$slope < 0, ]
print(paste("Edges decreasing along N->B->T:", nrow(decreasing)))

# Filter by minimum edge weight and R-squared
strong_trends <- results_regression[results_regression$pAdj < 0.05 & 
                                     results_regression$rSquared > 0.7 &
                                     abs(results_regression$meanEdge) > 0.1, ]
}
}
