## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----setup, message = FALSE---------------------------------------------------
library("tidyverse")
library("rstatix")
library("magrittr")
library("GimmeMyStats")

## -----------------------------------------------------------------------------
set.seed(123)
n <- 150 # Number of patients
clinical_data <- tibble(
    Country = sample(c("France", "Germany", "UK", "Italy", "Spain"), n, replace = TRUE),
    Age = rnorm(n, mean = 60, sd = 10),
    Sex = sample(c("Male", "Female"), n, replace = TRUE),
    Cancer_Type = sample(c("Lung", "Breast", "Colorectal", "Healthy"), n, replace = TRUE),
    Cancer_Stage = sample(1:4, n, replace = TRUE),
    Weight = rnorm(n, mean = 75, sd = 15),
    Height = rnorm(n, mean = 170, sd = 10),
    Fatigue_Score = sample(0:10, n, replace = TRUE),
    Physician_Score = sample(0:10, n, replace = TRUE),
    CRP = rnorm(n, mean = 5, sd = 2),
    IL6 = rnorm(n, mean = 10, sd = 5),
    Leukocytes = rnorm(n, mean = 6.5, sd = 2),
    Neutrophils = rnorm(n, mean = 55, sd = 10),
    Lymphocytes = rnorm(n, mean = 35, sd = 8),
    KRAS_Mutation = sample(c("Mutated", "Wild-type"), n, replace = TRUE),
    Treatment_Response = sample(c("Complete", "Partial", "None"), n, replace = TRUE)
)
head(clinical_data)

## -----------------------------------------------------------------------------
print_multinomial(select(clinical_data, "Cancer_Type"))

## -----------------------------------------------------------------------------
summary_binomial(select(clinical_data, c("KRAS_Mutation", "Sex")))

## -----------------------------------------------------------------------------
print_numeric(select(clinical_data, c("Age", "Weight", "CRP")))
summary_numeric(clinical_data$Age)

## -----------------------------------------------------------------------------
identify_outliers(clinical_data$CRP, method = "iqr")

## -----------------------------------------------------------------------------
identify_outliers(clinical_data$CRP, method = "percentiles")

## -----------------------------------------------------------------------------
identify_outliers(clinical_data$CRP, method = "hampel")

## -----------------------------------------------------------------------------
identify_outliers(clinical_data$CRP, method = "mad")

## -----------------------------------------------------------------------------
identify_outliers(select(clinical_data, CRP), method = "sd")

## -----------------------------------------------------------------------------
mcor_test(clinical_data[, c("CRP", "IL6", "Leukocytes")], method = "pearson")

## -----------------------------------------------------------------------------
mcor_test(
    clinical_data[, c("CRP", "IL6", "Leukocytes")],
    clinical_data[, c("Physician_Score", "Fatigue_Score")],
    method = "spearman",
    p.value = TRUE,
    method_adjust = "bonferroni"
)

## -----------------------------------------------------------------------------
anova_res <- anova_test(data = clinical_data, Age ~ Country)
print_test(anova_res)

## -----------------------------------------------------------------------------
kruskal_res <- kruskal_test(data = clinical_data, CRP ~ Cancer_Type)
print_test(kruskal_res)

## -----------------------------------------------------------------------------
wilcox_res <- wilcox_test(data = clinical_data, IL6 ~ KRAS_Mutation)
print_test(wilcox_res)

## -----------------------------------------------------------------------------
chi2_res <- chisq_test(table(clinical_data$Cancer_Type, clinical_data$Treatment_Response))
print_chi2_test(chi2_res)

## -----------------------------------------------------------------------------
post_hoc_chi2(clinical_data$Cancer_Type, method = "chisq")

## ----end, echo = FALSE--------------------------------------------------------
sessionInfo()

