## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(semboottools)
library(lavaan)

## -----------------------------------------------------------------------------
# Set seed for reproducibility
set.seed(1234)

# Generate data
n <- 1000
x <- runif(n) - 0.5
m <- 0.20 * x + rnorm(n)
y <- 0.17 * m + rnorm(n)
dat <- data.frame(x, y, m)

# Specify mediation model in lavaan syntax
mod <- '
  m ~ a * x
  y ~ b * m + cp * x
  ab := a * b
  total := a * b + cp
'

## -----------------------------------------------------------------------------
fit <- sem(mod, data = dat, fixed.x = FALSE)
summary(fit, ci = TRUE)
# Ensure bootstrap estimates are stored
fit <- store_boot(fit) 

## -----------------------------------------------------------------------------
# Basic usage: default settings
# Compute standardized solution with percentile bootstrap CIs
std_boot <- standardizedSolution_boot(fit)
print(std_boot)

## -----------------------------------------------------------------------------
# Basic usage: default settings
# Compute unstandardized solution with percentile bootstrap CIs
est_boot <- parameterEstimates_boot(fit)

# Print results
print(est_boot)

## ----fig.width = 6, fig.height = 3, fig.align='center'------------------------
# For estimates of user-defined parameters,
hist_qq_boot(fit, "ab", standardized = FALSE)
# For estimates in standardized solution,
hist_qq_boot(fit, "ab", standardized = TRUE)

## -----------------------------------------------------------------------------
# standardized solution
scatter_boot(fit, c("a", "b", "ab"), standardized = TRUE)
# unstandardized solution
scatter_boot(fit, c("a", "b", "ab"), standardized = FALSE)

