## ----setup, include=FALSE-----------------------------------------------------
old_ops <- options()
suppressPackageStartupMessages(library(TemporalForest))
knitr::opts_chunk$set(
  collapse   = TRUE,  
  comment    = "#>",   
  fig.width  = 7,
  fig.height = 5,
  message    = FALSE, 
  warning    = FALSE  
)
options(stringsAsFactors = FALSE)
suppressPackageStartupMessages({
  ok_wgcna <- requireNamespace("WGCNA", quietly = TRUE)
})
if (ok_wgcna && "disableWGCNAThreads" %in% getNamespaceExports("WGCNA")) {
  suppressMessages(WGCNA::disableWGCNAThreads())
}

## ----eval=FALSE---------------------------------------------------------------
# # install.packages("remotes")
# remotes::install_github("SisiShao/TemporalForest")

## -----------------------------------------------------------------------------
set.seed(11) # For reproducibility
n_subjects <- 60; n_timepoints <- 2; p <- 20

# Build X (two time points) with matching colnames
X <- replicate(n_timepoints, matrix(rnorm(n_subjects * p), n_subjects, p), simplify = FALSE)
colnames(X[[1]]) <- colnames(X[[2]]) <- paste0("V", 1:p)

# Long view and IDs
X_long <- do.call(rbind, X)
id     <- rep(seq_len(n_subjects), each = n_timepoints)
time   <- rep(seq_len(n_timepoints), times = n_subjects)

# Strong signal on V1, V2, V3 + modest subject random effect + small noise
u_subj <- rnorm(n_subjects, 0, 0.7)
eps    <- rnorm(length(id), 0, 0.08)
Y <- 4*X_long[, "V1"] + 3.5*X_long[, "V2"] + 3.2*X_long[, "V3"] +
     rep(u_subj, each = n_timepoints) + eps

# Lightweight dissimilarity to skip Stage 1 (fast on CRAN)
A <- 1 - abs(stats::cor(X_long)); diag(A) <- 0
dimnames(A) <- list(colnames(X[[1]]), colnames(X[[1]]))

## -----------------------------------------------------------------------------
# Run TemporalForest with minimal settings for vignette
tf_result <- temporal_forest(
  X = X, Y = Y, id = id, time = time,
  dissimilarity_matrix = A,       # skip WGCNA/TOM (Stage 1)
  n_features_to_select = 3,       
  n_boot_screen = 4, # Very low for quick demo
  n_boot_select =8, # Very low for quick demo
  keep_fraction_screen = 1,       # Permissive screening
  min_module_size = 2,
  alpha_screen = 0.5,             # Permissive screening
  alpha_select = 0.6
)

## -----------------------------------------------------------------------------
print(tf_result)

## -----------------------------------------------------------------------------
# Validate against ground truth
true_predictors <- c("V1", "V2", "V3")
cat("True predictors found:", sum(true_predictors %in% tf_result$top_features), 
    "out of", length(true_predictors), "\n")

## ----error=TRUE---------------------------------------------------------------
try({
# This will produce a clear error message
mat1 <- matrix(1:4, nrow=2, dimnames=list(NULL, c("A", "B")))
mat2 <- matrix(1:4, nrow=2, dimnames=list(NULL, c("A", "C")))
bad_X <- list(mat1, mat2)

TemporalForest::check_temporal_consistency(bad_X)
})

## ----citation-----------------------------------------------------------------
citation("TemporalForest")

## -----------------------------------------------------------------------------
sessionInfo()
options(old_ops)

