## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----load-package-------------------------------------------------------------
library(SimtablR)
data(epitabl)

## -----------------------------------------------------------------------------
tb(epitabl, disease)

## -----------------------------------------------------------------------------
# Distribution of Smoking Status
tb(epitabl, smoking)

## -----------------------------------------------------------------------------
# Disease status by Sex
tb(epitabl, smoking, disease)

## -----------------------------------------------------------------------------
# Disease status by Sex
tb(epitabl, smoking, disease, row, d =3)

## -----------------------------------------------------------------------------
# Showing "% (n)"
tb(epitabl, smoking, disease, row, style = "{p}% ({n})")

## -----------------------------------------------------------------------------
tb(epitabl, smoking, disease, col, test= "chsqr") #could also use test= TRUE

## -----------------------------------------------------------------------------
# Calculate Prevalence Ratio of Disease by Smoking status
tb(epitabl, smoking, disease, col, 
   test = TRUE, 
   rp,            # Calculates Prevalence Ratios; change for or to get ORs
   ref = "Never",     # Reference level for smoking group
   conf.level = 0.95) # Define the level of confidence (0.95 by default)    

## ----tb-continuous------------------------------------------------------------
tb(epitabl, age, disease, var.type = c(age = "continuous"))

## -----------------------------------------------------------------------------
tb(epitabl, systolic_bp, disease, 
   var.type = c(systolic_bp = "continuous"),
   stat.cont = "mean")

## -----------------------------------------------------------------------------
tb(epitabl, disease, smoking, col, strat = sex, test = TRUE)

## -----------------------------------------------------------------------------
# Create a table object
table <- tb(epitabl, smoking, disease, col, 
         test = TRUE, 
         rp,            
         ref = "Never",     
         conf.level = 0.95) 

library(flextable) #load the flextable package to output into word or pptx
ft <- as_flextable(table)
ft <- autofit(ft)  # Optional: Adjust column widths
ft

save_as_docx(ft, path = "Table1.docx") 
save_as_pptx(ft, path = "Table1.pptx")

## ----diag-basic---------------------------------------------------------------
results <- diag_test(
  data = epitabl,
  test = rapid_test,
  ref = lab_confirmed,
  positive = "Yes"
)

print(results)

## -----------------------------------------------------------------------------
plot(results, main = "Rapid Test Performance")

## ----diag-table---------------------------------------------------------------
metrics_table <- as.data.frame(results)
print(metrics_table)

## ----regtab-basic-------------------------------------------------------------
healthcare_model <- regtab(
  data = epitabl,
  outcomes = c("outcome1", "outcome2", "outcome3"),
  predictors = ~ age + sex + disease + smoking,
  family = poisson(link = "log"),
  robust = TRUE,
  labels = c(
    outcome1 = "Primary Care",
    outcome2 = "Specialist",
    outcome3 = "Emergency Dept")
  )
  
print(healthcare_model)

## ----regtab-logistic----------------------------------------------------------
hospital_model <- regtab(
  data = epitabl,
  outcomes = "hospitalized",
  predictors = ~ age + disease + smoking + comorbidity_score,
  family = binomial(link = "logit")
)

print(hospital_model)

## ----regtab-pvalues-----------------------------------------------------------
hospital_model_p <- regtab(
  data = epitabl,
  outcomes = "hospitalized",
  predictors = ~ age + disease + smoking + comorbidity_score,
  family = binomial(),
  p_values = TRUE
)

print(hospital_model_p)

## ----regtab-gaussian----------------------------------------------------------
clinical_model <- regtab(
  data = epitabl,
  outcomes = c("systolic_bp", "cholesterol"),
  predictors = ~ age + sex + bmi + exercise + smoking,
  family = gaussian(),
  labels = c(
    systolic_bp = "Systolic BP (mmHg)",
    cholesterol = "Cholesterol (mg/dL)"
  ),
  d = 1  # 1 decimal place
)

print(clinical_model)

## -----------------------------------------------------------------------------
# Export to CSV
write.csv(clinical_model, "Table2_Regression_Results.csv", row.names = FALSE)

