#define PERL_NO_GET_CONTEXT     /* we want efficiency */
#include "xshelper.h"

#define IsObject(sv)    (SvROK(sv) && SvOBJECT(SvRV(sv)))
#define IsArrayRef(sv)  (SvROK(sv) && !SvOBJECT(SvRV(sv)) && SvTYPE(SvRV(sv)) == SVt_PVAV)
#define IsHashRef(sv)   (SvROK(sv) && !SvOBJECT(SvRV(sv)) && SvTYPE(SvRV(sv)) == SVt_PVHV)
#define IsCodeRef(sv)   (SvROK(sv) && !SvOBJECT(SvRV(sv)) && SvTYPE(SvRV(sv)) == SVt_PVCV)
#define IsScalarRef(sv) (SvROK(sv) && !SvOBJECT(SvRV(sv)) && SvTYPE(SvRV(sv)) <= SVt_PVMG)

#define XSCON_xc_stash(a)       ( (HV*)XSCON_av_at((a), XSCON_XC_STASH) )

struct delete_ent_ctx {
    HV *hv;
    SV *key;
};

static void
delete_mutex(pTHX_ void *p)
{
    struct delete_ent_ctx *ctx = (struct delete_ent_ctx *)p;
    hv_delete_ent(ctx->hv, ctx->key, G_DISCARD, 0);
}

static void
dec_sv_refcnt(pTHX_ void *p)
{
    SV *sv = (SV *)p;
    SvREFCNT_dec(sv);
}

enum {
    XSCON_FLAG_REQUIRED             =    1,
    XSCON_FLAG_HAS_TYPE_CONSTRAINT  =    2,
    XSCON_FLAG_HAS_TYPE_COERCION    =    4,
    XSCON_FLAG_HAS_DEFAULT          =    8,
    XSCON_FLAG_NO_INIT_ARG          =   16,
    XSCON_FLAG_HAS_INIT_ARG         =   32,
    XSCON_FLAG_HAS_TRIGGER          =   64,
    XSCON_FLAG_WEAKEN               =  128,
    XSCON_FLAG_HAS_ALIASES          =  256,

    XSCON_BITSHIFT_DEFAULTS         =   16,
    XSCON_BITSHIFT_TYPES            =   24,
};

enum {
    XSCON_TYPE_BASE_ANY             =    0,
    XSCON_TYPE_BASE_DEFINED         =    1,
    XSCON_TYPE_BASE_REF             =    2,
    XSCON_TYPE_BASE_BOOL            =    3,
    XSCON_TYPE_BASE_INT             =    4,
    XSCON_TYPE_BASE_PZINT           =    5,
    XSCON_TYPE_BASE_NUM             =    6,
    XSCON_TYPE_BASE_PZNUM           =    7,
    XSCON_TYPE_BASE_STR             =    8,
    XSCON_TYPE_BASE_NESTR           =    9,
    XSCON_TYPE_BASE_CLASSNAME       =   10,
    XSCON_TYPE_BASE_OBJECT          =   12,
    XSCON_TYPE_BASE_SCALARREF       =   13,
    XSCON_TYPE_BASE_CODEREF         =   14,

    XSCON_TYPE_OTHER                =   15,

    XSCON_TYPE_ARRAYREF             =   16,
    XSCON_TYPE_HASHREF              =   32,
};

typedef struct {
    char   *name;
    I32     flags;
    char   *init_arg;

    char  **aliases;
    I32     num_aliases;

    SV     *default_sv;
    SV     *trigger_sv;
    CV     *check_cv;
    CV     *coercion_cv;
} xscon_param_t;

typedef struct {
    char *package;
    bool is_placeholder;

    xscon_param_t *params;
    I32 num_params;

    CV  **build_methods;
    I32   num_build_methods;

    bool strict_params;
    char  **allow;
    I32     num_allow;
} xscon_constructor_t;

typedef struct {
    char *package;
    bool is_placeholder;

    CV  **demolish_methods;
    I32   num_demolish_methods;
} xscon_destructor_t;

typedef struct {
    char   *slot;
    bool    has_default;
    I32     default_flags;
    SV     *default_sv;
    bool    has_check;
    I32     check_flags;
    CV     *check_cv;
    bool    has_coercion;
    CV     *coercion_cv;
} xscon_reader_t;

typedef struct {
    char *slot;
    char *method_name;
    bool has_curried;
    AV *curried;
    bool is_accessor;
    bool is_try;
} xscon_delegation_t;

xscon_constructor_t*
xscon_constructor_create(SV *sig_sv) {

    dTHX;
    dSP;

    /* Validate and dereference the top-level hashref */
    if (!SvROK(sig_sv) || SvTYPE(SvRV(sig_sv)) != SVt_PVHV) {
        croak("signature must be a hashref");
    }
    HV *sig_hv = (HV *)SvRV(sig_sv);

    SV **svp;

    /* Allocate the signature struct */
    xscon_constructor_t *sig;
    Newxz(sig, 1, xscon_constructor_t);

    /* This is not a placeholder. */
    sig->is_placeholder = FALSE;

    /* Extract package */
    svp = hv_fetchs(sig_hv, "package", 0);
    if (svp && *svp && SvOK(*svp)) {
        sig->package = savepv(SvPV_nolen(*svp));
    } else {
        sig->package = NULL;
    }

    /* Get build methods */
    {
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        XPUSHs(sv_2mortal(newSVpv(sig->package, 0)));
        PUTBACK;
        I32 count = call_pv("Class::XSConstructor::get_build_methods", G_ARRAY);
        SPAGAIN;
        if (count > 0) {
            Newxz(sig->build_methods, count, CV *);
            sig->num_build_methods = count;
            for (I32 i = count - 1; i >= 0; i--) {
                SV *sv = POPs;
                if (!SvROK(sv) || SvTYPE(SvRV(sv)) != SVt_PVCV) {
                    croak("get_build_methods must return only coderefs");
                }
                sig->build_methods[i] = (CV *)SvREFCNT_inc(SvRV(sv));
            }
        }
        else {
            sig->num_build_methods = 0;
        }
        PUTBACK;
        FREETMPS;
        LEAVE;
    }

    /* Extract strict_params */
    svp = hv_fetchs(sig_hv, "strict_params", 0);
    sig->strict_params = (svp && *svp && SvTRUE(*svp));

    /* Extract aliases (arrayref of strings) */
    svp = hv_fetchs(sig_hv, "allow", 0);
    if (sig->strict_params && svp && *svp) {
        if (!SvROK(*svp) || SvTYPE(SvRV(*svp)) != SVt_PVAV) {
            croak("allow must be an arrayref");
        }

        AV *aav = (AV *)SvRV(*svp);
        I32 na = av_count(aav);
        sig->num_allow = na;
        Newxz(sig->allow, na, char *);
        for (I32 j = 0; j < na; j++) {
            SV **asv = av_fetch(aav, j, 0);
            if (!asv || !*asv || !SvOK(*asv)) {
                croak("allow value must be a string");
            }
            sig->allow[j] = savepv(SvPV_nolen(*asv));
        }
    }

    /* Fetch and validate params arrayref */
    svp = hv_fetchs(sig_hv, "params", 0);
    if (!svp || !*svp || !SvROK(*svp) || SvTYPE(SvRV(*svp)) != SVt_PVAV) {
        croak("'params' must be an arrayref");
    }
    AV *params_av = (AV *)SvRV(*svp);

    /* Allocate the params array */
    I32 num_params = av_count(params_av);
    sig->num_params = num_params;
    Newxz(sig->params, num_params, xscon_param_t);

    /* Iterate over params array */
    for (I32 i = 0; i < num_params; i++) {

        /* Extract param hashref */
        SV **elem = av_fetch(params_av, i, 0);
        if (!elem || !*elem || !SvROK(*elem) || SvTYPE(SvRV(*elem)) != SVt_PVHV) {
            croak("params[%d] must be a hashref", i);
        }

        HV *phv = (HV *)SvRV(*elem);
        xscon_param_t *p = &sig->params[i];

        /* Extract simple scalar fields */

        /* name */
        svp = hv_fetchs(phv, "name", 0);
        if (!svp || !*svp || !SvOK(*svp)) {
            croak("params[%d]{name} is required", i);
        }
        p->name = savepv(SvPV_nolen(*svp));

        /* flags */
        svp = hv_fetchs(phv, "flags", 0);
        p->flags = (svp && *svp) ? SvIV(*svp) : 0;

        /* init_arg */
        svp = hv_fetchs(phv, "init_arg", 0);
        if (svp && *svp && SvOK(*svp)) {
            p->init_arg = savepv(SvPV_nolen(*svp));
        }
        else {
            p->init_arg = NULL;
        }

        /* Extract aliases (arrayref of strings) */
        svp = hv_fetchs(phv, "aliases", 0);
        if (svp && *svp) {
            if (!SvROK(*svp) || SvTYPE(SvRV(*svp)) != SVt_PVAV) {
                croak("aliases must be an arrayref");
            }

            AV *aav = (AV *)SvRV(*svp);
            I32 na = av_count(aav);
            p->num_aliases = na;
            Newxz(p->aliases, na, char *);
            for (I32 j = 0; j < na; j++) {
                SV **asv = av_fetch(aav, j, 0);
                if (!asv || !*asv || !SvOK(*asv)) {
                    croak("alias must be a string");
                }
                p->aliases[j] = savepv(SvPV_nolen(*asv));
            }
        }

        svp = hv_fetchs(phv, "default", 0);
        if (svp && SvOK(*svp)) {
            p->default_sv = SvREFCNT_inc(*svp);
        } else {
            p->default_sv = NULL;
        }

        svp = hv_fetchs(phv, "trigger", 0);
        if (svp && SvOK(*svp)) {
            p->trigger_sv = SvREFCNT_inc(*svp);
        } else {
            p->trigger_sv = NULL;
        }

        svp = hv_fetchs(phv, "check", 0);
        if (svp && SvOK(*svp)) {
            if (!SvROK(*svp) || SvTYPE(SvRV(*svp)) != SVt_PVCV)
                croak("check must be a coderef");
            p->check_cv = (CV *)SvREFCNT_inc(SvRV(*svp));
        }
        else {
            p->check_cv = NULL;
        }

        svp = hv_fetchs(phv, "coercion", 0);
        if (svp && SvOK(*svp)) {
            if (!SvROK(*svp) || SvTYPE(SvRV(*svp)) != SVt_PVCV)
                croak("coercion must be a coderef");
            p->coercion_cv = (CV *)SvREFCNT_inc(SvRV(*svp));
        }
        else {
            p->coercion_cv = NULL;
        }
    }
    
    return sig;
}

xscon_destructor_t*
xscon_destructor_create(char *packagename) {

    dTHX;
    dSP;

    /* Allocate the signature struct */
    xscon_destructor_t *sig;
    Newxz(sig, 1, xscon_destructor_t);

    /* This is not a placeholder. */
    sig->is_placeholder = FALSE;

    /* Extract package */
    sig->package = packagename;

    /* Get demolish methods */
    {
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        XPUSHs(sv_2mortal(newSVpv(sig->package, 0)));
        PUTBACK;
        I32 count = call_pv("Class::XSConstructor::get_demolish_methods", G_ARRAY);
        SPAGAIN;
        if (count > 0) {
            Newxz(sig->demolish_methods, count, CV *);
            sig->num_demolish_methods = count;
            for (I32 i = count - 1; i >= 0; i--) {
                SV *sv = POPs;
                if (!SvROK(sv) || SvTYPE(SvRV(sv)) != SVt_PVCV) {
                    croak("get_demolish_methods must return only coderefs");
                }
                sig->demolish_methods[i] = (CV *)SvREFCNT_inc(SvRV(sv));
            }
        }
        else {
            sig->num_demolish_methods = 0;
        }
        PUTBACK;
        FREETMPS;
        LEAVE;
    }

    return sig;
}

void
xscon_constructor_free(xscon_constructor_t *sig)
{
    dTHX;

    if (sig->params) {
        for (I32 i = 0; i < sig->num_params; i++) {
            xscon_param_t *p = &sig->params[i];
            Safefree(p->name);
            Safefree(p->init_arg);
            for (I32 j = 0; j < p->num_aliases; j++)
                Safefree(p->aliases[j]);
            Safefree(p->aliases);
            SvREFCNT_dec(p->default_sv);
            SvREFCNT_dec(p->trigger_sv);
            SvREFCNT_dec(p->check_cv);
            SvREFCNT_dec(p->coercion_cv);
        }
        Safefree(sig->params);
    }
    if (sig->allow) {
        for (I32 j = 0; j < sig->num_allow; j++)
            Safefree(sig->allow[j]);
        Safefree(sig->allow);
    }
    if (sig->build_methods) {
        for (I32 i = 0; i < sig->num_build_methods; i++) {
            if (sig->build_methods[i]) {
                SvREFCNT_dec(sig->build_methods[i]);
            }
        }
        Safefree(sig->build_methods);
    }
    Safefree(sig);
}

void
xscon_destructor_free(xscon_destructor_t *sig)
{
    dTHX;

    if (sig->demolish_methods) {
        for (I32 i = 0; i < sig->num_demolish_methods; i++) {
            if (sig->demolish_methods[i]) {
                SvREFCNT_dec(sig->demolish_methods[i]);
            }
        }
        Safefree(sig->demolish_methods);
    }
    Safefree(sig);
}

SV*
join_with_commas(AV *av) {
    dTHX;

    SV *out = newSVpvs("");
    I32 len = av_len(av) + 1;

    for (I32 i = 0; i <= len; i++) {
        SV **svp = av_fetch(av, i, 0);
        if (!svp) continue;
        if (i > 0)
            sv_catpvs(out, ", ");
        sv_catsv(out, *svp);
    }

    return out;
}

static HV*
xscon_buildargs(const char* klass, I32 ax, I32 items) {
    dTHX;
    HV* args;

    /* shift @_ */
    ax++;
    items--;

    if(items == 1){
        SV* const args_ref = ST(0);
        if(!IsHashRef(args_ref)){
            croak("Single parameters to new() must be a HASH ref");
        }
        args = newHVhv((HV*)SvRV(args_ref));
        sv_2mortal((SV*)args);
    }
    else{
        I32 i;

        if( (items % 2) != 0 ){
            croak("Odd number of parameters to new()");
        }

        args = newHV_mortal();
        for(i = 0; i < items; i += 2){
            (void)hv_store_ent(args, ST(i), newSVsv(ST(i+1)), 0U);
        }

    }
    return args;
}

static SV*
xscon_create_instance(const char* klass) {
    dTHX;
    SV* instance;
    instance = sv_bless( newRV_noinc((SV*)newHV()), gv_stashpv(klass, 1) );
    return sv_2mortal(instance);
}

static bool
_S_pv_is_integer (char* const pv) {
    dTHX;
    const char* p;
    p = &pv[0];

    /* -?[0-9]+ */
    if(*p == '-') p++;

    if (!*p) return FALSE;

    while(*p){
        if(!isDIGIT(*p)){
            return FALSE;
        }
        p++;
    }
    return TRUE;
}

static bool
_S_nv_is_integer (NV const nv) {
    dTHX;
    if(nv == (NV)(IV)nv){
        return TRUE;
    }
    else {
        char buf[64];  /* Must fit sprintf/Gconvert of longest NV */
        const char* p;
        (void)Gconvert(nv, NV_DIG, 0, buf);
        return _S_pv_is_integer(buf);
    }
}

bool
_is_class_loaded (SV* const klass ) {
    dTHX;
    HV *stash;
    GV** gvp;
    HE* he;

    if ( !SvPOKp(klass) || !SvCUR(klass) ) { /* XXX: SvPOK does not work with magical scalars */
        return FALSE;
    }

    stash = gv_stashsv( klass, FALSE );
    if ( !stash ) {
        return FALSE;
    }

    if (( gvp = (GV**)hv_fetchs(stash, "VERSION", FALSE) )) {
        if ( isGV(*gvp) && GvSV(*gvp) && SvOK(GvSV(*gvp)) ){
            return TRUE;
        }
    }

    if (( gvp = (GV**)hv_fetchs(stash, "ISA", FALSE) )) {
        if ( isGV(*gvp) && GvAV(*gvp) && av_len(GvAV(*gvp)) != -1 ) {
            return TRUE;
        }
    }

    hv_iterinit(stash);
    while (( he = hv_iternext(stash) )) {
        GV* const gv = (GV*)HeVAL(he);
        if ( isGV(gv) ) {
            if ( GvCVu(gv) ) { /* is GV and has CV */
                hv_iterinit(stash); /* reset */
                return TRUE;
            }
        }
        else if ( SvOK(gv) ) { /* is a stub or constant */
            hv_iterinit(stash); /* reset */
            return TRUE;
        }
    }
    return FALSE;
}

static bool
xscon_check_type(char* keyname, SV* const val, int flags, CV* check_cv)
{
    dTHX;
    assert(value);

    // An unknown type constraint
    // We need to dive into the isa_hash to check it
    if ( flags & XSCON_TYPE_OTHER == XSCON_TYPE_OTHER ) {
        if ( !check_cv ) {
            warn( "Type constraint check coderef gone AWOL for attribute '%s', so just assuming value passes", keyname ? keyname : "unknown" );
            return 1;
        }
        
        SV* result;

        dSP;
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 1);
        PUSHs(sv_2mortal(val));
        PUTBACK;
        count  = call_sv((SV *)check_cv, G_SCALAR);
        SPAGAIN;
        result = POPs;
        bool return_val = SvTRUE(result);
        FREETMPS;
        LEAVE;
        
        return return_val;
    }
    
    if ( flags & XSCON_TYPE_ARRAYREF ) {
        if ( !IsArrayRef(val) ) {
            return FALSE;
        }
        if ( flags == XSCON_TYPE_ARRAYREF ) {
            return TRUE;
        }
        int newflags = flags & ( XSCON_TYPE_ARRAYREF - 1 );
        AV* const av = (AV*)SvRV(val);
        I32 const len = av_len(av) + 1;
        I32 i;
        for (i = 0; i < len; i++) {
            SV* const subval = *av_fetch(av, i, TRUE);
            if ( ! xscon_check_type(NULL, subval, newflags, NULL) ) {
                return FALSE;
            }
        }
        return TRUE;
    }

    if ( flags & XSCON_TYPE_HASHREF ) {
        if ( !IsHashRef(val) ) {
            return FALSE;
        }
        // HashRef[Any] or HashRef
        if ( flags == XSCON_TYPE_HASHREF ) {
            return TRUE;
        }
        int newflags = flags & ( XSCON_TYPE_HASHREF - 1 );
        HV* const hv = (HV*)SvRV(val);
        HE* he;
        hv_iterinit(hv);
        while ((he = hv_iternext(hv))) {
            SV* const subval = hv_iterval(hv, he);
            if ( ! xscon_check_type(NULL, subval, newflags, NULL) ) {
                hv_iterinit(hv); /* reset */
                return FALSE;
            }
        }
        return TRUE;
    }
    
    switch ( flags ) {
        case XSCON_TYPE_BASE_ANY:
            return TRUE;
        case XSCON_TYPE_BASE_DEFINED:
            return SvOK(val);
        case XSCON_TYPE_BASE_REF:
            return SvOK(val) && SvROK(val);
        case XSCON_TYPE_BASE_BOOL:
            if ( SvROK(val) || isGV(val) ) {
                return FALSE;
            }
            else if ( sv_true( val ) ) {
                if ( SvPOKp(val) ) {
                    // String "1"
                    return SvCUR(val) == 1 && SvPVX(val)[0] == '1';
                }
                else if ( SvIOKp(val) ) {
                    // Integer 1
                    return SvIVX(val) == 1;
                }
                else if( SvNOKp(val) ) {
                    // Float 1.0
                    return SvNVX(val) == 1.0;
                }
                else {
                    // Another way to check for string "1"???
                    STRLEN len;
                    char* ptr = SvPV(val, len);
                    return len == 1 && ptr[0] == '1';
                }
            }
            else {
                // Any non-reference non-true value (0, undef, "", "0")
                // is a valid Bool.
                return TRUE;
            }
        case XSCON_TYPE_BASE_INT:
            if ( SvOK(val) && !SvROK(val) && !isGV(val) ) {
                if ( SvPOK(val) ) {
                    return _S_pv_is_integer( SvPVX(val) );
                }
                else if ( SvIOK(val) ) {
                    return TRUE;
                }
                else if ( SvNOK(val) ) {
                    return _S_nv_is_integer( SvNVX(val) );
                }
            }
            return FALSE;
        case XSCON_TYPE_BASE_PZINT:
            // Discard non-integers
            if ( (!SvOK(val)) || SvROK(val) || isGV(val) ) {
                return FALSE;
            }
            if ( SvPOKp(val) ){
                if ( ! _S_pv_is_integer( SvPVX(val) ) ) {
                    return FALSE;
                }
            }
            else if ( SvIOKp(val) ) {
                /* ok */
            }
            else if ( SvNOKp(val) ) {
                if ( ! _S_nv_is_integer( SvNVX(val) ) ) {
                    return FALSE;
                }
            }

            // Check that the string representation is non-empty and
            // doesn't start with a minus sign. We already checked
            // for strings that don't look like integers at all.
            STRLEN len;
            char* i = SvPVx(val, len);
            return ( (len > 0 && i[0] != '-') ? TRUE : FALSE );
        case XSCON_TYPE_BASE_NUM:
            // In Perl We Trust
            return looks_like_number(val);
        case XSCON_TYPE_BASE_PZNUM:
            if ( ! looks_like_number(val) ) {
                return FALSE;
            }
            NV numeric = SvNV(val);
            return numeric >= 0.0;
        case XSCON_TYPE_BASE_STR:
            return SvOK(val) && !SvROK(val) && !isGV(val);
        case XSCON_TYPE_BASE_NESTR:
            if ( SvOK(val) && !SvROK(val) && !isGV(val) ) {
                STRLEN l = sv_len(val);
                return ( (l==0) ? FALSE : TRUE );
            }
            return FALSE;
        case XSCON_TYPE_BASE_CLASSNAME:
            return _is_class_loaded(val);
        case 11:
            // might use later
            croak("PANIC!");
        case XSCON_TYPE_BASE_OBJECT:
            return IsObject(val);
        case XSCON_TYPE_BASE_SCALARREF:
            return IsScalarRef(val);
        case XSCON_TYPE_BASE_CODEREF:
            return IsCodeRef(val);
        case XSCON_TYPE_OTHER:
            // Should have already been checked by if block at start of function.
            croak("PANIC!");
        default:
            // Should never happen
            croak("PANIC!");
    } // switch ( flags )
}

SV*
xscon_run_default(SV *object, char* keyname, int has_common_default, SV *default_sv)
{
    dTHX;

    switch ( has_common_default ) {

        // Undef
        case 1:
            return newSV(0);

        // Number 0
        case 2:
            return newSViv(0);

        // We're number 1
        case 3:
            return newSViv(1);

        // False
        case 4:
            return &PL_sv_no;

        // True
        case 5:
            return &PL_sv_yes;

        // Empty string
        case 6:
            return newSVpvs("");

        // Empty arrayref
        case 7:
            AV *av = newAV();
            return newRV_noinc((SV*)av);

        // Empty hashref
        case 8:
            HV *hv = newHV();
            return newRV_noinc((SV*)hv);

    }

    if ( !default_sv ) {
        croak("Attribute '%s' is required, but default is AWOL", keyname);
        return &PL_sv_no;
    }

    // Coderef, call as method
    if (IsCodeRef( default_sv )) {
        dSP;
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 1);
        PUSHs(object);
        PUTBACK;
        count = call_sv((SV*)default_sv, G_SCALAR);
        SPAGAIN;
        SV* got = POPs;
        SV* val = newSVsv(got);
        FREETMPS;
        LEAVE;
        return val;
    }

    // Scalarref to the name of a builder, call as method
    if (IsScalarRef(default_sv)) {
        STRLEN len;
        SV *method_name_sv = SvRV(default_sv);
        char *method_name = SvPV(method_name_sv, len);
        dSP;
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 1);
        PUSHs(object);
        PUTBACK;
        count = call_method(method_name, G_SCALAR);
        SPAGAIN;
        SV* got = POPs;
        SV* val = newSVsv(got);
        FREETMPS;
        LEAVE;
        return val;
    }

    return newSVsv(default_sv);
}

void
xscon_run_trigger(SV *object,
                  xscon_param_t* param)
{
    dTHX;
    dSP;

    char* attr_name = param->name;
    STRLEN attr_len = strlen(attr_name);

    SV *mutexkey_sv;
    SV **svp;
    SV *value_sv;

    HV *object_hv = (HV *)SvRV(object);

    mutexkey_sv = newSV(attr_len + sizeof(":trigger_mutex") - 1);
    sv_setpvn(mutexkey_sv, attr_name, attr_len);
    sv_catpvs(mutexkey_sv, ":trigger_mutex");

    if (hv_exists_ent(object_hv, mutexkey_sv, 0)) {
        SvREFCNT_dec(mutexkey_sv);
        return;
    }

    hv_store_ent(object_hv, mutexkey_sv, newSViv(1), 0);

    ENTER;
    SAVETMPS;

    /* Ensure the key SV is released */
    SAVEDESTRUCTOR_X(dec_sv_refcnt, (void *)mutexkey_sv);

    /* Ensure the mutex is deleted on scope exit */
    struct delete_ent_ctx *ctx;
    Newxz(ctx, 1, struct delete_ent_ctx);
    ctx->hv  = object_hv;
    ctx->key = mutexkey_sv;
    SAVEDESTRUCTOR_X(delete_mutex, ctx);

    svp = hv_fetch(object_hv, attr_name, attr_len, 0);
    value_sv = svp ? *svp : &PL_sv_undef;

    SV* trigger_sv = param->trigger_sv;

    if (!SvROK(trigger_sv)) {
        PUSHMARK(SP);
        XPUSHs((SV *)object);
        XPUSHs(value_sv);
        PUTBACK;
        call_method(SvPV_nolen(trigger_sv), G_VOID);
    }
    else if (SvTYPE(SvRV(trigger_sv)) == SVt_PVCV) {
        PUSHMARK(SP);
        XPUSHs((SV *)object);
        XPUSHs(value_sv);
        PUTBACK;
        call_sv(trigger_sv, G_VOID);
    }
    else {
        croak("Unexpected trigger type");
    }

    FREETMPS;
    LEAVE;
}

int
xscon_initialize_object(const xscon_constructor_t* sig, const char* klass, SV* const object, HV* const args, bool const is_cloning)
{
    dTHX;

    assert(sig);
    assert(object);
    assert(args);

    if (sig->is_placeholder) {
        croak("Called on a placeholder");
    }

    if(mg_find((SV*)args, PERL_MAGIC_tied)){
        croak("You cannot use tied HASH reference as initializing arguments");
    }

    I32 i;
    int used = 0;

    /* copy allowed attributes */
    for (i = 0; i < sig->num_params; i++) {
        xscon_param_t *param = &sig->params[i];
        int flags = param->flags;
        char *keyname = param->name;
        int keylen = strlen(param->name);
        char *init_arg = param->init_arg;
        int init_arg_len = -1;
        if ( param->init_arg ) {
            init_arg_len = strlen(param->init_arg);
        }

        SV** valref;
        SV* val;
        bool has_value = FALSE;
        bool value_was_from_args = FALSE;

        if ( (!( flags & XSCON_FLAG_NO_INIT_ARG )) && init_arg_len >= 0 && hv_exists(args, init_arg, init_arg_len) ) {
            // Value provided in args hash
            valref = hv_fetch(args, init_arg, init_arg_len, 0);
            val = newSVsv(*valref);
            has_value = TRUE;
            value_was_from_args = TRUE;
            used++;
        }

        if ( flags & XSCON_FLAG_HAS_ALIASES ) {
            for (I32 i = 0; i < param->num_aliases; i++) {
                char *alias = param->aliases[i];
                int alias_len = strlen(alias);
                if ( hv_exists(args, alias, alias_len) ) {
                    if ( has_value ) {
                        croak("Superfluous alias used for attribute '%s': %s", keyname, alias);
                    }
                    else {
                        valref = hv_fetch(args, alias, alias_len, 0);
                        val = newSVsv(*valref);
                        has_value = TRUE;
                        value_was_from_args = TRUE;
                        used++;
                    }
                }
            }
        }

        if ( !has_value && flags & XSCON_FLAG_HAS_DEFAULT ) {
            // There is a default/builder
            has_value = TRUE;
            // Some very common defaults are worth hardcoding into the flags
            // so we won't even need to do a hash lookup to find the default
            // value.
            I32 has_common_default = ( flags >> XSCON_BITSHIFT_DEFAULTS ) & 255;
            val = xscon_run_default( object, keyname, has_common_default, param->default_sv );
        }

        if ( has_value ) {
            /* there exists an isa check */
            if ( flags & XSCON_FLAG_HAS_TYPE_CONSTRAINT ) {
                int type_flags = flags >> XSCON_BITSHIFT_TYPES;
                bool result = xscon_check_type(keyname, newSVsv(val), type_flags, param->check_cv);
            
                /* we failed type check */
                if ( !result ) {
                    if ( flags & XSCON_FLAG_HAS_TYPE_COERCION && param->coercion_cv ) {
                        SV* newval;
                        dSP;
                        int count;
                        ENTER;
                        SAVETMPS;
                        PUSHMARK(SP);
                        EXTEND(SP, 1);
                        PUSHs(val);
                        PUTBACK;
                        count  = call_sv((SV *)param->coercion_cv, G_SCALAR);
                        SPAGAIN;
                        SV* tmpval = POPs;
                        newval = newSVsv(tmpval);
                        FREETMPS;
                        LEAVE;
                        
                        bool result = xscon_check_type(keyname, newSVsv(newval), type_flags, param->check_cv);
                        if ( result ) {
                            val = newSVsv(newval);
                        }
                        else {
                            croak("Coercion result '%s' failed type constraint for '%s'", SvPV_nolen(newval), keyname);
                        }
                    }
                    else {
                        croak("Value '%s' failed type constraint for '%s'", SvPV_nolen(val), keyname);
                    }
                }
            }
            
            (void)hv_store((HV *)SvRV(object), keyname, keylen, val, 0);
            
            if ( value_was_from_args && flags & XSCON_FLAG_HAS_TRIGGER ) {
                xscon_run_trigger(object, param);
            }
            
            if ( SvROK(val) && flags & XSCON_FLAG_WEAKEN ) {
                sv_rvweaken(val);
            }
        }
        else if ( flags & XSCON_FLAG_REQUIRED ) {
            if ( flags & XSCON_FLAG_HAS_INIT_ARG && strcmp(keyname, init_arg) != 0 ) {
                croak("Attribute '%s' (init arg '%s') is required", keyname, init_arg);
            }
            else {
                croak("Attribute '%s' is required", keyname);
            }
        }
    }
    
    return used;
}

static void
xscon_buildall(const xscon_constructor_t* sig, const char* klass, SV* const object, SV* const args) {
    dTHX;
    dSP;

    assert(object);
    assert(args);

    /* __no_BUILD__ support */
    if (hv_exists((HV *)SvRV(args), "__no_BUILD__", 12)) {
        SV** val = hv_fetch((HV *)SvRV(args), "__no_BUILD__", 12, 0);
        if (SvOK(*val) && SvTRUE(*val)) {
            return;
        }
    }

    /* If we can take the fast route... */
    if ( strcmp(klass, sig->package) == 0 ) {
        if ( sig->num_build_methods <= 0 )
            return;
        for (I32 i = 0; i < sig->num_build_methods; i++) {
            CV *cv = sig->build_methods[i];
            if (!cv)
                continue;
            ENTER;
            SAVETMPS;
            PUSHMARK(SP);
            EXTEND(SP, 2);
            PUSHs(object);
            PUSHs(args);
            PUTBACK;
            call_sv((SV *)cv, G_VOID);
            FREETMPS;
            LEAVE;
        }
        return;
    }

    /* Fall back to slow route because BUILDALL called on a subclass */
    HV* const stash = gv_stashpv(sig->package, 1);
    assert(stash != NULL);
    
    SV *pkgsv = newSVpv(sig->package, 0);
    SV *klasssv = newSVpv(klass, 0);
    
    /* get cache stuff */
    SV** const globref = hv_fetch(stash, "__XSCON_BUILD", 13, 0);
    HV* buildall_hash = buildall_hash = GvHV(*globref);
    STRLEN klass_len = strlen(klass);
    SV** buildall = hv_fetch(buildall_hash, klass, klass_len, 0);
    
    if ( !buildall || !SvOK(*buildall) ) {
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 2);
        PUSHs(pkgsv);
        PUSHs(klasssv);
        PUTBACK;
        count = call_pv("Class::XSConstructor::populate_build", G_VOID);
        PUTBACK;
        FREETMPS;
        LEAVE;
        buildall = hv_fetch(buildall_hash, klass, klass_len, 0);
    }
    
    if (!buildall || !SvOK(*buildall)) {
        croak("something should have happened!");
    }
    
    if (!SvROK(*buildall)) {
        return;
    }

    AV* const builds = (AV*)SvRV(*buildall);
    I32 const len = av_len(builds) + 1;
    SV** tmp;
    SV* build;
    I32 i;

    for (i = 0; i < len; i++) {
        tmp = av_fetch(builds, i, 0);
        assert(tmp);
        build = *tmp;
        
        dSP;
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 2);
        PUSHs(object);
        PUSHs(args);
        PUTBACK;
        count = call_sv(build, G_VOID);
        PUTBACK;
        FREETMPS;
        LEAVE;
    }
}

static void
xscon_demolishall(const xscon_destructor_t* sig, const char* klass, SV* const object, I32 in_global_destruction) {
    dTHX;
    dSP;

    assert(object);

    /* If we can take the fast route... */
    if ( strcmp(klass, sig->package) == 0 ) {
        if ( sig->num_demolish_methods <= 0 )
            return;
        for (I32 i = 0; i < sig->num_demolish_methods; i++) {
            CV *cv = sig->demolish_methods[i];
            if (!cv)
                continue;
            ENTER;
            SAVETMPS;
            PUSHMARK(SP);
            EXTEND(SP, 2);
            PUSHs(object);
            XPUSHs(sv_2mortal(newSViv((IV)in_global_destruction)));
            PUTBACK;
            call_sv((SV *)cv, G_VOID);
            FREETMPS;
            LEAVE;
        }
        return;
    }

    /* Fall back to slow route because DEMOLISHALL called on a subclass */
    HV* const stash = gv_stashpv(sig->package, 1);
    assert(stash != NULL);
    
    SV *pkgsv = newSVpv(sig->package, 0);
    SV *klasssv = newSVpv(klass, 0);
    
    /* get cache stuff */
    SV** const globref = hv_fetch(stash, "__XSCON_DEMOLISH", 16, 0);
    HV* demolishall_hash = GvHV(*globref);
    
    STRLEN klass_len = strlen(klass);
    SV** demolishall = hv_fetch(demolishall_hash, klass, klass_len, 0);
    
    if ( !demolishall || !SvOK(*demolishall) ) {
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 2);
        PUSHs(pkgsv);
        PUSHs(klasssv);
        PUTBACK;
        count = call_pv("Class::XSConstructor::populate_demolish", G_VOID);
        PUTBACK;
        FREETMPS;
        LEAVE;
        
        demolishall = hv_fetch(demolishall_hash, klass, klass_len, 0);
    }
    
    if (!SvOK(*demolishall)) {
        croak("something should have happened!");
    }
    
    if (!SvROK(*demolishall)) {
        return;
    }

    AV* const demolishes = (AV*)SvRV(*demolishall);
    I32 const len = av_len(demolishes) + 1;
    SV** tmp;
    SV* demolish;
    I32 i;

    for (i = 0; i < len; i++) {
        tmp = av_fetch(demolishes, i, 0);
        assert(tmp);
        demolish = *tmp;
        
        dSP;
        int count;
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        EXTEND(SP, 2);
        PUSHs(object);
        XPUSHs(sv_2mortal(newSViv((IV)in_global_destruction)));
        PUTBACK;
        count = call_sv(demolish, G_VOID);
        PUTBACK;
        FREETMPS;
        LEAVE;
    }
}

static void
xscon_strictcon(const xscon_constructor_t* sig, const char* klassname, SV* const object, SV* const args) {
    dTHX;

    assert(object);
    assert(args);

    if ( ! sig->strict_params ) {
        return;
    }

    AV *badattrs = newAV();

    HV* argshv = (HV*)SvRV(args);
    HE* he;

    hv_iterinit(argshv);
    while ((he = hv_iternext(argshv))) {
        SV* const k = hv_iterkeysv(he);
        STRLEN k_len;
        char *k_str = SvPV(k, k_len);
        bool found = FALSE;

        I32 i;
        for (i = 0; i < sig->num_allow; i++) {
            if (k_len == strlen(sig->allow[i]) && memcmp(k_str, sig->allow[i], k_len) == 0) {
                found = TRUE;
                break;
            }
        }

        if (!found) {
            av_push(badattrs, k);
        }
    }

    I32 const badattrs_len = av_len(badattrs) + 1;
    if ( badattrs_len > 0 ) {
        SV* const badattrs_commas = join_with_commas(badattrs);
        if ( badattrs_len == 1 ) {
            croak("Found unknown attribute passed to the constructor: %s", SvPV_nolen(badattrs_commas));
        }
        else {
            croak("Found unknown attributes passed to the constructor: %s", SvPV_nolen(badattrs_commas));
        }
    }
}

MODULE = Class::XSConstructor  PACKAGE = Class::XSConstructor

BOOT:
{
    HV *stash = gv_stashpv("Class::XSConstructor", GV_ADD);

    newCONSTSUB(stash, "XSCON_FLAG_REQUIRED",             newSViv(XSCON_FLAG_REQUIRED));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_TYPE_CONSTRAINT",  newSViv(XSCON_FLAG_HAS_TYPE_CONSTRAINT));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_TYPE_COERCION",    newSViv(XSCON_FLAG_HAS_TYPE_COERCION));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_DEFAULT",          newSViv(XSCON_FLAG_HAS_DEFAULT));
    newCONSTSUB(stash, "XSCON_FLAG_NO_INIT_ARG",          newSViv(XSCON_FLAG_NO_INIT_ARG));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_INIT_ARG",         newSViv(XSCON_FLAG_HAS_INIT_ARG));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_TRIGGER",          newSViv(XSCON_FLAG_HAS_TRIGGER));
    newCONSTSUB(stash, "XSCON_FLAG_WEAKEN",               newSViv(XSCON_FLAG_WEAKEN));
    newCONSTSUB(stash, "XSCON_FLAG_HAS_ALIASES",          newSViv(XSCON_FLAG_HAS_ALIASES));

    newCONSTSUB(stash, "XSCON_BITSHIFT_DEFAULTS",         newSViv(XSCON_BITSHIFT_DEFAULTS));
    newCONSTSUB(stash, "XSCON_BITSHIFT_TYPES",            newSViv(XSCON_BITSHIFT_TYPES));

    newCONSTSUB(stash, "XSCON_TYPE_BASE_ANY",             newSViv(XSCON_TYPE_BASE_ANY));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_DEFINED",         newSViv(XSCON_TYPE_BASE_DEFINED));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_REF",             newSViv(XSCON_TYPE_BASE_REF));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_BOOL",            newSViv(XSCON_TYPE_BASE_BOOL));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_INT",             newSViv(XSCON_TYPE_BASE_INT));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_PZINT",           newSViv(XSCON_TYPE_BASE_PZINT));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_NUM",             newSViv(XSCON_TYPE_BASE_NUM));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_PZNUM",           newSViv(XSCON_TYPE_BASE_PZNUM));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_STR",             newSViv(XSCON_TYPE_BASE_STR));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_NESTR",           newSViv(XSCON_TYPE_BASE_NESTR));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_CLASSNAME",       newSViv(XSCON_TYPE_BASE_CLASSNAME));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_OBJECT",          newSViv(XSCON_TYPE_BASE_OBJECT));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_SCALARREF",       newSViv(XSCON_TYPE_BASE_SCALARREF));
    newCONSTSUB(stash, "XSCON_TYPE_BASE_CODEREF",         newSViv(XSCON_TYPE_BASE_CODEREF));

    newCONSTSUB(stash, "XSCON_TYPE_OTHER",                newSViv(XSCON_TYPE_OTHER));

    newCONSTSUB(stash, "XSCON_TYPE_ARRAYREF",             newSViv(XSCON_TYPE_ARRAYREF));
    newCONSTSUB(stash, "XSCON_TYPE_HASHREF",              newSViv(XSCON_TYPE_HASHREF));
}

void
new_object(SV* klass, ...)
CODE:
{
    dTHX;
    dSP;

    const char* klassname;
    SV* args;
    SV* object;

    xscon_constructor_t *sig = (xscon_constructor_t *) CvXSUBANY(cv).any_ptr;

    if (sig->is_placeholder) {
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        XPUSHs(sv_2mortal(newSVpv(sig->package, 0)));
        PUTBACK;
        I32 count = call_pv("Class::XSConstructor::get_metadata", G_SCALAR);
        SPAGAIN;
        SV *sv = POPs;
        xscon_constructor_free(sig);
        sig = xscon_constructor_create(sv);
        sig->is_placeholder = FALSE;
        PUTBACK;
        FREETMPS;
        LEAVE;

        CvXSUBANY(cv).any_ptr = sig;
    }

    /* $klassname = shift */
    klassname = SvROK(klass) ? sv_reftype(SvRV(klass), 1) : SvPV_nolen_const(klass);

    /* $args = ref($_[0]) eq 'HASH' ? %{+shift} : @_ */
    args = newRV_inc((SV*)xscon_buildargs(klassname, ax, items));
    sv_2mortal(args);

    /* $object = bless( {}, $klassname ); */
    object = xscon_create_instance(klassname);

    /* Initialize parameters */
    int used = xscon_initialize_object(sig, klassname, object, (HV*)SvRV(args), FALSE);

    /* Call BUILD methods */
    xscon_buildall(sig, klassname, object, args);

    /* Strict constructor */
    //if ( HvUSEDKEYS(args) > used ) {
        xscon_strictcon(sig, klassname, object, args);
    //}

    /* return $object */
    ST(0) = object; /* because object is mortal, we should return it as is */
    XSRETURN(1);
}

void
destroy(SV* object, ...)
CODE:
{
    dTHX;
    xscon_destructor_t *sig = (xscon_destructor_t *) CvXSUBANY(cv).any_ptr;
    if (sig->is_placeholder) {
        char* pkg = sig->package;
        xscon_destructor_free(sig);
        sig = xscon_destructor_create(pkg);
        sig->is_placeholder = FALSE;
        CvXSUBANY(cv).any_ptr = sig;
    }
    const char* klassname = SvROK(object) ? sv_reftype(SvRV(object), 1) : SvPV_nolen_const(object);
    I32 in_global_destruction = PL_dirty;
    xscon_demolishall(sig, klassname, object, in_global_destruction);
    XSRETURN(0);
}

void
reader(SV* object, ...)
CODE:
{
    dTHX;
    dSP;
    
    xscon_reader_t *sig = (xscon_reader_t *) CvXSUBANY(cv).any_ptr;
    
    HV *object_hv = (HV *)SvRV(object);
    STRLEN slotlen = strlen(sig->slot);
    
    if ( sig->has_default && !hv_exists(object_hv, sig->slot, slotlen) ) {
        SV* val = xscon_run_default(object, sig->slot, sig->default_flags, sig->default_sv);
        if ( sig->has_check && !xscon_check_type(sig->slot, newSVsv(val), sig->check_flags, sig->check_cv) ) {
            if ( sig->has_coercion ) {
                SV* newval;
                int count;
                dSP;
                ENTER;
                SAVETMPS;
                PUSHMARK(SP);
                EXTEND(SP, 1);
                PUSHs(val);
                PUTBACK;
                count  = call_sv((SV *)sig->coercion_cv, G_SCALAR);
                SPAGAIN;
                SV* tmpval = POPs;
                newval = newSVsv(tmpval);
                FREETMPS;
                LEAVE;
                
                if ( xscon_check_type(sig->slot, newSVsv(newval), sig->check_flags, sig->check_cv) ) {
                    val = newSVsv(newval);
                }
                else {
                    croak("Coercion result '%s' failed type constraint for '%s'", SvPV_nolen(newval), sig->slot);
                }
            }
            else {
                croak("Value '%s' failed type constraint for '%s'", SvPV_nolen(val), sig->slot);
            }
        }
        (void)hv_store(object_hv, sig->slot, slotlen, val, 0);
    }
    
    SV** svp = hv_fetch(object_hv, sig->slot, slotlen, 0);
    ST(0) = svp ? newSVsv(*svp) : &PL_sv_undef;
    XSRETURN(1);
}

void
delegation(SV* object, ...)
CODE:
{
    dTHX;
    dSP;
    
    I32 gimme = GIMME_V;
    bool inc = FALSE;
    
    I32 nargs = items - 1;
    AV *args = newAV();
    for ( I32 i = 0; i < nargs; i++ ) {
        SV *arg = ST( i + 1 );
        av_push( args, arg );
    }
    
    xscon_delegation_t *sig = (xscon_delegation_t *) CvXSUBANY(cv).any_ptr;
    
    /* Get handler object */
    SV* handler;
    if ( sig->is_accessor ) {
        ENTER;
        SAVETMPS;
        
        PUSHMARK(SP);
        XPUSHs(object);
        PUTBACK;
        
        I32 count = call_method(sig->slot, G_SCALAR | G_EVAL);
        if (SvTRUE(ERRSV)) croak(NULL);
        
        SPAGAIN;
        handler = POPs;
        if ( handler != &PL_sv_undef ) {
            SvREFCNT_inc(handler);
            inc = TRUE;
        }
        
        PUTBACK;
        FREETMPS;
        LEAVE;
    }
    else {
        HV *object_hv = (HV *)SvRV(object);
        STRLEN slotlen = strlen(sig->slot);
        SV** svp = hv_fetch(object_hv, sig->slot, slotlen, 0);
        handler = svp ? *svp : &PL_sv_undef;
        if ( handler != &PL_sv_undef ) {
            SvREFCNT_inc(handler);
            inc = TRUE;
        }
    }
    
    if ( !IsObject(handler) ) {
        if ( sig->is_try ) {
            ST(0) = &PL_sv_undef;
            XSRETURN(1);
            return;
        }
        croak(
            "Expected blessed object to delegate to; got %s",
            ( handler == &PL_sv_undef ) ? "undef" : SvPV_nolen(handler)
        );
    }
    
    SP = MARK;
    
    ENTER;
    SAVETMPS;
    
    PUSHMARK(SP);
    XPUSHs(handler);
    
    /* add curried arguments */
    if ( sig->has_curried ) {
        I32 n = av_count(sig->curried);
        for (I32 i = 0; i < n; i++) {
            SV **svp = av_fetch(sig->curried, i, 0);
            XPUSHs( svp ? *svp : &PL_sv_undef );
        }
    }
    
    /* forward all arguments except $object */
    for (I32 i = 0; i < nargs; i++) {
        SV **svp = av_fetch(args, i, 0);
        XPUSHs( svp ? *svp : &PL_sv_undef );
    }
    
    PUTBACK;
    
    I32 count = call_method(sig->method_name, gimme);
    LEAVE;
    
    if (inc) SvREFCNT_dec(handler);
    
    /* Do not SPAGAIN or POPs: return values left on stack! */
    XSRETURN(count);
}

void
install_constructor(char* name)
CODE:
{
    dTHX;
    CV *cv = newXS(name, XS_Class__XSConstructor_new_object, (char*)__FILE__);
    if (cv == NULL)
        croak("ARG! Something went really wrong while installing a new XSUB!");
    
    char *full = savepv(name);
    const char *last = NULL;
    for (const char *p = full; (p = strstr(p, "::")); p += 2) {
        last = p;
    }
    char *pkg;
    if (last) {
        size_t len = (size_t)(last - full);
        pkg = (char *)malloc(len + 1);
        memcpy(pkg, full, len);
        pkg[len] = '\0';
    } else {
        pkg = strdup("");
    }
    
    xscon_constructor_t *sig;
    Newxz(sig, 1, xscon_constructor_t);
    sig->package = savepv(pkg);
    sig->is_placeholder = TRUE;
    
    CvXSUBANY(cv).any_ptr = sig;
}

void
install_destructor(char* name)
CODE:
{
    dTHX;
    CV *cv = newXS(name, XS_Class__XSConstructor_destroy, (char*)__FILE__);
    if (cv == NULL)
        croak("ARG! Something went really wrong while installing a new XSUB!");
    
    char *full = savepv(name);
    const char *last = NULL;
    for (const char *p = full; (p = strstr(p, "::")); p += 2) {
        last = p;
    }
    char *pkg;
    if (last) {
        size_t len = (size_t)(last - full);
        pkg = (char *)malloc(len + 1);
        memcpy(pkg, full, len);
        pkg[len] = '\0';
    } else {
        pkg = strdup("");
    }
    
    xscon_destructor_t *sig;
    Newxz(sig, 1, xscon_destructor_t);
    sig->package = savepv(pkg);
    sig->is_placeholder = TRUE;
    CvXSUBANY(cv).any_ptr = sig;
}

void
install_delegation(char *name, char *slot, char *method_name, SV *curried, bool is_accessor, bool is_try)
CODE:
{
    dTHX;
    CV *cv = newXS(name, XS_Class__XSConstructor_delegation, (char*)__FILE__);
    if (cv == NULL)
        croak("ARG! Something went really wrong while installing a new XSUB!");
    
    xscon_delegation_t *sig;
    Newxz(sig, 1, xscon_delegation_t);
    sig->slot = savepv(slot);
    sig->is_try = is_try;
    sig->is_accessor = is_accessor;
    sig->method_name = savepv(method_name);
    
    if (!curried || !SvROK(curried) || SvTYPE(SvRV(curried)) != SVt_PVAV) {
        sig->has_curried = FALSE;
        sig->curried = NULL;
    }
    else {
        AV *curried_av = (AV *)SvRV(curried);
        sig->has_curried = TRUE;
        sig->curried = (AV *)SvREFCNT_inc((SV *)curried_av);
    }
    
    CvXSUBANY(cv).any_ptr = sig;
}

void
install_reader(char *name, char *slot, bool has_default, int default_flags, SV* default_sv, int check_flags, SV* check, SV* coercion)
CODE:
{
    dTHX;
    CV *cv = newXS(name, XS_Class__XSConstructor_reader, (char*)__FILE__);
    if (cv == NULL)
        croak("ARG! Something went really wrong while installing a new XSUB!");
    
    xscon_reader_t *sig;
    Newxz(sig, 1, xscon_reader_t);
    sig->slot           = savepv(slot);
    sig->has_default    = has_default;
    sig->default_flags  = default_flags;
    sig->default_sv     = SvREFCNT_inc(default_sv);
    sig->check_flags    = check_flags;
    
    if (check && IsCodeRef(check)) {
        sig->has_check = TRUE;
        sig->check_cv = (CV *)SvREFCNT_inc(SvRV(check));
    }
    else {
        sig->has_check = FALSE;
        sig->check_cv = NULL;
    }
    
    if (coercion && IsCodeRef(coercion)) {
        sig->has_coercion = TRUE;
        sig->coercion_cv = (CV *)SvREFCNT_inc(SvRV(coercion));
    }
    else {
        sig->has_coercion = FALSE;
        sig->coercion_cv = NULL;
    }

    CvXSUBANY(cv).any_ptr = sig;
}
