package Venus;

use 5.018;

use strict;
use warnings;

# VERSION

our $VERSION = '2.50';

# AUTHORITY

our $AUTHORITY = 'cpan:AWNCORP';

# IMPORTS

sub import {
  my ($self, @args) = @_;

  my $target = caller;

  no strict 'refs';

  my %exports = (
    args => 1,
    assert => 1,
    box => 1,
    call => 1,
    cast => 1,
    catch => 1,
    caught => 1,
    chain => 1,
    check => 1,
    cop => 1,
    error => 1,
    false => 1,
    fault => 1,
    json => 1,
    load => 1,
    log => 1,
    make => 1,
    merge => 1,
    perl => 1,
    raise => 1,
    roll => 1,
    space => 1,
    then => 1,
    true => 1,
    unpack => 1,
    venus => 1,
    work => 1,
    wrap => 1,
    yaml => 1,
  );

  @args = grep defined && !ref && /^[A-Za-z]/ && $exports{$_}, @args;

  my %seen;
  for my $name (grep !$seen{$_}++, @args, 'true', 'false') {
    *{"${target}::${name}"} = $self->can($name) if !$target->can($name);
  }

  return $self;
}

# FUNCTIONS

sub args (@) {
  my (@args) = @_;

  return (!@args)
    ? ({})
    : ((@args == 1 && ref($args[0]) eq 'HASH')
    ? (!%{$args[0]} ? {} : {%{$args[0]}})
    : (@args % 2 ? {@args, undef} : {@args}));
}

sub assert ($$) {
  my ($data, $expr) = @_;

  require Venus::Assert;

  my $assert = Venus::Assert->new('name', 'assert(?, ?)')->expression($expr);

  return $assert->validate($data);
}

sub box ($) {
  my ($data) = @_;

  require Venus::Box;

  my $box = Venus::Box->new($data);

  return $box;
}

sub call (@) {
  my ($data, @args) = @_;
  my $next = @args;
  if ($next && UNIVERSAL::isa($data, 'CODE')) {
    return $data->(@args);
  }
  my $code = shift(@args);
  if ($next && Scalar::Util::blessed($data)) {
    return $data->$code(@args) if UNIVERSAL::can($data, $code)
      || UNIVERSAL::can($data, 'AUTOLOAD');
    $next = 0;
  }
  if ($next && ref($data) eq 'SCALAR') {
    return $$data->$code(@args) if UNIVERSAL::can(load($$data)->package, $code);
    $next = 0;
  }
  if ($next && UNIVERSAL::can(load($data)->package, $code)) {
    no strict 'refs';
    return *{"${data}::${code}"}{"CODE"} ?
      &{"${data}::${code}"}(@args) : $data->$code(@args[1..$#args]);
  }
  if ($next && UNIVERSAL::can($data, 'AUTOLOAD')) {
    no strict 'refs';
    return &{"${data}::${code}"}(@args);
  }
  fault("Exception! call(@{[join(', ', map qq('$_'), @_)]}) failed.");
}

sub cast (;$$) {
  my ($data, $into) = (@_ ? (@_) : ($_));

  require Venus::Type;

  my $type = Venus::Type->new($data);

  return $into ? $type->cast($into) : $type->deduce;
}

sub catch (&) {
  my ($data) = @_;

  my $error;

  require Venus::Try;

  my @result = Venus::Try->new($data)->error(\$error)->result;

  return wantarray ? ($error ? ($error, undef) : ($error, @result)) : $error;
}

sub caught ($$;&) {
  my ($data, $type, $code) = @_;

  ($type, my($name)) = @$type if ref $type eq 'ARRAY';

  my $is_true = $data
    && UNIVERSAL::isa($data, $type)
    && UNIVERSAL::isa($data, 'Venus::Error')
    && (
      $data->name
        ? ($name ? $data->of($name) : true())
        : (!$name ? true() : false())
    );

  if ($is_true) {
    local $_ = $data;

    return $code ? $code->($data) : $data;
  }
  else {
    return undef;
  }
}

sub chain {
  my ($data, @args) = @_;

  return if !$data;

  for my $next (map +(ref($_) eq 'ARRAY' ? $_ : [$_]), @args) {
    $data = call($data, @$next);
  }

  return $data;
}

sub check ($$) {
  my ($data, $expr) = @_;

  require Venus::Assert;

  return Venus::Assert->new->expression($expr)->check($data);
}

sub cop (@) {
  my ($data, @args) = @_;

  require Scalar::Util;

  ($data, $args[0]) = map {
    ref eq 'SCALAR' ? $$_ : Scalar::Util::blessed($_) ? ref($_) : $_
  } ($data, $args[0]);

  return space("$data")->cop(@args);
}

sub error (;$) {
  my ($data) = @_;

  $data //= {};
  $data->{context} //= (caller(1))[3];

  require Venus::Throw;

  return Venus::Throw->new->error($data);
}

sub false () {

  require Venus::False;

  return Venus::False->value;
}

sub fault (;$) {
  my ($data) = @_;

  require Venus::Fault;

  return Venus::Fault->new($data)->throw;
}

sub json ($;$) {
  my ($code, $data) = @_;

  require Venus::Json;

  if (lc($code) eq 'decode') {
    return Venus::Json->new->decode($data);
  }

  if (lc($code) eq 'encode') {
    return Venus::Json->new($data)->encode;
  }

  return undef;
}

sub load ($) {
  my ($data) = @_;

  return space($data)->do('load');
}

sub log (@) {
  my (@args) = @_;

  require Venus::Log;

  return Venus::Log->new->debug(@args);
}

sub make (@) {

  return if !@_;

  return call($_[0], 'new', @_);
}

sub merge (@) {
  my (@args) = @_;

  require Venus::Hash;

  return Venus::Hash->new({})->merge(@args);
}

sub perl ($;$) {
  my ($code, $data) = @_;

  require Venus::Dump;

  if (lc($code) eq 'decode') {
    return Venus::Dump->new->decode($data);
  }

  if (lc($code) eq 'encode') {
    return Venus::Dump->new($data)->encode;
  }

  return undef;
}

sub raise ($;$) {
  my ($self, $data) = @_;

  ($self, my $parent) = (@$self) if (ref($self) eq 'ARRAY');

  $data //= {};
  $data->{context} //= (caller(1))[3];

  $parent = 'Venus::Error' if !$parent;

  require Venus::Throw;

  return Venus::Throw->new(package => $self, parent => $parent)->error($data);
}

sub roll (@) {

  return (@_[1,0,2..$#_]);
}

sub space ($) {
  my ($data) = @_;

  require Venus::Space;

  return Venus::Space->new($data);
}

sub then (@) {

  return ($_[0], call(@_));
}

sub true () {

  require Venus::True;

  return Venus::True->value;
}

sub unpack (@) {
  my (@args) = @_;

  require Venus::Unpack;

  return Venus::Unpack->new->do('args', @args)->all;
}

sub venus ($;@) {
  my ($name, @args) = @_;

  @args = ('new') if !@args;

  return chain(\space(join('/', 'Venus', $name))->package, @args);
}

sub work ($) {
  my ($data) = @_;

  require Venus::Process;

  return Venus::Process->new->do('work', $data);
}

sub wrap ($;$) {
  my ($data, $name) = @_;

  return if !@_;

  my $moniker = $name // $data =~ s/\W//gr;
  my $caller = caller(0);

  no strict 'refs';
  no warnings 'redefine';

  return *{"${caller}::${moniker}"} = sub {@_ ? make($data, @_) : $data};
}

sub yaml ($;$) {
  my ($code, $data) = @_;

  require Venus::Yaml;

  if (lc($code) eq 'decode') {
    return Venus::Yaml->new->decode($data);
  }

  if (lc($code) eq 'encode') {
    return Venus::Yaml->new($data)->encode;
  }

  return undef;
}

1;


=head1 NAME

Venus - OO Library

=cut

=head1 ABSTRACT

OO Standard Library for Perl 5

=cut

=head1 VERSION

2.50

=cut

=head1 SYNOPSIS

  package main;

  use Venus qw(
    catch
    error
    raise
  );

  # error handling
  my ($error, $result) = catch {
    error;
  };

  # boolean keywords
  if ($result and $result eq false) {
    true;
  }

  # raise exceptions
  if (false) {
    raise 'MyApp::Error';
  }

  # and much more!
  true ne false;

=cut

=head1 DESCRIPTION

This library provides an object-orientation framework and extendible standard
library for Perl 5 with classes which wrap most native Perl data types. Venus
has a simple modular architecture, robust library of classes, methods, and
roles, supports pure-Perl autoboxing, advanced exception handling, "true" and
"false" functions, package introspection, command-line options parsing, and
more. This package will always automatically exports C<true> and C<false>
keyword functions (unless existing routines of the same name already exist in
the calling package or its parents), otherwise exports keyword functions as
requested at import. This library requires Perl C<5.18+>.

=head1 CAPABILITIES

The following is a short list of capabilities:

=over 4

=item *

Perl 5.18.0+

=item *

Zero Dependencies

=item *

Fast Object-Orientation

=item *

Robust Standard Library

=item *

Intuitive Value Classes

=item *

Pure Perl Autoboxing

=item *

Convenient Utility Classes

=item *

Simple Package Reflection

=item *

Flexible Exception Handling

=item *

Composable Standards

=item *

Pluggable (no monkeypatching)

=item *

Proxyable Methods

=item *

Type Assertions

=item *

Type Coercions

=item *

Value Casting

=item *

Boolean Values

=item *

Complete Documentation

=item *

Complete Test Coverage

=back

=cut

=head1 FUNCTIONS

This package provides the following functions:

=cut

=head2 args

  args(Any @args) (HashRef)

The args function takes a list of arguments and returns a hashref.

I<Since C<2.32>>

=over 4

=item args example 1

  package main;

  use Venus 'args';

  my $args = args(content => 'example');

  # {content => "example"}

=back

=over 4

=item args example 2

  package main;

  use Venus 'args';

  my $args = args({content => 'example'});

  # {content => "example"}

=back

=over 4

=item args example 3

  package main;

  use Venus 'args';

  my $args = args('content');

  # {content => undef}

=back

=over 4

=item args example 4

  package main;

  use Venus 'args';

  my $args = args('content', 'example', 'algorithm');

  # {content => "example", algorithm => undef}

=back

=cut

=head2 assert

  assert(Any $data, Str $expr) (Any)

The assert function builds a L<Venus::Assert> object and returns the result of
a L<Venus::Assert/validate> operation.

I<Since C<2.40>>

=over 4

=item assert example 1

  package main;

  use Venus 'assert';

  my $assert = assert(1234567890, 'number');

  # 1234567890

=back

=over 4

=item assert example 2

  package main;

  use Venus 'assert';

  my $assert = assert(1234567890, 'float');

  # Exception! (isa Venus::Assert::Error)

=back

=cut

=head2 box

  box(Any $data) (Box)

The box function returns a L<Venus::Box> object for the argument provided.

I<Since C<2.32>>

=over 4

=item box example 1

  package main;

  use Venus 'box';

  my $box = box({});

  # bless({value => bless({value => {}}, 'Venus::Hash')}, 'Venus::Box')

=back

=over 4

=item box example 2

  package main;

  use Venus 'box';

  my $box = box([]);

  # bless({value => bless({value => []}, 'Venus::Array')}, 'Venus::Box')

=back

=cut

=head2 call

  call(Str | Object | CodeRef $data, Any @args) (Any)

The call function dispatches function and method calls to a package and returns
the result.

I<Since C<2.32>>

=over 4

=item call example 1

  package main;

  use Venus 'call';

  require Digest::SHA;

  my $result = call(\'Digest::SHA', 'new');

  # bless(do{\(my $o = '...')}, 'digest::sha')

=back

=over 4

=item call example 2

  package main;

  use Venus 'call';

  require Digest::SHA;

  my $result = call('Digest::SHA', 'sha1_hex');

  # "da39a3ee5e6b4b0d3255bfef95601890afd80709"

=back

=over 4

=item call example 3

  package main;

  use Venus 'call';

  require Venus::Hash;

  my $result = call(sub{'Venus::Hash'->new(@_)}, {1..4});

  # bless({value => {1..4}}, 'Venus::Hash')

=back

=over 4

=item call example 4

  package main;

  use Venus 'call';

  require Venus::Box;

  my $result = call(Venus::Box->new(value => {}), 'merge', {1..4});

  # bless({value => bless({value => {1..4}}, 'Venus::Hash')}, 'Venus::Box')

=back

=cut

=head2 cast

  cast(Any $data, Str $type) (Object)

The cast function returns the argument provided as an object, promoting native
Perl data types to data type objects. The optional second argument can be the
name of the type for the object to cast to explicitly.

I<Since C<1.40>>

=over 4

=item cast example 1

  package main;

  use Venus 'cast';

  my $undef = cast;

  # bless({value => undef}, "Venus::Undef")

=back

=over 4

=item cast example 2

  package main;

  use Venus 'cast';

  my @booleans = map cast, true, false;

  # (bless({value => 1}, "Venus::Boolean"), bless({value => 0}, "Venus::Boolean"))

=back

=over 4

=item cast example 3

  package main;

  use Venus 'cast';

  my $example = cast bless({}, "Example");

  # bless({value => 1}, "Example")

=back

=over 4

=item cast example 4

  package main;

  use Venus 'cast';

  my $float = cast 1.23;

  # bless({value => "1.23"}, "Venus::Float")

=back

=cut

=head2 catch

  catch(CodeRef $block) (Error, Any)

The catch function executes the code block trapping errors and returning the
caught exception in scalar context, and also returning the result as a second
argument in list context.

I<Since C<0.01>>

=over 4

=item catch example 1

  package main;

  use Venus 'catch';

  my $error = catch {die};

  $error;

  # "Died at ..."

=back

=over 4

=item catch example 2

  package main;

  use Venus 'catch';

  my ($error, $result) = catch {error};

  $error;

  # bless({...}, 'Venus::Error')

=back

=over 4

=item catch example 3

  package main;

  use Venus 'catch';

  my ($error, $result) = catch {true};

  $result;

  # 1

=back

=cut

=head2 caught

  caught(Object $error, Str | Tuple[Str, Str] $identity, CodeRef $block) (Any)

The caught function evaluates the exception object provided and validates its
identity and name (if provided) then executes the code block provided returning
the result of the callback. If no callback is provided this function returns
the exception object on success and C<undef> on failure.

I<Since C<1.95>>

=over 4

=item caught example 1

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error };

  my $result = caught $error, 'Venus::Error';

  # bless(..., 'Venus::Error')

=back

=over 4

=item caught example 2

  package main;

  use Venus 'catch', 'caught', 'raise';

  my $error = catch { raise 'Example::Error' };

  my $result = caught $error, 'Venus::Error';

  # bless(..., 'Venus::Error')

=back

=over 4

=item caught example 3

  package main;

  use Venus 'catch', 'caught', 'raise';

  my $error = catch { raise 'Example::Error' };

  my $result = caught $error, 'Example::Error';

  # bless(..., 'Venus::Error')

=back

=over 4

=item caught example 4

  package main;

  use Venus 'catch', 'caught', 'raise';

  my $error = catch { raise 'Example::Error', { name => 'on.test' } };

  my $result = caught $error, ['Example::Error', 'on.test'];

  # bless(..., 'Venus::Error')

=back

=over 4

=item caught example 5

  package main;

  use Venus 'catch', 'caught', 'raise';

  my $error = catch { raise 'Example::Error', { name => 'on.recv' } };

  my $result = caught $error, ['Example::Error', 'on.send'];

  # undef

=back

=over 4

=item caught example 6

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error };

  my $result = caught $error, ['Example::Error', 'on.send'];

  # undef

=back

=over 4

=item caught example 7

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error };

  my $result = caught $error, ['Example::Error'];

  # undef

=back

=over 4

=item caught example 8

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error };

  my $result = caught $error, 'Example::Error';

  # undef

=back

=over 4

=item caught example 9

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error { name => 'on.send' } };

  my $result = caught $error, ['Venus::Error', 'on.send'];

  # bless(..., 'Venus::Error')

=back

=over 4

=item caught example 10

  package main;

  use Venus 'catch', 'caught', 'error';

  my $error = catch { error { name => 'on.send.open' } };

  my $result = caught $error, ['Venus::Error', 'on.send'], sub {
    $error->stash('caught', true) if $error->is('on.send.open');
    return $error;
  };

  # bless(..., 'Venus::Error')

=back

=cut

=head2 chain

  chain(Str | Object | CodeRef $self, Str | ArrayRef[Str] @args) (Any)

The chain function chains function and method calls to a package (and return
values) and returns the result.

I<Since C<2.32>>

=over 4

=item chain example 1

  package main;

  use Venus 'chain';

  my $result = chain('Venus::Path', ['new', 't'], 'exists');

  # 1

=back

=over 4

=item chain example 2

  package main;

  use Venus 'chain';

  my $result = chain('Venus::Path', ['new', 't'], ['test', 'd']);

  # 1

=back

=cut

=head2 check

  check(Any $data, Str $expr) (Bool)

The check function builds a L<Venus::Assert> object and returns the result of
a L<Venus::Assert/check> operation.

I<Since C<2.40>>

=over 4

=item check example 1

  package main;

  use Venus 'check';

  my $check = check(rand, 'float');

  # true

=back

=over 4

=item check example 2

  package main;

  use Venus 'check';

  my $check = check(rand, 'string');

  # false

=back

=cut

=head2 cop

  cop(Str | Object | CodeRef $self, Str $name) (CodeRef)

The cop function attempts to curry the given subroutine on the object or class
and if successful returns a closure.

I<Since C<2.32>>

=over 4

=item cop example 1

  package main;

  use Venus 'cop';

  my $coderef = cop('Digest::SHA', 'sha1_hex');

  # sub { ... }

=back

=over 4

=item cop example 2

  package main;

  use Venus 'cop';

  require Digest::SHA;

  my $coderef = cop(Digest::SHA->new, 'digest');

  # sub { ... }

=back

=cut

=head2 error

  error(Maybe[HashRef] $args) (Error)

The error function throws a L<Venus::Error> exception object using the
exception object arguments provided.

I<Since C<0.01>>

=over 4

=item error example 1

  package main;

  use Venus 'error';

  my $error = error;

  # bless({...}, 'Venus::Error')

=back

=over 4

=item error example 2

  package main;

  use Venus 'error';

  my $error = error {
    message => 'Something failed!',
  };

  # bless({message => 'Something failed!', ...}, 'Venus::Error')

=back

=cut

=head2 false

  false() (Bool)

The false function returns a falsy boolean value which is designed to be
practically indistinguishable from the conventional numerical C<0> value.

I<Since C<0.01>>

=over 4

=item false example 1

  package main;

  use Venus;

  my $false = false;

  # 0

=back

=over 4

=item false example 2

  package main;

  use Venus;

  my $true = !false;

  # 1

=back

=cut

=head2 fault

  fault(Str $args) (Fault)

The fault function throws a L<Venus::Fault> exception object and represents a
system failure, and isn't meant to be caught.

I<Since C<1.80>>

=over 4

=item fault example 1

  package main;

  use Venus 'fault';

  my $fault = fault;

  # bless({message => 'Exception!'}, 'Venus::Fault')

=back

=over 4

=item fault example 2

  package main;

  use Venus 'fault';

  my $fault = fault 'Something failed!';

  # bless({message => 'Something failed!'}, 'Venus::Fault')

=back

=cut

=head2 json

  json(Str $call, Any $data) (Any)

The json function builds a L<Venus::Json> object and will either
L<Venus::Json/decode> or L<Venus::Json/encode> based on the argument provided
and returns the result.

I<Since C<2.40>>

=over 4

=item json example 1

  package main;

  use Venus 'json';

  my $decode = json 'decode', '{"codename":["Ready","Robot"],"stable":true}';

  # { codename => ["Ready", "Robot"], stable => 1 }

=back

=over 4

=item json example 2

  package main;

  use Venus 'json';

  my $encode = json 'encode', { codename => ["Ready", "Robot"], stable => true };

  # '{"codename":["Ready","Robot"],"stable":true}'

=back

=cut

=head2 load

  load(Any $name) (Space)

The load function loads the package provided and returns a L<Venus::Space> object.

I<Since C<2.32>>

=over 4

=item load example 1

  package main;

  use Venus 'load';

  my $space = load 'Venus::Scalar';

  # bless({value => 'Venus::Scalar'}, 'Venus::Space')

=back

=cut

=head2 log

  log(Any @args) (Log)

The log function prints the arguments provided to STDOUT, stringifying complex
values, and returns a L<Venus::Log> object.

I<Since C<2.40>>

=over 4

=item log example 1

  package main;

  use Venus 'log';

  my $log = log;

  # bless({...}, 'Venus::Log')

  # log time, rand, 1..9;

  # 00000000 0.000000, 1..9

=back

=cut

=head2 make

  make(Str $package, Any @args) (Any)

The make function L<"calls"|Venus/call> the C<new> routine on the invocant and
returns the result which should be a package string or an object.

I<Since C<2.32>>

=over 4

=item make example 1

  package main;

  use Venus 'make';

  my $made = make('Digest::SHA');

  # bless(do{\(my $o = '...')}, 'Digest::SHA')

=back

=over 4

=item make example 2

  package main;

  use Venus 'make';

  my $made = make('Digest', 'SHA');

  # bless(do{\(my $o = '...')}, 'Digest::SHA')

=back

=cut

=head2 merge

  merge(HashRef @args) (HashRef)

The merge function returns a hash reference which is a merger of all of the
hashref arguments provided.

I<Since C<2.32>>

=over 4

=item merge example 1

  package main;

  use Venus 'merge';

  my $merged = merge({1..4}, {5, 6});

  # {1..6}

=back

=over 4

=item merge example 2

  package main;

  use Venus 'merge';

  my $merged = merge({1..4}, {5, 6}, {7, 8, 9, 0});

  # {1..9, 0}

=back

=cut

=head2 perl

  perl(Str $call, Any $data) (Any)

The perl function builds a L<Venus::Dump> object and will either
L<Venus::Dump/decode> or L<Venus::Dump/encode> based on the argument provided
and returns the result.

I<Since C<2.40>>

=over 4

=item perl example 1

  package main;

  use Venus 'perl';

  my $decode = perl 'decode', '{stable=>bless({},\'Venus::True\')}';

  # { stable => 1 }

=back

=over 4

=item perl example 2

  package main;

  use Venus 'perl';

  my $encode = perl 'encode', { stable => true };

  # '{stable=>bless({},\'Venus::True\')}'

=back

=cut

=head2 raise

  raise(Str $class | Tuple[Str, Str] $class, Maybe[HashRef] $args) (Error)

The raise function generates and throws a named exception object derived from
L<Venus::Error>, or provided base class, using the exception object arguments
provided.

I<Since C<0.01>>

=over 4

=item raise example 1

  package main;

  use Venus 'raise';

  my $error = raise 'MyApp::Error';

  # bless({...}, 'MyApp::Error')

=back

=over 4

=item raise example 2

  package main;

  use Venus 'raise';

  my $error = raise ['MyApp::Error', 'Venus::Error'];

  # bless({...}, 'MyApp::Error')

=back

=over 4

=item raise example 3

  package main;

  use Venus 'raise';

  my $error = raise ['MyApp::Error', 'Venus::Error'], {
    message => 'Something failed!',
  };

  # bless({message => 'Something failed!', ...}, 'MyApp::Error')

=back

=cut

=head2 roll

  roll(Str $name, Any @args) (Any)

The roll function takes a list of arguments, assuming the first argument is
invokable, and reorders the list such that the routine name provided comes
after the invocant (i.e. the 1st argument), creating a list acceptable to the
L</call> function.

I<Since C<2.32>>

=over 4

=item roll example 1

  package main;

  use Venus 'roll';

  my @list = roll('sha1_hex', 'Digest::SHA');

  # ('Digest::SHA', 'sha1_hex');

=back

=over 4

=item roll example 2

  package main;

  use Venus 'roll';

  my @list = roll('sha1_hex', call(\'Digest::SHA', 'new'));

  # (bless(do{\(my $o = '...')}, 'Digest::SHA'), 'sha1_hex');

=back

=cut

=head2 space

  space(Any $name) (Space)

The space function returns a L<Venus::Space> object for the package provided.

I<Since C<2.32>>

=over 4

=item space example 1

  package main;

  use Venus 'space';

  my $space = space 'Venus::Scalar';

  # bless({value => 'Venus::Scalar'}, 'Venus::Space')

=back

=cut

=head2 then

  then(Str | Object | CodeRef $self, Any @args) (Any)

The then function proxies the call request to the L</call> function and returns
the result as a list, prepended with the invocant.

I<Since C<2.32>>

=over 4

=item then example 1

  package main;

  use Venus 'then';

  my @list = then('Digest::SHA', 'sha1_hex');

  # ("Digest::SHA", "da39a3ee5e6b4b0d3255bfef95601890afd80709")

=back

=cut

=head2 true

  true() (Bool)

The true function returns a truthy boolean value which is designed to be
practically indistinguishable from the conventional numerical C<1> value.

I<Since C<0.01>>

=over 4

=item true example 1

  package main;

  use Venus;

  my $true = true;

  # 1

=back

=over 4

=item true example 2

  package main;

  use Venus;

  my $false = !true;

  # 0

=back

=cut

=head2 unpack

  unpack(Any @args) (Unpack)

The unpack function builds and returns a L<Venus::Unpack> object.

I<Since C<2.40>>

=over 4

=item unpack example 1

  package main;

  use Venus 'unpack';

  my $unpack = unpack;

  # bless({...}, 'Venus::Unpack')

  # $unpack->checks('string');

  # false

  # $unpack->checks('undef');

  # false

=back

=over 4

=item unpack example 2

  package main;

  use Venus 'unpack';

  my $unpack = unpack rand;

  # bless({...}, 'Venus::Unpack')

  # $unpack->check('number');

  # false

  # $unpack->check('float');

  # true

=back

=cut

=head2 venus

  venus(Str $name, Any @args) (Any)

The venus function build a L<Venus> package via the L</chain> function based on
the name provided and returns an instance of that package.

I<Since C<2.40>>

=over 4

=item venus example 1

  package main;

  use Venus 'venus';

  my $space = venus 'space';

  # bless({value => 'Venus'}, 'Venus::Space')

=back

=over 4

=item venus example 2

  package main;

  use Venus 'venus';

  my $space = venus 'space', ['new', 'venus/string'];

  # bless({value => 'Venus::String'}, 'Venus::Space')

=back

=over 4

=item venus example 3

  package main;

  use Venus 'venus';

  my $space = venus 'code';

  # bless({value => sub{...}}, 'Venus::Code')

=back

=cut

=head2 work

  work(CodeRef $callback) (Process)

The work function builds a L<Venus::Process> object, forks the current process
using the callback provided via the L<Venus::Process/work> operation, and
returns an instance of L<Venus::Process> representing the current process.

I<Since C<2.40>>

=over 4

=item work example 1

  package main;

  use Venus 'work';

  my $parent = work sub {
    my ($process) = @_;
    # in forked process ...
    $process->exit;
  };

  # bless({...}, 'Venus::Process')

=back

=cut

=head2 wrap

  wrap(Str $data, Str $name) (CodeRef)

The wrap function installs a wrapper function in the calling package which when
called either returns the package string if no arguments are provided, or calls
L</make> on the package with whatever arguments are provided and returns the
result. Unless an alias is provided as a second argument, special characters
are stripped from the package to create the function name.

I<Since C<2.32>>

=over 4

=item wrap example 1

  package main;

  use Venus 'wrap';

  my $coderef = wrap('Digest::SHA');

  # sub { ... }

  # my $digest = DigestSHA();

  # "Digest::SHA"

  # my $digest = DigestSHA(1);

  # bless(do{\(my $o = '...')}, 'Digest::SHA')

=back

=over 4

=item wrap example 2

  package main;

  use Venus 'wrap';

  my $coderef = wrap('Digest::SHA', 'SHA');

  # sub { ... }

  # my $digest = SHA();

  # "Digest::SHA"

  # my $digest = SHA(1);

  # bless(do{\(my $o = '...')}, 'Digest::SHA')

=back

=cut

=head2 yaml

  yaml(Str $call, Any $data) (Any)

The yaml function builds a L<Venus::Yaml> object and will either
L<Venus::Yaml/decode> or L<Venus::Yaml/encode> based on the argument provided
and returns the result.

I<Since C<2.40>>

=over 4

=item yaml example 1

  package main;

  use Venus 'yaml';

  my $decode = yaml 'decode', "---\nname:\n- Ready\n- Robot\nstable: true\n";

  # { name => ["Ready", "Robot"], stable => 1 }

=back

=over 4

=item yaml example 2

  package main;

  use Venus 'yaml';

  my $encode = yaml 'encode', { name => ["Ready", "Robot"], stable => true };

  # '---\nname:\n- Ready\n- Robot\nstable: true\n'

=back

=cut

=head1 FEATURES

This package provides the following features:

=cut

=over 4

=item venus-args

This library contains a L<Venus::Args> class which provides methods for
accessing C<@ARGS> items.

=back

=over 4

=item venus-array

This library contains a L<Venus::Array> class which provides methods for
manipulating array data.

=back

=over 4

=item venus-assert

This library contains a L<Venus::Assert> class which provides a mechanism for
asserting type constraints and coercion.

=back

=over 4

=item venus-boolean

This library contains a L<Venus::Boolean> class which provides a representation
for boolean values.

=back

=over 4

=item venus-box

This library contains a L<Venus::Box> class which provides a pure Perl boxing
mechanism.

=back

=over 4

=item venus-class

This library contains a L<Venus::Class> class which provides a class builder.

=back

=over 4

=item venus-cli

This library contains a L<Venus::Cli> class which provides a superclass for
creating CLIs.

=back

=over 4

=item venus-code

This library contains a L<Venus::Code> class which provides methods for
manipulating subroutines.

=back

=over 4

=item venus-config

This library contains a L<Venus::Config> class which provides methods for
loading Perl, YAML, and JSON configuration data.

=back

=over 4

=item venus-data

This library contains a L<Venus::Data> class which provides methods for
extracting C<DATA> sections and POD block.

=back

=over 4

=item venus-date

This library contains a L<Venus::Date> class which provides methods for
formatting, parsing, and manipulating dates.

=back

=over 4

=item venus-dump

This library contains a L<Venus::Dump> class which provides methods for reading
and writing dumped Perl data.

=back

=over 4

=item venus-error

This library contains a L<Venus::Error> class which represents a context-aware
error (exception object).

=back

=over 4

=item venus-false

This library contains a L<Venus::False> class which provides the global
C<false> value.

=back

=over 4

=item venus-fault

This library contains a L<Venus::Fault> class which represents a generic system
error (exception object).

=back

=over 4

=item venus-float

This library contains a L<Venus::Float> class which provides methods for
manipulating float data.

=back

=over 4

=item venus-gather

This library contains a L<Venus::Gather> class which provides an
object-oriented interface for complex pattern matching operations on
collections of data, e.g. array references.

=back

=over 4

=item venus-hash

This library contains a L<Venus::Hash> class which provides methods for
manipulating hash data.

=back

=over 4

=item venus-json

This library contains a L<Venus::Json> class which provides methods for reading
and writing JSON data.

=back

=over 4

=item venus-log

This library contains a L<Venus::Log> class which provides methods for logging
information using various log levels.

=back

=over 4

=item venus-match

This library contains a L<Venus::Match> class which provides an object-oriented
interface for complex pattern matching operations on scalar values.

=back

=over 4

=item venus-meta

This library contains a L<Venus::Meta> class which provides configuration
information for L<Venus> derived classes.

=back

=over 4

=item venus-mixin

This library contains a L<Venus::Mixin> class which provides a mixin builder.

=back

=over 4

=item venus-name

This library contains a L<Venus::Name> class which provides methods for parsing
and formatting package namespaces.

=back

=over 4

=item venus-number

This library contains a L<Venus::Number> class which provides methods for
manipulating number data.

=back

=over 4

=item venus-opts

This library contains a L<Venus::Opts> class which provides methods for
handling command-line arguments.

=back

=over 4

=item venus-path

This library contains a L<Venus::Path> class which provides methods for working
with file system paths.

=back

=over 4

=item venus-process

This library contains a L<Venus::Process> class which provides methods for
handling and forking processes.

=back

=over 4

=item venus-prototype

This library contains a L<Venus::Prototype> class which provides a simple
construct for enabling prototype-base programming.

=back

=over 4

=item venus-random

This library contains a L<Venus::Random> class which provides an
object-oriented interface for Perl's pseudo-random number generator.

=back

=over 4

=item venus-regexp

This library contains a L<Venus::Regexp> class which provides methods for
manipulating regexp data.

=back

=over 4

=item venus-replace

This library contains a L<Venus::Replace> class which provides methods for
manipulating regexp replacement data.

=back

=over 4

=item venus-scalar

This library contains a L<Venus::Scalar> class which provides methods for
manipulating scalar data.

=back

=over 4

=item venus-search

This library contains a L<Venus::Search> class which provides methods for
manipulating regexp search data.

=back

=over 4

=item venus-space

This library contains a L<Venus::Space> class which provides methods for
parsing and manipulating package namespaces.

=back

=over 4

=item venus-string

This library contains a L<Venus::String> class which provides methods for
manipulating string data.

=back

=over 4

=item venus-template

This library contains a L<Venus::Template> class which provides a templating
system, and methods for rendering template.

=back

=over 4

=item venus-test

This library contains a L<Venus::Test> class which aims to provide a standard
for documenting L<Venus> derived software projects.

=back

=over 4

=item venus-throw

This library contains a L<Venus::Throw> class which provides a mechanism for
generating and raising error objects.

=back

=over 4

=item venus-true

This library contains a L<Venus::True> class which provides the global C<true>
value.

=back

=over 4

=item venus-try

This library contains a L<Venus::Try> class which provides an object-oriented
interface for performing complex try/catch operations.

=back

=over 4

=item venus-type

This library contains a L<Venus::Type> class which provides methods for casting
native data types to objects.

=back

=over 4

=item venus-undef

This library contains a L<Venus::Undef> class which provides methods for
manipulating undef data.

=back

=over 4

=item venus-unpack

This library contains a L<Venus::Unpack> class which provides methods for
validating, coercing, and otherwise operating on lists of arguments.

=back

=over 4

=item venus-vars

This library contains a L<Venus::Vars> class which provides methods for
accessing C<%ENV> items.

=back

=over 4

=item venus-yaml

This library contains a L<Venus::Yaml> class which provides methods for reading
and writing YAML data.

=back

=head1 AUTHORS

Awncorp, C<awncorp@cpan.org>

=cut

=head1 LICENSE

Copyright (C) 2000, Al Newkirk.

This program is free software, you can redistribute it and/or modify it under
the terms of the Apache license version 2.0.

=cut