use strict;
use warnings;

use Test::Lib;
use Test::Net::SAML2;

use Net::SAML2::Protocol::Assertion;
use XML::Sig;
use MIME::Base64 qw/decode_base64/;

my $response = <<'RESPONSE';
PHNhbWxwOlJlc3BvbnNlIHhtbG5zOnNhbWxwPSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6cHJvdG9jb2wiIHhtbG5zOnNhbWw9InVybjpvYXNpczpuYW1lczp0YzpTQU1MOjIuMDphc3NlcnRpb24iIERlc3RpbmF0aW9uPSJodHRwczovL25ldHNhbWwyLXRlc3RhcHAubG9jYWwvY29uc3VtZXItcG9zdCIgSUQ9IklEXzZhNjhiZDJmLTU2NTgtNDgzZC04NjZiLWFkZTI1NmRlOGMwNiIgSW5SZXNwb25zZVRvPSJORVRTQU1MMl85MzVjNzgyZDVmNWU0OTk2MzhhNzQ3MWIyNTcwNjJiNSIgSXNzdWVJbnN0YW50PSIyMDIyLTAzLTI3VDEyOjA2OjU2Ljc1M1oiIFZlcnNpb249IjIuMCI+PHNhbWw6SXNzdWVyPmh0dHBzOi8va2V5Y2xvYWsubG9jYWw6ODQ0My9hdXRoL3JlYWxtcy9Gb3N3aWtpPC9zYW1sOklzc3Vlcj48ZHNpZzpTaWduYXR1cmUgeG1sbnM6ZHNpZz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC8wOS94bWxkc2lnIyI+PGRzaWc6U2lnbmVkSW5mbz48ZHNpZzpDYW5vbmljYWxpemF0aW9uTWV0aG9kIEFsZ29yaXRobT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS8xMC94bWwtZXhjLWMxNG4jIi8+PGRzaWc6U2lnbmF0dXJlTWV0aG9kIEFsZ29yaXRobT0iaHR0cDovL3d3dy53My5vcmcvMjAwMC8wOS94bWxkc2lnI3JzYS1zaGExIi8+PGRzaWc6UmVmZXJlbmNlIFVSST0iI0lEXzZhNjhiZDJmLTU2NTgtNDgzZC04NjZiLWFkZTI1NmRlOGMwNiI+PGRzaWc6VHJhbnNmb3Jtcz48ZHNpZzpUcmFuc2Zvcm0gQWxnb3JpdGhtPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwLzA5L3htbGRzaWcjZW52ZWxvcGVkLXNpZ25hdHVyZSIvPjxkc2lnOlRyYW5zZm9ybSBBbGdvcml0aG09Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvMTAveG1sLWV4Yy1jMTRuIyIvPjwvZHNpZzpUcmFuc2Zvcm1zPjxkc2lnOkRpZ2VzdE1ldGhvZCBBbGdvcml0aG09Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvMDkveG1sZHNpZyNzaGExIi8+PGRzaWc6RGlnZXN0VmFsdWU+ZkZzdExEQklvd1hsNDNUME9VczZGK0hJbEY4PTwvZHNpZzpEaWdlc3RWYWx1ZT48L2RzaWc6UmVmZXJlbmNlPjwvZHNpZzpTaWduZWRJbmZvPjxkc2lnOlNpZ25hdHVyZVZhbHVlPkI2anROUkxhMFhyVDU0aFVGMzhuTzVrUXdnNGIrekREeDFHbFlpVktIc1JVUnp6MFZ5Y2FVRHY2ai84SllQRFRIZUh1Q3VNc2Rjbi9wcFB3R0J4SzdLVVdCaUtwOUNjR2IyT1ljeUxCTmZkWmNPL2dsUVgva2ZPWkN5Zlc1b2xtb2FwQS80QWYwc2E0Ym5CbEZrbmZPcEhmRCtpME0yYk5lblM3OUF0bHZJR1k5bHRkSkFUamV1VG5leXdXQVMrTjNxaCtDTHVLZGU0R245VUw5VnhDWlZrNVhLZ2E0VHFhZ3YwZmVRckIxTHd2UkNxY1lmSUt6RGdzSHQyTlNZcE15SHpQeE5nTjU3VDB3R255R011NSt1ZXgwQU1wNG9jN280OHdzUG9teHkwT1pVSFg3bWRzazliVU9rZitmQlVxbGNXNFVTb3ZhY01LR1pZRG96VE5aUT09PC9kc2lnOlNpZ25hdHVyZVZhbHVlPjxkc2lnOktleUluZm8+PGRzaWc6WDUwOURhdGE+PGRzaWc6WDUwOUNlcnRpZmljYXRlPk1JSUNuVENDQVlVQ0JnRjVZcXRRQlRBTkJna3Foa2lHOXcwQkFRc0ZBREFTTVJBd0RnWURWUVFEREFkR2IzTjNhV3RwTUI0WERUSXhNRFV4TWpJeU1Ua3lORm9YRFRNeE1EVXhNakl5TWpFd05Gb3dFakVRTUE0R0ExVUVBd3dIUm05emQybHJhVENDQVNJd0RRWUpLb1pJaHZjTkFRRUJCUUFEZ2dFUEFEQ0NBUW9DZ2dFQkFKTUdHNmpyZGFkdy82cm5PQUdtTnRtZElaeTExNkp5b2NLbHNveGcraVFUbFJJMmUzZ2Vsc2lPVzdyWE5JWUhIL2Y0b3pROEY0YmE3R3hKTU5XbHJESkZOMjNEaWo1MjFQVnFKSHN1M1pBOEpPUCt0eE1DTjIyemhDTzZPWWlXeDVQOXdtN3pXVmNmZzNzUzk1NjRMUTRNN0pCUTh0RFl4WTlSTENEUitzTk5kMGhXbTZTcmtFeWdocWJjeE5ZK3JnWGZ4TEJLNWVHWHlYMVprME5MQTVYcVJnNWE4QkR6MW9VWjZPNGMyMXRWT3ZWOHZxQ1V0Y254M2hXeGNCZ1hpelc4cGtTUXBRaVE5NnpYcXVBdkR3a0x0WW5RTFY1R1FsdDZjNDE0QTdVNGRzQVpaQ2M0OTBycW5jZnNqRGZiRk16ajg5cy9XQ3RGRE96U2ExNjNwcUVDQXdFQUFUQU5CZ2txaGtpRzl3MEJBUXNGQUFPQ0FRRUFQcGVHc0JPSk4zeEdVdnR4SnFQTTJqYTNnN0c3TGlPSkd2elpTSU9GcjUwYmFlYnNvSk5Sd0wyR0RmWVVUTTFTV0R6NFVIbkdlYnNtZTVUVG16alZPM1lFdm5PTVR0VkM2L2ZZWWRvdUFxSUorY1RtbUYzQ3hkL3RPcjVma2FQc2NCMHgwK3pxV3FnQlpMbzBGVkVDRE10K0RZazFIYVFKUHhzQVhHYWhVbUlJcGZJS083QVV4NXRENzRQUjhYZUhXeUwwdzhqZzFoOG5WdGM0OVA3aDA4U3ptU0ZZMHBoSjlwbExwU3ViQ3NkLzFLTVBPSjBEaDdrWUVhT0pPT1d3akxnZ2lobzVONEtCeXRwdHM2SElqbVBsS3ZWN1VKRUFtUXlrdWhPNlB5RmZHandYeHBZUlR0R2EzZlpRcXU2Qnp0UkhEU1pRZmMrSzA4VlRtQWpyaXc9PTwvZHNpZzpYNTA5Q2VydGlmaWNhdGU+PC9kc2lnOlg1MDlEYXRhPjxkc2lnOktleVZhbHVlPjxkc2lnOlJTQUtleVZhbHVlPjxkc2lnOk1vZHVsdXM+a3dZYnFPdDFwM0QvcXVjNEFhWTIyWjBobkxYWG9uS2h3cVd5akdENkpCT1ZFalo3ZUI2V3lJNWJ1dGMwaGdjZjkvaWpORHdYaHRyc2JFa3cxYVdzTWtVM2JjT0tQbmJVOVdva2V5N2RrRHdrNC82M0V3STNiYk9FSTdvNWlKYkhrLzNDYnZOWlZ4K0RleEwzbnJndERnenNrRkR5ME5qRmoxRXNJTkg2dzAxM1NGYWJwS3VRVEtDR3B0ekUxajZ1QmQvRXNFcmw0WmZKZlZtVFEwc0RsZXBHRGxyd0VQUFdoUm5vN2h6YlcxVTY5WHkrb0pTMXlmSGVGYkZ3R0JlTE5ieW1SSkNsQ0pEM3JOZXE0QzhQQ1F1MWlkQXRYa1pDVzNwempYZ0R0VGgyd0Jsa0p6ajNTdXFkeCt5TU45c1V6T1B6Mno5WUswVU03TkpyWHJlbW9RPT08L2RzaWc6TW9kdWx1cz48ZHNpZzpFeHBvbmVudD5BUUFCPC9kc2lnOkV4cG9uZW50PjwvZHNpZzpSU0FLZXlWYWx1ZT48L2RzaWc6S2V5VmFsdWU+PC9kc2lnOktleUluZm8+PC9kc2lnOlNpZ25hdHVyZT48c2FtbHA6U3RhdHVzPjxzYW1scDpTdGF0dXNDb2RlIFZhbHVlPSJ1cm46b2FzaXM6bmFtZXM6dGM6U0FNTDoyLjA6c3RhdHVzOlN1Y2Nlc3MiLz48L3NhbWxwOlN0YXR1cz48c2FtbDpFbmNyeXB0ZWRBc3NlcnRpb24+PHhlbmM6RW5jcnlwdGVkRGF0YSB4bWxuczp4ZW5jPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxLzA0L3htbGVuYyMiIFR5cGU9Imh0dHA6Ly93d3cudzMub3JnLzIwMDEvMDQveG1sZW5jI0VsZW1lbnQiPjx4ZW5jOkVuY3J5cHRpb25NZXRob2QgQWxnb3JpdGhtPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxLzA0L3htbGVuYyNhZXMxMjgtY2JjIi8+PGRzOktleUluZm8geG1sbnM6ZHM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvMDkveG1sZHNpZyMiPjx4ZW5jOkVuY3J5cHRlZEtleT48eGVuYzpFbmNyeXB0aW9uTWV0aG9kIEFsZ29yaXRobT0iaHR0cDovL3d3dy53My5vcmcvMjAwMS8wNC94bWxlbmMjcnNhLW9hZXAtbWdmMXAiLz48eGVuYzpDaXBoZXJEYXRhPjx4ZW5jOkNpcGhlclZhbHVlPk1oZGhFdzl4dUlWRVhxa2dvR1RHNzRWUlhGaHZZenFqY2ovcE5XQ3B0K3BYbnh0MzBIRmdSZmxmY0ZRdlc1WU9XYStic2gwR2tXbi9PQ3FDeStGM254NEdDaW1iVlZ3bTY3VDllQjJXaEV5MGZ3NVpJbUtuYkcxOWJvZ0pCL1M3T3l5bktoY05GT2w3ZnNja1U5ZUh2UkdTZVpKUHZ3MmxwU0VBSFMxbWRPVmFBMFdHVkhIR3F0T2RWNk8xTjArRnVOSWpjUEorVTJjTjNwYU9uZGpTOHlkbjgvcVFEWGVFWXRzSThGNUI1b0JmUUVLbE53OUNodGpBeTU3YlFsVmJHTkNzMndUWk0vaG9mQVgwaXQwMzVwSzdpd0NPMFBLYWozUTY1TS96azU4dGxvbWpyOVNJOGNBdXdRdFJVZHVhdWhnd2x0WTVCMG5XaVFTNlVjRUhLTWNORk9sUUhnRGRta0FXVTMxS0o2R3psSkdFckFqK2x0MXpSVWNLOTJXeUU5dWNCSndwTkVWYWdsc1U4amxvekdjU0xSR2pONjQ4SWpUN0lQK2tVVkZ3UFg3N1hDbXdVQUVOczZBSDJmVGhHV0M1T0g0Z1VpNnV2RGFoWTRSLzJFeXl5SFhwY3RBSmpnSFQ1ckltNXZIeDA1WkdiR2hBSmxuNytLcFFwWXlBU0ZYSUpwTkVSdDk2U2I4ZWtRd3lvUFd5OG1wdzZWSytUd0VQbWJBSytFZVJpRnZOREhBU2xIZXA3R0E3Q1ZWYkVaTnZzSUJwMHErYVY1M3hyb0hhb09VNk4zaG5SQkdrM2xVU25OMkM5VmowK3lqNHJZTGJOZ0ppelI0bWxQVGE3RDNKb01QN2tTaWdaWVExczc1eHJvRGtaOE5GV05xSFhVaE80WHpWVG0wPTwveGVuYzpDaXBoZXJWYWx1ZT48L3hlbmM6Q2lwaGVyRGF0YT48L3hlbmM6RW5jcnlwdGVkS2V5PjwvZHM6S2V5SW5mbz48eGVuYzpDaXBoZXJEYXRhPjx4ZW5jOkNpcGhlclZhbHVlPlgvMkM4MWxtVkUzcGZSTVlZbWxjaW9tSHlxSkhwcEdpdkdsUTYxVmVIWGdYY3F3bnpUS0xDZHlMNCs2MDFUenVxZlJWWm5oblg0SUtHcUJhTkhsNGlpNG5QczF0STIyZnkvKzRCKzZYSm5jd2NrcXV0NXRESlpKN2F4S1ZCZUtYam5EN01ueHJsV0M0WmdNME8zWnJCTHdDRDRoVjFQTUlmMTE4YzlVOFhqVTF1ZndlRU9zQklheWNvOGlxYmtyWlFQQXlXL2FOQTVwZFFlUGpjRVd3cmhqdTZUTnVnS21pWHVlTXNrMGNLM3FGTTlpSHA0Y2FLSDhYR25kVFdvOEs5dldEem1tQURndUNoeEVncXhCWDBJTzU2UGVLdmNMdHY2SVg0eWdFNHMxcmUvOGNpVDlFL21OSmpmbFVRT0dPRDRHajcxTHlST2s1NERGRFRMR3laY1JUZE1NQ3NQOXcrOERXYnI2T2JoNDlrN1VaVUlOU21ldGJEWUI5MUtWT0ZKNUtOZjdhOWNsV2g5dmcrb3BGYW8yd1hYZmJicTQ3L1hhWjZUZHdkWmM5VzhYb05DcE1RWnQyYU1JbHdEdWxhYnQzZzUzZkJWQXYxcjhXNWxmUGgyVW9XY21USzlzRzdyd3dSd0F1dzZXYlVEMC9KVUVUWlp5VGRxMDhJNGcxYm53L2I1Q2ZKeit5OFNtQnhUU2YybkYwS2FMd0J4MkhwYUdSeUxoU09FNHBRcDFyV1A1NVc2YWMwNXRWRm9QamxDZjlyTTc4a3pIdzhITGF5Z3VyY0JPRHRjUGhlU1k5UUxSWWF0ejNnK0dZN0FrT09mMTFabmxUWnhNNUpsS3JLWXppSHVCeEJ2N3BYaVI4SFdLUUtJTzkvZjIrbTdlWVJOeHp0ZFk4TTZWQ1BoT2NJa0p3MExUZlRLNDY0SnJURXpYbGRrdmtzUjNIVnJPbGxXalY4dGNINTc3YXV3NnhTREV4QXptTnE0emJEQnZacUlWZCtPdjAwWXRiejgrRVJvVmMrekIzZW8yTElMbDh0UnZXVis3RGMvUWRpZWw5OUxIZlc4ZE9UYitoSFdwaXIxQUE5SFRIb3QzLzhtcENPdU8zR0g0R2RHLzdxOUVldmhEcjViYzlVRmpRUit4NUdNQllrVlZlakhEZmJ1N1FtWGFyVmRjWVBUZ2dxQWlaZ2poUXF4ZVRUb1RZV0xaVXFmSXdVSk9yUUlUVlB0Q01lVWJDNnBRY1IwVmZldmN3aE82ZEtzRWlNZzFXTGIvc01iUWZ0bVhtVy9jTGY1amhPdjUrWUlSWTRnWXlUeENEMFg2bllPaTN1MUZGZGdRd1VUWEpNZEFZcGRPdERJeVZDZExnVnhLbGhKRG53ZjdOWE1rMlJJUWthN1A1cFV4b2lUVERxYW84cGpiVDVlODFUdGVBcTZnM3J1Zkl5QjZYbHpPdGNOcXZZaWMrUlkyY3J5dHZKNzdrNlhMOU5wbWN1RXV3dUNpcmRMaEU0bkRReHFFT1duamZYSVAwMGZwWk5Ba0NxbDdSaTVtSFljTXgxRWZvMEd1clFGY3JCMlNIR3AxY2lva0FiMkhWNGZ4Vi9TUGhIOEFEVVVCamN4Qjlrd2VSanpNb3NMK3NyVjAxZis0cFhHaUpnclhkUXNGOUhLWm5ZODV5NFkvT3EzQ044TlQ1dHlzRVpGUDVZN3pOK0h5VGU1b3AyUS9yWWYzNS9ETXFFRTBVWkJhanJLRmlNTytuQ1Qwc3RJcDN4a3Fkd3d0SUIwOHVMZGFQdVhsL2VEZlFHWFF5YlRRcjEyYTVXTmRYR2xIb2htc1lvYlB0VktWN0JHOTRJMXVzME16bXEzdXN5VmRWb1ZsZHUrdEg2QlFUaUVJVUplUElvMDhwL0UweU80ak9BRnQ0ZHZGZTJFa1JaTTNaV1pRVUFnRDU5TTNOUlVudXBheXlFREIwOUVHU2lwdFlJb1VZMVZGVGcrNy9SNnh6cGZpZXEzVS9hNFBKVVNKeTVZbzVkK3pZVCtjZnUzTkRGMHVCMkZvZ3VJNUpDaUx6OFhUeStNK3hJK2pQVGFWc3Y0TkFieXB3OHVMaG9Ba052MjZjUGdHRUx2TzgzUms1MFJFdk1CTEFMTm50RlFReHhxL3hQcVNDbUV3bDBNWmJHdlB4K3ZTZTVRczA2enp1aVBTT3dNZFdzMGYzS0dKTEtneE5VL2N1MzNTWmRmUEV2Rm16MzZXdEFNRU9WMTJsNEhVTDNBaVpQMUJyNTRZVFlNMHczazhad2NNbFRHdC9EcmlhdkViaDBVaURoWkcxekE5Nzgvd1lZMnFjYzd4RXUxY2JjNU9xa29DdHlsMGpPUUM0ODVOU1lmL2N1K3VCNDVPNkJCeTVYcWQ0dngyWDVEbmZaZmJjazl3YWpNTkI4cFJ3bFB0S1UzaE1nOUZlWGVDTjhKZTF6UjBOdndNcXh3SFlpbDF3R1FjbWl5UjRLZEFXTkJUZ2hZQnJRN1ZxSmJhUWtnVTloRnVpZ2x0YnZBUitOei9yYlhpVXh6UVFEbHNuaUhBUkxjNlRjdmo4Ujd2RlNFU2I5ckhzZnJGVFRYWm92cUFFRTB5ZWx1RVRkNHBqaVZoNDdkc2NlYzBxdlp0eE9MbE9OdkVhUlVwVDhhTlRsKzlTcWhRTnQ1QmVLYXhla3lxS3h2TG5ETTIxeTl6L3ZZN2RKWCtzeVByMVdLWjF3RFdpUnFtdnU2akVZZ25oaHNwOFNFTG9xVmM5azMxbjhnMjJiZURldTRRL0hPZ3dEaUl6SjJJTnVGZVU3aHluZDBHTVc2NFdHNkYzbDJXMXJzc0VnclhpcnBjQUNabW83ODMxUXdVQ2xreUFjbCtubHVtdHJoSjlWNlN5eFpBSkwrSWpBelNueG1oK1M1b2wrNzJIRnpKQjB5VjhXMTNMTVJQR1BXRUVOZ0xzb1Q3dlNzVk1MU3BMOGN0eithVUlkZDZDZnBWemlMNjFFblZqVisvZFpEOFFyclRDby9PNGdISFFzaWRkd0Q0YzhWQmhydXJVYXdmSmFjYzZyclNWaFZGdytWcGZSdVVmNjFvWkMxQldxSjZSbFkwOHFXdlRzL3FHZnM0L3J5ODJMWmtDM1pXaFIrZ1AyZzRYY3QydTVyT3d2YkhTRDdlc3E3MzlFSjFBMC9kSExZd2hZcVZlQWFBYXRjMk1qTEU3RkplNXBjU2FXWU8zc1l5Y2Z2Z0pLRm9sZXRTaGovN2JkZWdRRDRRN3pldHdubkQrSkYzM0x2S3lHa1NYbEcxbWoyZm16RmNLN2o3YnpaNzBWbUxoUHNDVklEelFmK3FJaTd5L3luN2RrdnFtd0hBMXRuSGdNSWdxUUpQNTNmdmRqQU1NTVcxeTJPcjlxL2cyZXJFRmdITTh4ZkhYR0VPd1ZsbFduLzg3Vng3T2NXRFZ0QjVGbGNZQjNqRis3am1ueXZYeU1UZzg5NTVUWHlGYTlSRnJLVitDK3FObnZDSWpWekI3TlRrdmgrVzltWWdHVEJFckkxV0JNTUFCUGtTSXk0dElRWEhJeEVCYllDSVdEQkUzN3lNb1NjYkhlMFROeFhYV2JtcDBFNndtRWJUOWNGZ1J1dUlDNlhkcUJXcEVDWkZZbmJCQ29hMkppa0NGUVUxbDMxNDR2cXJ4QnJBYVRKeURTa0NvcTJwdUx1WDFQblZKelFPZ25iK2xUcXhPN2Z4R2p2TEVIakVPa1p0cHlSWjFEa3h4VFZqZ2pZbmFiMi9lVjJtcGJ0SDR3M3ZVTFRiQTIxNGRaTy8rR3FRTktLeXhySDkvSGZON1p6N2p2cm96cUY4RWxDNi9wdFJsbEoxN3ZIT09ZcDJxSndVbG9Zb3Y3RHhKSEFsaUltMnNTbWhQNVVlc0lJVjBpVWZ5eUgvTkl4ZkJTR1paNjBxSE0yeC8zOE1IZ0diZCtFU0dpK3RjUGY2aGFpUHoyV3B6UVdDTEljL2pVU2FuaVYwTTAxUGVibklZcjZxMmMwbHpQQzlxZjNVcWxJbVBnZFZ1ZlBCUzZXc0d6b1g2ZU4wTysxb3F2YngrbWZhMGZsYnFOWDIyWGxoSHY4NFc2SGJzU2JSZU5tTEE1M0lDVXNGcXVWaU16Z0tuZ3BUTHZiSFFMT2FHdlRsNDdKNkNrOG5yZFdWdjFWcldBMUZ2c1JNRnJvR29XUjVKbTFoalhLWlhkdDlDU3BnR0JNNnhOdHRxdGxIRzBsNHZ5ZDBBeklHSVI4VTdXSWpiYzlnOTBTQ3Z1YVRwY1M2WnE2WS90Z25KSUx5MFFsNjd3WUpVcms0emJkMVlmS1FIR2tKT2R3TzJVc0J4ZVA1MjZLTE9GVyt4ejUzT3laUTduK2RJNHpicDRsTnkzTk5memVxc1MxTU11Q2pzTExXeWlVb2pYSUl5OWxvSW94MzJMUGVyajN3RmxFdlVxa3BiL3RUSTduMWtqTUprVWZIcGJYcG5Xc0tMbVVDa1E4c0tOQ2VrVjBjck1qa0hqOXVJQTAxRUVvbE1ONG9GN3ZFS0dtSlZUaEVVdjhuSW95OFFWVm5yU0hwYm9na3NvVHljMUZZckN1SlVZaEUwRFhkb2VtRk5KRXVpaDNxa2lFQ3ZBYVdTdVNoZDB2OGdjZk4reXpmM1h5UmRmclB6ZmVhS2owa2tJeXhaSkt6aUNxZHE0WHdJb2ErUGh5ME4zNlRidVdOYS9uTmV6TWM4Z3FldG9TU2JacTRSaHkwQlljelZPMlU0dXRrMWpYcVJOM0tldGdwK1phOEVWREdHOW51cXJHRzdjdWovcGNwVUtmUGF5TmFPaGpJUXhoeHdHRU5rbEowM29MMDF0SVZyRTNqaUNBaG1nOW03d0praGE4SVIxU2Z2N05iU0E3Qm1MTnByaGVQSDMwZFBHSTVKWW9kYWFCeEJmVmtrWjhyK2NVdDFBdXpDWnRPb25UcG5zb09wMDJzY1E3c3pZN1RNdTBhMVArQlVjZ2dORnlJZi9hSzVvMmRjTE9MTEpIWFJZQTlhUFNEeFYveTVSMjVpUUJPWTA4Z0MyUW5jMU95WHBxVXdESDBIZkdrOEZPVTJyd0NjOWdJUFlmT05VQlJ4TUNTRGRielJxTnhJbHFoaklZQyt0UUFpUzVFNFpPVXgrMHdZT1N3cnBlenVidmtPbDJxUTFLME83M0M0K0dyQmZsemJFRHFYMTVYWmVkMW1GaEZ0WUo1NVBpdEhPMzNsRWZrNGZHTDBPYndvVW5LZ0luNDcyUDBaVGZ0UlpzVktBV2pZTGNTWklYaWpzL3JVeWtrTUVweVNIb2ZxdkNsRXE4QVhqTGxkZ2h3ZTQ0YktqUWdXMlZqK2hqU2RrbUZHdU9xZVVUUnhuSCsyNFdpeHNrbzBjTklOSWFLQmJ3UFJWcDZTcUJTek9MbS9DTmV3S2ZQMU92ckhPbk9RUHFwZnpucHdsd3d0ZS9uVHlqYk9xRUF2RTk1ak5SN2NoSC9vVUttd0RWdlRFaFVSQ3FDY0t1aGg1WitOMTZCajFpV1VPNXdwaFlCVU9xQWV6MzB2dHFid2xLN0lmVXB6NlZqQTZzZzdTTFhwSVp1ODYrYTdTMUtxcXFXajM2ZWNlNGVleG9SeGpmVjA1eE9yK09wVXRqaWtoVDRsZk1LVGNJMGx2d0k5Y2dndXVNSHhOYnc0UW0rNFdJNEx5RmROMzUzV3lhbllvMlRlRFlGOVZHclQ1TnpwdVZRSlF5ckZYb01FbE9VS2xjSVpRVGRGVlBLc09jeGIzQS9Vd0J5cUtjRjIvOWJRVjRTS2pJVVEwbDVPbTMvSXdLWGdEMXpDdzVoL29jai9yOGEyWXJQOFp1UkI0L0lCTFlLV2lXVjRGMGlkRVA0eHo0QXJyRUIzWE50WFk0bVhlaHFtU0NobHowSW9ZazZVM3psbXJyOW9NaFNreXhDY2d5ekIxMnFKbk9jUjk2djJCZnRwR0ZoVHZFL2VMZ0JkVW56S2UwM0R5U1ZXdGFDOUJSSnN5ZDFTaXZjYTgwNlkyYkdEbWU3eFBXbmRGU2cvWENvdldaUFY2MmMzRUFNR2dWV2hoSEZYTDlMdHNYWTJLZ0p0OUd2TmZVbHpZS2lCdm1IWU1sdmd4WjUzRUZnWTk1dFdjOG1KaUdtOVFDOGZsM0pqS1Rka3JRUmFsQ1FXZDNyWGpsUXExQ2FGV2lGU2Q4cGJFZHltKzdQREV3a01QUk9LSEI4OENXVWg1ZS94Y3JQdzZGbDZOSDc3aFg3YVFBK0l2QnJsckVzSUdvTjZWa20xR0hKNmtNaXh1aVAzeTY2QkVXWWZaNU9pSG9iTk9Dbk5lZGhtck0zYmdVS3k3aDhJMjdDKzFEQkorSU93STJqVXhpQ2dTSHZzdk9YWWZsQ2lkSzJlYVZIeXo4ZGJNUlpzYk5jMFIyK1FMR3RlTE5lSmJEa09MemhNY3poT3JjS1ZENWwrNFVWaEpqU1oyTWVxN1ZHSGZBengxd3BONGRGWmE3ZGk0cXFqa09tT2QvZmFnR3RuWEwyYklHUldwM0dzS3FSZGhHSUs1a0NmZ2dCRWd1ekoydXZGRjNPM1p1R3kzbkxmbWpoa2tSVnhsNkxRQ0EwRVF4dzlnbWpxY0RBdDFzVGkydWJoU0pOdm80aWh2SE5QYVdmVXlLNXBDcTQzeTgwZFp0Tm8ra1J2N3NEdHZaYVROREd2eDQwWmwvTGs2cmtRVWE5Z3NMblVQRlpIeUxuS1ZQN2JtS3RBRWVpTEVkRkE2UzBycmJEUzl6UlVqMXdVRUNFZHZQME5BU1U1eERDRmlVWkpRc2dOd0NNMnM0KzRHd3FtczNMeVMwbmVwQlp3ZlVjRU1IQVI5V2EvVGNEVXpaWWJRM1lTRkprd04zbkV6blgwMndyMkFOcDdRaGlBYjlWcnB3cENueDZEaTM2REk5bXFzOC95dDc0VHlMaDl5WVRYM3A5UkxNN1VXb3FEazk5N1ptdnU2Y0FuQm82ZXQ0UkVtQVM2bnZiUnR5WFB2MUxpZ1R2SWtma3RxSXdhYVdnL2wzcll0dTNEL1V4a29qS3hzM3VDZ0piYndDUEVCQlV2U2I2cE1hclJDNFRySDByRjRSY2ZxUTlJNzBsc2NhSllOMDE2VUoyQVBvL1R5MEcvOGp0ZytGYzF6cUxCUGxtRzVxK0ozV2VtVWlJZGlMTFh2VUZKNndxTC9CMFBKZmdVT3huWWQ3R1Z3amkwZGxnZU1DdXY5K1VEKzU0MVIrcVRrcXRsR1U1V0dJSDJCQ0tsOGh0YUVUM2Rrc2ZGVXQ1VmV5SUNNc3o2dVpBZ1ZsL1o0dkhJRWZXNURLZVJDR2h5czUvK0NkWUhFUURrVisxZUE3dlhBU0dWSVNQR1AzRFpLSXp1QXdQalVBZnluVVhzUVNObmxxb1FVazdkWldzcXJDRTlzeTl6VXhiRksxbXJKeGN0UkgzaUl5c3g5MGR3SW4wTUhtQXYvRldkZWpLY1FLL3Bpb1RjdktiMTNPbWc2eDBHY0pzaTJmUVJjWDI3ODRMT0ptb3ZENUM0MUxwamh2R2hCY3ZjM2xGSDlYVmp6cnQ2TGpuYVdoVWJZa09QelZzbkQ4ai9GUEhWeVcxMWZnWGpXNE1PNzVKNGN5SUs3eDVTVHZHRUZGMTRSckFJQUVYN290ZU55Q3gvZGlSRjVwSmhHbCszSDRQQVovMlRIY2o0M3BQVUpjR201VmhjenhhTC9TcnR5M1FpVmxWZC85dlYvMXRUVXprODZLZUFjdEpSS1FlT05nQnhLZUFCcy9IcWFnMnBuNlhQMTlNRzV3cmtRMG05aTV4QWxhNFNGUEhPakxiYzVYRDhZMG9uTXRhWHlhY0c1TXJwU09CTHlKTXNLVUtQRDQ2b09YeWRXUDhSN0w2bitEVGdFa3RUS1JwZnNtVU0wQlZrMjR0TkxmZktFZkh4Tk1UMXpnU1AvSEZUSjJRRWhIdm5ZUWN3K0RYUTUxQmZFanNDWWtEMXhoaUxaajFGUFRFUHNzbVRycjdiNTZ1UkhDd2xQeE9CSnBTV0h6ckJNV21laFQ5TkJOdG4rcVl1KzZ4SUlYSms5RFNTVjl1dmV2bE9Fd0JDcmpCRjArR1BtREFpeVdXZi9yZEZ1TWxuVlFCNU11aTQ3enZXU3BiMFFqNHFlM3czSDgwUTE0T3hQMDVabUgvZTVVcTRrK0t1TG1ZZGQ0eklFMDZKY0tkWTVjdHRJcWxtK055ZGJrZE03TkRFYi9nOExEZEIzOHQ3Rm5QMU5vTVhsbGVLY0E3dWZpcld0Ky90VkVKRzZOakZnWXFWVzE0ZnNkOHlBT1NoM3IyTXFiczA4Nnp2engvTW5GajJmSlh1SzN3RVlpcWwvN1VGTytPbHh1TndJMW90RHlDekhoVGkxMUFieGtLVnNEU2VmV1M3Wm80b29oWGI1ZEI5d0tYeGE5dytkbk8zei9waFJkc2N3UEorUXU4L3ZPSDBsLzNKUEtoK1NNNWdMUDd0dmlzT0YwRXFyaUtlbEs1a1VkQlNvTHFsUlFnVzQ2S2N4TGJXU0tSeHhrZ082alpYRFRWUTA1U3FsZzF6VzcyU3pGRFNiTEk5OEM2b2lQNXFBSEFvNXY0b1ZYSHRDcHU4RkE4VTFBM1dlbmd2cy9jeGlzYVBtajRDRmtnaUdiNm5aT0ZhSHVEaXA5N0NMNzZ2VlhZdmtnRWZiSDFpZzYza0VXaEFhOTFreEZ6dURQd2lpZ0NPWWtXYkp5aGZqNU9yV0tEWCtmU0p4SHp3RHkrc1JYa0p6MHI5ZFY3UEhYOWs1WjE4a2o3SFpKMU5PMjZXbjZwT1JobnU0NWpRMSt5TE9tYUZjWW5FNHYvb044VjNNcG00RXhlWEk5bXRpdmZ1ZnJKNHJDakNMSkxTbU15djRCcXJRdXpuTFB1dUtCS21yUkRBQjhLK0lacm5IakQrMENWZ0ZnNjFpL0ovWktheTdVZ2VUVU93VG03NWR1MDRzcGpxclA4dFB2cUFOZEtDNS9WTnFCYm9yRU50djN0UjBvRStnMmVBUEd1Q1UrZGxwUnFqOE40Zy9PblhDbDdObkhwb0pFVjZ2WjAwSjUxZlQySmt0dHIwc0l2OFVTbHBBUGdWRU1BTXhVRUJxTmpOME1udTVKV1NSRXJJUjliMUNNa0o0bUJTUTluaDlSMTVoQTdGMjNCcEIxYnErWUZOSmxVbE5MNWdOL3h3cEwyc0ZaTlBtTXVDM1ZhQ2dUeS9SNFZ1WWRLTjBsaUxLTjBMeDdMcHl6cnhQNDA1bi9uUGhwVGlRdzBPS29uTXY5bDZnenVkUU1RYTAvczhqR0puck5uN0c1QmJlcjFDZzZVVmhtVUY3elQrVUFxTStDd2ZIa3U0aWp1NTRrRm1OSzZNM3htZnVPTlhublptMWtMUUI1TzhxK3dWWnVRaU14NGpMazJVZURhbHhNTk1SbDJidXJwOFlLenp6Y0x6ZDJZaGsxRXNhNEwwMXMvY1R2Q3dUczBLai9ibUlTM1g1UXIrd1JYTldnSGJjTzhtWVNNSnEwSE0zVDhHaG9PWktlTFpuSWNwTXlOcnpTUlZwdzdYREFrYm9Zd0s4YTVvL3FsTHEzbUdNSjVOM3NTc2gweUVtTEpqeUN6dGR5dWM2QU9MRGVVdVVxZjJrR1Z4RkVOUG5wd0l1REE2TjlTTWFISzNHWXR6V2o4Zkk0U1lVZEF0alNiOFVPQzBvRCtmYlBuZGl2M0VENlgrN1pjQytjVWpoRlhWN1YrV3JyTmtRdjNuSXBVK2pvbGMxSkdOckVVQ0JGMk14WHJ3MUF2L3pSZHg0NXdaVlJaZGoyMGMvb0UyZjZUbXJzaE8xcmlyWktoOEQ0Yy9IbzJycCsrK0pMZEh6TVZkcHNxU3lEbVNaK1RvYjNDdTkxT21GQkVtbTh4K09VT0lsSENJR1BucWNRTWNOWEpWVm5DeEpaVkZ0NHpGcjBIdFBkYzRwMnRsWktVYktZWG94MVl2WW80ODZiRGl6WSsrRTVidmFkbEhZTXFRcSs2WlB5ZzhKaU1mK2kwUzhCeW8zM0xwRDBnV1JjT0lPRDNZcnFES0wvY0ZHclQzOVFhVm43VE9NSTNLQVIxSlhpSzQwLzZXNkpreFJGTlpGckhVSTYxNjYrb0ozLyszZkYzUWRmWldSMGhzN1lEMTM1VnBaV0RBNDVJYk13VExVbVZXNHAxOEwvbWt4QTFROWFjSmozT1I0aDVLTlEwYXhQdnMxMmZ6MTA2eGNISzIxeDc0azgxZ0V5aVpqNFkxeW5YWElZblh6Um1XdG96Vzg1R1liY3hZK0w4dzNvUGVyVGFGM1BGc1ZMenYrM20vWGg4MmFhTXBBRHpVY093akt6amY4b1lVejR1Q040elhDK2pWNGRzWVdxeHdta2llQURQOVZCNENDQWlpVnlVTS9UVW0xU3Ruc2V0QUVwTlBGMlg1OXR0RmtLVE9tTEQ2L3pNaXQ1VEpTSEJjK1dXcVVsbEtYQ2d2ZkhkbGtaU2F5Zm9lU21qNDJQOVpnbWg5YTZDdHVwbGJEekdsTGYxVkpFMkVReHZSOFJPenQxQzFOY3UzZFN4NHhEeklYMXJVYUs0MWRHaWttWDZpZnk2cUE1c0k4T2VxR1JlMTQ4Rk5kKzZUd0thM2FBTGRQRnJhbEptanNvRzFPdW84QVdFODBqVlBzWDgwSEtrTk1VMU1ackdvS0k0VVhwT3FVaWJXeHNtaXp0UFkvdFJvTkVXSU9JdnJ0VDVMSVk1NURYSm1XRjdqdUhwaFJZMlNCQjlPeHNQZURWb2ZrN2thTURMMTJ5WnBhaTRaRkZBSkVHcjJ4b0Q1bGRHcHF1ZCtNaThQRDFMbTJyemRRRk5HbkVGWUFZK2xYRU1KSDlLZUdtaW5ZSEdsV2o1MXJ1cDFaNW1QZjh2OEpFYUFZdGFqelM0Nk4xMXh1aWV0QUVQZ0VteWpVVGJEeXBUVUFEemVyWGF5OVViSUdvVWR6YStvODVVTFdnazNPMnZETklZV0VoeDZHY1FWcEpKN2VpR3Avem12dWhEKzV6SlRxZGtHcGU0c1VwZ2JaeDA0RFMxSGVVTytRT1k0Vy8zVDhNRFlzaHB6aFZLSUVzTEhoOUhQQzJINXdVWUVNSHdtVEM1MUNlY1R3a0NXbkZVeTlPMWxvbDRCaHZ1c1JHc1RRZXE1S0toT24vaGI0anA2Z2lhY0plMlU3Y2piVnZZSDM1djRpNFU2TFg0cTVLRmFQUForSjZvZ2RUYVhlWkw2Z3NNRU9SWVF1SmFhd3Q2VnYrYWlqNXBGT0NGMDFWaS81aGRhUG1ZcEtRcjJ5Ni9ZTUNVa1Z3aTNmQkltR1RQR2YyRmZ4UDFERWttSmpEMDFIRHNtYWpLWStzUEJYVHl0QjhJUEpUSmZoY2J4S3ZuWXZ4dFEwVTYwSkVtVlozUThZMnl6R1FjWVRjMWRMbFM1dVlqYjZYQnQ1YmNGV1Z0S1lEVkpobWR0Y3hqT2s0eUo2T2hFVVpJeFJnb3NFL203V1hkK3IzYmJuZHdubEJkRG1OT3RYUW9JNCtxSmcwMFczOTZKQ0Q0V3RwVm5DcDJtMXpST1pza29LNi9sQUJ3M0k5ckgvaTBRRTBHSFhIS21NeGlCM1djcHRYbG14U3NwYjNTTUEzcEFLaFZXRzh4ejQvUzNGSTdMRUU5WFpKSzdKQjFLZk02RUFEQmMxZ1IvelVQR1Y2a3VnWmh6T3ZRcUZnS2tCOWtQbXJwYTZEMGh3elRvRDhpbmNNRWlnbFZTZ1BocCtZTkV4RUd0M25YNS9HRWVwRmM2eGI2M0EyRGJQckZvNUg2N1hsZExBTnM4OHhaVStuOVp0U0piMFdtNjVFNXNOTmhFeUVrPTwveGVuYzpDaXBoZXJWYWx1ZT48L3hlbmM6Q2lwaGVyRGF0YT48L3hlbmM6RW5jcnlwdGVkRGF0YT48L3NhbWw6RW5jcnlwdGVkQXNzZXJ0aW9uPjwvc2FtbHA6UmVzcG9uc2U+
RESPONSE

my $encryptedresponse = $response;

my $sp = net_saml2_sp();

$sp->{cacert} = 't/keycloak-cacert.pem';

my $post = $sp->post_binding;

my $subject;

lives_ok(
    sub {
        $subject = $post->handle_response($response);
    },
    '$sp->handle_response works'
);

my $assertion = Net::SAML2::Protocol::Assertion->new_from_xml(
                        xml => decode_base64($response),
                        key_file => 't/encrypted-sign-private.pem',
                        cacert => 't/keycloak-cacert.pem',
                        );

isa_ok($assertion, 'Net::SAML2::Protocol::Assertion');

is($assertion->in_response_to, 'NETSAML2_935c782d5f5e499638a7471b257062b5');

is(scalar keys %{ $assertion->attributes }, 4);
is(scalar @{ $assertion->attributes->{EmailAddress} }, 1);

is($assertion->session, '91af7e38-e3a6-4be4-9d91-04a32c2ddee2::29499342-7453-4345-b702-68351fcad4f2', 'Session ID is correct');
is($assertion->nameid,  'timlegge@cpan.org', 'Name ID is correct');

cmp_deeply(
    $assertion->attributes,
    {
        EmailAddress => [qw(timlegge@cpan.org)],
        CN           => [qw(keycloak.local)],
        FirstName    => [qw(Timothy)],
        Role         => [qw(manage-account)],
    },
    "Assertion attributes are ok"
);

isa_ok($assertion->not_before, 'DateTime');
isa_ok($assertion->not_after,  'DateTime');

is($assertion->audience, 'https://netsaml2-testapp.local', "Assertion audience is ct.local");
is($assertion->valid('foo'),             0, "foo isn't a valid assertion");
is($assertion->valid('ct.local'), 0, "ct.local isn't valid either");

# fudge validity times to test valid()
$assertion->{not_before} = DateTime->now;
$assertion->{not_after} = DateTime->now->add(minutes => 15);
is($assertion->valid('https://netsaml2-testapp.local'), 1, "https://netsaml2-testapp.local is valid now - InResponseTo not Checked");
is($assertion->valid('https://netsaml2-testapp.local', 'NETSAML2_935c782d5f5e499638a7471b257062b5'), 1, "https://netsaml2-testapp.local is valid now - InResponseTo Checked");
is($assertion->valid('https://netsaml2-testapp.local', 'NETSAML2_935c782d5f5e499638a7471b257062'), 0, "Invalid InResponseTo Checked and failed");

$assertion->{not_before} = DateTime->now->add(minutes => 5);
is($assertion->valid('http://ct.local'), 0, "and invalid again - InResponseTo not Checked");
is($assertion->valid('http://ct.local', 'N3k95Hg41WCHdwc9mqXynLPhB'), 0, "and invalid again - InResponseTo Checked");

done_testing;
