package Finance::Currency::FiatX::Source::gmc;

our $DATE = '2019-01-10'; # DATE
our $VERSION = '0.011'; # VERSION

use 5.010001;
use strict;
use warnings;
use Log::ger;

use Role::Tiny;
with 'Finance::Currency::FiatX::Role::Source';

sub get_all_spot_rates {
    require Finance::Currency::Convert::GMC;
    my $res = Finance::Currency::Convert::GMC::get_currencies();

    return $res unless $res->[0] == 200;

    my @recs;
    for my $to (sort keys %{ $res->[2]{currencies} }) {
        my $h = $res->[2]{currencies}{$to};
        push @recs, (
            {
                pair => "$to/IDR",
                type => "buy",
                rate => $h->{buy},
                mtime => $res->[2]{mtime},
            },
            {
                pair => "$to/IDR",
                type => "sell",
                rate => $h->{sell},
                mtime => $res->[2]{mtime},
            },
            {
                pair => "IDR/$to",
                type => "buy",
                rate => 1/$h->{sell},
                note => "1/sell",
                mtime => $res->[2]{mtime},
            },
            {
                pair => "IDR/$to",
                type => "sell",
                rate => 1/$h->{buy},
                note => "1/buy",
                mtime => $res->[2]{mtime},
            },
        );
    }

    [200, "OK", \@recs];
}

sub get_spot_rate {
    my %args = @_;

    my $from = $args{from} or return [400, "Please specify from"];
    my $to   = $args{to} or return [400, "Please specify to"];
    my $type = $args{type} or return [400, "Please specify type"];

    return [501, "This source only provides buy/sell rate types"]
        unless $type =~ /\A(buy|sell)\z/;
    return [501, "This source only provides IDR/* or */IDR spot rates"]
        unless $from eq 'IDR' || $to eq 'IDR';

    require Finance::Currency::Convert::GMC;
    my $res = Finance::Currency::Convert::GMC::get_currencies();
    return $res unless $res->[0] == 200;

    my $h = $res->[2]{currencies}{$to} || $res->[2]{currencies}{$from};
    return [404, "Cannot find rate for $from/$to"] unless $h;

    my $rate = {
        pair => "$from/$to",
        mtime => $res->[2]{mtime},
        type => $type,
    };
    ;
    if ($from eq 'IDR') {
        if ($type eq 'buy') {
            $rate->{rate} = 1/$h->{sell};
            $rate->{note} = "1/sell";
        } elsif ($type eq 'sell') {
            $rate->{rate} = 1/$h->{buy};
            $rate->{note} = "1/buy";
        }
    } else {
        if ($type eq 'buy') {
            $rate->{rate} = $h->{buy};
            $rate->{note} = "buy";
        } elsif ($type eq 'sell') {
            $rate->{rate} = $h->{sell};
            $rate->{note} = "sell";
        }
    }

    [200, "OK", $rate];
}

sub get_historical_rate {
    return [501, "This source does not provide historical rates"];
}

1;
# ABSTRACT: Get currency conversion rates from GMC (Golden Money Changer)

__END__

=pod

=encoding UTF-8

=head1 NAME

Finance::Currency::FiatX::Source::gmc - Get currency conversion rates from GMC (Golden Money Changer)

=head1 VERSION

This document describes version 0.011 of Finance::Currency::FiatX::Source::gmc (from Perl distribution Finance-Currency-FiatX), released on 2019-01-10.

=head1 DESCRIPTION

Golden Money Changer is a popular money changer in Jakarta/Bandung, Indonesia.

=for Pod::Coverage ^(.+)$

=head1 BUGS

Please report all bug reports or feature requests to L<mailto:stevenharyanto@gmail.com>.

=head1 SEE ALSO

L<https://www.gmc.co.id/>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2019, 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
