% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/search_regression.R
\name{regression_opt}
\alias{regression_opt}
\title{Stepwise Multiple Regression Model Search based on Information Criteria}
\usage{
regression_opt(
  data = NULL,
  n = NULL,
  mat = NULL,
  dep_ind,
  n_calc = "individual",
  ic_type = "bic",
  ordered = FALSE,
  missing_handling = "stacked-mi",
  nimp = 20,
  imp_method = "pmm",
  ...
)
}
\arguments{
\item{data}{Raw data matrix or data frame containing the variables to be
included in the regression models. May include missing values. If \code{data} is
\code{NULL}, a covariance or correlation matrix must be supplied in \code{mat}.}

\item{n}{Numeric value specifying the sample size used in calculating the
information criteria. If not provided, it is derived from \code{data}. When \code{mat}
is supplied instead of raw data, \code{n} must be provided.}

\item{mat}{Optional covariance or correlation matrix for the variables to be
included in the regression. Used only when \code{data} is \code{NULL}.}

\item{dep_ind}{Index of the column in \code{data} to be used as the dependent
variable in the regression model.}

\item{n_calc}{Character string specifying how the sample size is calculated
when \code{n} is not provided. Possible values are:
\describe{
\item{\code{"individual"}}{Uses the number of non-missing observations for the
variable used as the dependent variable.}
\item{\code{"average"}}{Uses the average number of non-missing observations
across all variables.}
\item{\code{"max"}}{Uses the maximum number of non-missing observations across
all variables.}
\item{\code{"total"}}{Uses the total number of rows in \code{data}.}
}}

\item{ic_type}{Type of information criterion to compute for model selection.
Options are \code{bic} (default), \code{aic}, \code{aicc}.}

\item{ordered}{Logical vector indicating whether each variable in \code{data}
should be treated as ordered categorical when computing the correlation
matrix. If a single logical value is supplied, it is recycled to all
variables. Only used when \code{data} is provided.}

\item{missing_handling}{Character string specifying how the correlation
matrix is estimated from \code{data} in the presence of missing values. Possible
values are:
\describe{
\item{\code{"two-step-em"}}{Uses a classical EM algorithm to estimate the
correlation matrix from \code{data}.}
\item{\code{"stacked-mi"}}{Uses stacked multiple imputation to estimate the
correlation matrix from \code{data}.}
\item{\code{"pairwise"}}{Uses pairwise deletion to compute correlations from
\code{data}.}
\item{\code{"listwise"}}{Uses listwise deletion to compute correlations from
\code{data}.}
}}

\item{nimp}{Number of imputations (default: 20) to be used when
\code{missing_handling = "stacked-mi"}.}

\item{imp_method}{Character string specifying the imputation method to be
used when \code{missing_handling = "stacked-mi"} (default: \code{"pmm"} - predictive
mean matching).}

\item{...}{Further arguments passed to internal functions.}
}
\value{
A list with the following elements:
\describe{
\item{regression}{Named vector of regression coefficients for the dependent variable.}
\item{R2}{R-squared value of the regression model.}
\item{n}{Sample size used in the regression model.}
\item{args}{List of settings used in the regression model.}
}
}
\description{
Performs stepwise model selection for multiple regression using information
criteria to identify the optimal regression model.
}
\details{
\loadmathjax

This function performs stepwise model selection for multiple regression
using information criteria. It was originally developed as a component of
the neighborhood selection framework for network estimation
\insertCite{nehler.2024}{mantar}, where each node-wise regression model is
selected individually. However, the procedure can also be used as a
standalone tool for exploratory regression model search, particularly in
settings with missing data. Unlike standard stepwise regression functions,
this implementation explicitly supports missing-data handling strategies,
making it suitable for situations in which classical methods fail or produce
biased results.

The argument \code{ic_type} specifies which information criterion is computed.
All criteria are computed based on the log-likelihood of the maximum
likelihood estimated regression model, where the residual variance
determines the likelihood. The following options are available:

\describe{

\item{\code{"aic"}:}{
Akaike Information Criterion \insertCite{akaike.1974}{mantar}; defined as
\mjseqn{\mathrm{AIC} = -2 \ell + 2k},
where \eqn{\ell} is the log-likelihood of the model and \eqn{k} is the
number of estimated parameters (including the intercept).
}

\item{\code{"bic"}:}{
Bayesian Information Criterion \insertCite{schwarz.1978}{mantar}; defined as
\mjseqn{\mathrm{BIC} = -2 \ell + k \log(n)}, where \eqn{\ell} is
the log-likelihood of the model, \eqn{k} is the
number of estimated parameters (including the intercept)
and \eqn{n} is the sample size.
}

\item{\code{"aicc"}:}{
Corrected Akaike Information Criterion \insertCite{hurvich.1989}{mantar};
particularly useful in small samples where AIC tends to be biased.
Defined as
\mjseqn{\mathrm{AIC_c} = \mathrm{AIC} + \frac{2k(k+1)}{n - k - 1}},
where \eqn{k} is the number of estimated parameters (including
the intercept) and \eqn{n} is the sample size.
}

}
}
\examples{
# For full data using AIC
# First variable of the data set as dependent variable
result <- regression_opt(
  data = mantar_dummy_full_cont,
  dep_ind = 1,
  ic_type = "aic"
)

# View regression coefficients and R-squared
result$regression
result$R2

# For data with missingess using BIC
# Second variable of the data set as dependent variable
# Using individual sample size of the dependent variable and stacked Multiple Imputation

result_mis <- regression_opt(
 data = mantar_dummy_mis_cont,
 dep_ind = 2,
 n_calc = "individual",
 missing_handling = "two-step-em",
 ic_type = "bic"
 )

 # View regression coefficients and R-squared
 result_mis$regression
 result_mis$R2
}
\references{
\insertAllCited{}
}
