# jmSurface

## Semi-Parametric Association Surfaces for Joint Longitudinal-Survival Models

[![License: GPL v3](https://img.shields.io/badge/License-GPLv3-blue.svg)](https://www.gnu.org/licenses/gpl-3.0)

**jmSurface** implements interpretable multi-biomarker fusion in joint longitudinal-survival models via semi-parametric association surfaces for multi-state disease processes.

### Key Features

- **Two-stage estimation**: Stage 1 fits mixed-effects longitudinal models (nlme); Stage 2 fits transition-specific penalized Cox models with tensor-product spline surfaces (mgcv)
- **Multi-state support**: Transition-specific association surfaces with automatic REML smoothing
- **EDF diagnostics**: Data-driven assessment of nonlinearity (EDF ≈ 1 = linear; EDF > 3 = nonlinear/interaction)
- **Dynamic prediction**: Personalized transition probabilities from landmark time
- **Three visualization tools**: 3D surface plots, contour heatmaps, marginal effect slices
- **Interactive Shiny dashboard**: Upload data, explore, fit models, and compute personalized risks
- **Built-in simulator**: Generate realistic CKD/CVD/Diabetes multi-state cohort data

### Installation

```r
# Install from CRAN
install.packages("jmSurface")
```

### Quick Start

```r
library(jmSurface)

# ── Option A: Use bundled CKD/CVD/Diabetes cohort (2,000 patients) ──
dat <- load_example_data()

fit <- jmSurf(
  long_data = dat$long_data,
  surv_data = dat$surv_data,
  covariates = c("age_baseline", "sex")
)

# Summary with EDF diagnostics
summary(fit)

# EDF table
edf_diagnostics(fit)

# Dynamic prediction for patient 1
pred <- dynPred(fit, patient_id = 1, landmark = 2, horizon = 3)
head(pred)

# Visualize association surfaces
plot_surface(fit, transition = "CKD -> CVD")
contour_heatmap(fit, transition = "CKD -> CVD")
marginal_slices(fit, transition = "CKD -> CVD")

# ── Option B: Simulate fresh data ──
sim <- simulate_jmSurface(n_patients = 500)
fit2 <- jmSurf(sim$long_data, sim$surv_data)

# ── Option C: Launch interactive Shiny app ──
run_shiny_app()
```

### Bundled Data

The package ships with a real-scale example dataset (2,000 patients):
- `longitudinal_biomarkers.csv`: 68,112 biomarker measurements (eGFR, BNP, HbA1c)
- `survival_events.csv`: 4,701 survival rows with 9 transition types

Access via:
```r
dat <- load_example_data()           # convenience loader (returns list)
# OR directly:
path <- system.file("extdata", "longitudinal_biomarkers.csv", package = "jmSurface")
long <- read.csv(path)
```

### Model Overview

For each transition (r → s) in a multi-state process:

```
λᵢʳˢ(t | ηᵢ(t)) = λ₀ʳˢ(t) · exp{γᵣₛ'wᵢ + fᵣₛ(ηᵢ(t))}
```

where:
- `ηᵢ(t)` = BLUP-based latent longitudinal summaries (biomarker-specific)
- `fᵣₛ(·)` = semi-parametric association surface (tensor-product splines)
- Smoothing via REML; complexity controlled by EDF

### Package Functions

| Function | Description |
|----------|-------------|
| `jmSurf()` | Main fitting function (two-stage) |
| `fit_longitudinal()` | Stage 1: LME models per biomarker |
| `fit_gam_cox()` | Stage 2: GAM-Cox with spline surface |
| `edf_diagnostics()` | EDF table with complexity labels |
| `dynPred()` | Dynamic prediction of transition probabilities |
| `plot_surface()` | 3D perspective surface plot |
| `contour_heatmap()` | Filled contour heatmap |
| `marginal_slices()` | Marginal effect slices at quantiles |
| `simulate_jmSurface()` | Simulate CKD/CVD/Diabetes data |
| `run_shiny_app()` | Launch interactive Shiny dashboard |

### References

Bhattacharjee, A. (2025). Interpretable Multi-Biomarker Fusion in Joint Longitudinal-Survival Models via Semi-Parametric Association Surfaces. *Under review*.

Bhattacharjee, A. (2024). jmBIG: Scalable Joint Models for Big Data. *CRAN*.

Wood, S.N. (2017). *Generalized Additive Models: An Introduction with R.* Chapman & Hall/CRC.

### License

GPL (>= 3)
