\name{init_val}
\alias{g3_init_val}
\concept{G3 compilation}

\title{Gadget3 parameter value setter}
\description{
  Helper for setting initial parameter value
}

\usage{
g3_init_val(
        param_template,
        name_spec,
        value = NULL,
        spread = NULL,
        lower = if (!is.null(spread)) min(value * (1-spread), value * (1+spread)),
        upper = if (!is.null(spread)) max(value * (1-spread), value * (1+spread)),
        optimise = !is.null(lower) & !is.null(upper),
        parscale = if (is.null(lower) || is.null(upper)) NULL else 'auto',
        random = NULL,
        auto_exponentiate = TRUE)
}

\arguments{
  \item{param_template}{
    A parameter template generated by \code{\link{g3_to_r}} or \code{\link{g3_to_tmb}}
  }
  \item{name_spec}{
    A glob-like string to match parameter names, see Details
  }
  \item{value}{
    Numeric value / vector of values to set for value / 'value' column in template.
    Original value left if NULL
  }
  \item{spread}{
    Shortcut for setting \var{lower} & \var{upper}.
  }
  \item{lower}{
    Numeric value / vector of values to set for 'lower' column in template.
    Original value left if NULL
  }
  \item{upper}{
    Numeric value / vector of values to set for 'upper' column in template.
    Original value left if NULL
  }
  \item{optimise}{
    Boolean value to set for 'optimise' column in template.
    Default is true iff both lower and upper are non-NULL.
    Original value left if NULL
  }
  \item{parscale}{
    Numeric value / vector of values to set for 'parscale' column in template.
    Default (\code{auto}) is difference between lower & upper (or NULL if they're not set).
    Original value left if NULL
  }
  \item{random}{
    Boolean value to set for 'random' column in template.
    If set to TRUE, any existing optimise/lower/upper/parscale values will be cleared.
    Original value left if NULL
  }
  \item{auto_exponentiate}{
    If TRUE, will implicitly match parameters ending with "_exp",
    and if this is the case \code{log} all \var{value}/\var{lower}/\var{upper} values
  }
}

\details{
  \var{name_spec} is a glob (or wildcard) matching parameters.
  It is a string separated by \code{.}, where each part can be:
  \enumerate{
    \item{A wildcard matching anything (\code{*}), or a matching anything with a prefix, e.g. \code{m*}}
    \item{A wildcard matching any number (\code{#}), or a matching a number with a prefix, e.g. \code{age*}}
    \item{A range of numbers, e.g. \code{[1979-1984]}}
    \item{A choice of options can be separated with \code{|}, e.g. \code{init|rec} or \code{[1979-1984]|[2000-2003]}}
  }
}

\value{A new parameter template list/table containing modifications}

\seealso{
  \code{\link{g3_parameterized}}
}

\examples{
# A parameter template, would already be got via. attr(g3_to_tmb(...), "parameter_template")
pt <- data.frame(
    switch = c(
        paste0('fish.init.', 1:9),
        paste0('fish.rec.', 1990:2000),
        'fish.M'),
    value = NA,
    lower = NA,
    upper = NA,
    parscale = NA,
    optimise = FALSE,
    random = FALSE)

# Set all fish.init.# parameters to optimise
pt <- g3_init_val(pt, 'fish.init.#', 4, spread = 8)

# Set a fixed value for any .M
pt <- g3_init_val(pt, '*.M', value = 0.3, optimise = FALSE)

# Set a fixed value for a range of recruitment years, optimise the rest
pt |>
    g3_init_val('*.rec.#', value = 4, lower = 0, upper = 10) |>
    g3_init_val('*.rec.[1993-1996]', value = 0, optimise = FALSE) |>
    identity() -> pt

pt
}
