#' Estimation of the time-dependent ROC curve for correlated right-censored survival data
#'
#' @description  {This function computes a time-dependent ROC curve for correlated right censored survival data using the cumulative sensitivity and dynamic specificity definitions.
#' The ROC curves can be either empirical (non-smoothed) or smoothed with/wtihout boundary correction. It also calculates the time-dependent area under the ROC curve (AUC).}
#'
#' @param Y a numeric vector of event-times or observed times.
#' @param M a numeric vector of (bio)marker or risk score values.
#' @param censor a vector of censoring indicator, \code{1} if event, \code{0} otherwise.
#' @param group a categorical vector of group/cluster.
#' @param t  a scalar time for prediction. The default value is \code{0.0001}.
#' @param w a scalar window for prediction.
#' @param U a vector of grid points where the ROC curve is estimated. The default is a sequence of \code{151} numbers between \code{0} and \code{1}.
#' @param bw a character string specifying the bandwidth estimation method for the ROC itself. The possible options are "\code{NR}" for the normal reference, the plug-in "\code{PI}" and the cross-validation "\code{CV}". The default is the "\code{NR}" normal reference method. The user can also give a numerical value.
#' @param len a scalar value specifying the length of vector \code{U}. The default value is \code{151}.
#' @param method a character string specifying the method of ROC curve estimation. The possible options are "\code{emp}" empirical method; "\code{untra}" smooth without boundary correction and "\code{tra}" is smooth ROC curve estimation with boundary correction. The default is the "\code{tra}" smooth ROC curve estimate with boundary correction.
#' @param method1  a character string specifying prediction method applied on model. The possible options are "\code{cox}" for the classical Cox; "\code{marg}" for  marginal and "\code{cond}" conditional prediction methods on shared models. The default is "\code{cond}".
#' @param ktype a character string specifying the type kernel distribution to be used for smoothing the ROC curve: "\code{normal}", "\code{epanechnikov}", "\code{biweight}", or "\code{triweight}". By default, the "\code{normal}" kernel is used.
#' @param knots a scalar for specifying the number of knots to use. Value required in the penalized likelihood estimation. It corresponds to the (knots+2) splines functions for the approximation of the hazard or the survival functions. Rondeau, et al. (2012) suggested that the number of knots must be between 4 and 20. The default is \code{10}.
#' @param kappa a positive smoothing parameter value for the penalized likelihood estimation. The defaults is "\code{10000}".
#' @param RandDist a character string to state the distribution of random effect: "\code{Gamma}" for a gamma distribution, "\code{LogN}" for a log-normal distribution. Default is "\code{Gamma}".
#' @param hazard types of hazard functions: "\code{Splines}" represents a semi-parametric hazard function using equidistant intervals and is estimated via penalized likelihood, "\code{Splines-per}" uses percentiles instead of equidistant intervals, "\code{Piecewise-per}" is a piecewise constant hazard function based on percentiles, "\code{Piecewise-equi}" is a piecewise constant hazard using equidistant intervals, and "\code{Weibull}" is a fully parametric hazard function based on the Weibull distribution. "\code{Splines}" is used as the default setting.
#' @param maxit maximum number of iterations. The default is \code{300}.
#' @param B a number of bootstrap samples to be used for variance estimation. The default is \code{0}, no variance estimation.
#' @param alpha the significance level. The default is \code{0.05}.
#' @param plot a logical parameter to see the ROC curve plot. The default is \code{TRUE}.
#'
#' @details This function takes correlated right-censored survival data and returns an empirical (non-smoothed) ROC estimate and the smoothed time-dependent ROC estimate with/without boundary correction and the corresponding time-dependent AUC estimates.
#' For the smoothing parameter estimation, three data-driven methods: the normal reference "\code{NR}", the plug-in "\code{PI}" and the cross-validation "\code{CV}" introduced in Beyene and El Ghouch (2020) were implemented. See Beyene and El Ghouch (2020) for details.
#' The conditional survival function estimation can done by using semi-parametric or fully parametric shared frailty models.
#'
#' @return Returns the following items:
#'
#'  \item{ROC }{vector of estimated ROC values. These will be numeric values between zero and one.}
#'
#'  \item{AUC }{data frame of dimension \eqn{1 \times 4}. The columns are: AUC, standard error of AUC, the lower and upper limits of bootstrap CI.}
#'
#'  \item{U  }{vector of grid points used.}
#'
#'  \item{bw  }{computed value of bandwidth parameter. For the empirical method this is always \code{NA}.}
#'
#'  \item{Dt }{vector of estimated event status.}
#'
#'  \item{M }{vector of (bio)marker vlaues.}
#'
#' @importFrom stats pnorm qnorm quantile approx integrate sd
#' @import frailtypack
#' @importFrom graphics abline legend segments text lines polygon
#' @importFrom survival Surv
#'
#' @examples
#'   \donttest{
#'   library(frailtyROC)
#'
#'  data(kidney)
#'
#'  out1 <- frailtyROC(Y=kidney$time, M=kidney$Marker, censor=kidney$status, group = kidney$id, w=120,
#'         method = "emp", method1 = "cond", hazard = "Weibull")
#'
#'  out1$AUC
#' }
#'
#' @references Beyene, K. M., and Chen, D. G. (2024). Time-dependent receiver operating characteristic curve estimator for correlated right-censored time-to-event data. \emph{Statistical Methods in Medical Research}, 33(1), 162-181.
#' @references Beyene, K.M. and El Ghouch A. (2020). Smoothed time-dependent receiver operating characteristic curve for right censored survival data. \emph{Statistics in Medicine}. 39: 3373-3396.
#' @references Rondeau, V., Marzroui, Y., & Gonzalez, J. R. (2012). frailtypack: an R package for the analysis of correlated survival data with frailty models using penalized likelihood estimation or parametrical estimation. \emph{Journal of Statistical Software}, 47, 1-28.
#' @export

frailtyROC <- function(Y, M, censor, group=NULL, w, t = 0.0001, U = NULL, bw = "NR",  len = 151, method = "tra",
              method1 = "marg", ktype = "normal", knots = 10, kappa = 10000, RandDist = "Gamma",
              hazard = "Splines", maxit = 300, B = 0, alpha = 0.05, plot = "TRUE")
  {
    if (is.null(U)) {U <- seq(0, 1, length.out = len)}
    if (!is.vector(Y, mode = "numeric") |
        !is.vector(M, mode = "numeric") |
        !is.vector(censor, mode = "numeric"))
      warning("Error! all numeric vectors Y, M and censor should be specified")
    else{
      Dt <- Csurv(Y = Y, M = M, censor = censor, group = group, t = t, w = w, method = method1, knots = knots, kappa = kappa, RandDist = RandDist, hazard = hazard, maxit = maxit)$positive
      estim <- RocFun(U = U, D = Dt, M = M, method = method, bw = bw, ktype = ktype)
      ROC <- estim$roc
      AUCc <- 1 - estim$auc
      AUC <- data.frame(AUC = round(AUCc, 4) , sd = NA, LCL = NA, UCL = NA)
    }

  if (B > 0){
    aucb <- NULL
    rocb <- matrix(NA, nrow = length(U), ncol = B)
    if (method1=="cox"){
      data <- data.frame(Y=Y, M=M, censor=censor)
    } else {
      data <- data.frame(Y=Y, M=M, censor=censor, group=group)}
    for (i in 1:B){
      bootsample <- sample(1:nrow(data), nrow(data), replace=TRUE)
      dat <- data[bootsample, ]
      if (method1=="cox"){Dt <- Csurv(Y = dat$Y, M = dat$M, censor = dat$censor, t = t, w = w, method = method1, knots = knots, kappa = kappa, RandDist = RandDist, hazard = hazard, maxit = maxit)$positive}
      else{Dt <- Csurv(Y = dat$Y, M = dat$M, censor = dat$censor, group = dat$group, t = t, w = w, method = method1, knots = knots, kappa = kappa, RandDist = RandDist, hazard = hazard, maxit = maxit)$positive}
      estim <- RocFun(U = U, D = Dt, M = dat$M, method = method, bw = bw, ktype = ktype)
      aucb[i] <- 1 - estim$auc
      rocb[, i] <- estim$roc
    }
    SP <- unname(quantile(aucb, p=c(alpha/2, 1-alpha/2), na.rm = TRUE))
    AUC <- data.frame(AUC = round(AUCc, 4) , sd = round(sd(aucb, na.rm = TRUE), 4), LCL = round(SP[1], 4), UCL = round(SP[2], 4))
    qroc <- unname(apply(rocb, 1, quantile, probs = c(alpha/2, 1-alpha/2), na.rm = TRUE))
    ROC <- data.frame(ROC = ROC, LCL = qroc[1, ], UCL = qroc[2, ])
  }


  if (plot == "TRUE") {
    if (B==0){
      plot(c(0, U, 1), c(0, (ROC), 1), type = "l", lwd = 2, col.lab = "blue", col = "blue",
           xlab = "False positive rate", ylab = "True positive rate")
      abline(coef = c(0,1))
    }else{
      plot(c(0, U, 1), c(0, (ROC[, 1]), 1), type = "l", lwd = 2, col.lab = "blue", col = "blue",
           xlab = "False positive rate", ylab = "True positive rate")
      polygon(c(rev(c(0,U,1)), c(0,U,1)), c(rev(c(0,ROC[, 3],1)), c(0,ROC[, 2],1)), col = 'darkslategray1', border = NA)
      lines(c(0,U,1), c(0, (ROC[, 1]), 1), lty = 'solid', col = 'blue', lwd = 2)
      lines(c(0,U,1), c(0, (ROC[, 3]), 1), lty = 'dashed', col = 'red', lwd = 2)
      lines(c(0,U,1), c(0, (ROC[, 2]), 1), lty = 'dashed', col = 'red', lwd = 2)
    }
  }
  return(list(AUC = AUC, ROC = ROC, U = U, Dt = Dt, M = M, bw = estim$bw))
  }
