% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gcd.R
\name{gcd}
\alias{gcd}
\alias{scm}
\alias{gcd2}
\alias{scm2}
\title{Greatest common divisor and smallest common multiple}
\usage{
gcd(
  x,
  tol = sqrt(.Machine$double.eps),
  na_rm = TRUE,
  round = TRUE,
  break_early = TRUE
)

scm(x, tol = sqrt(.Machine$double.eps), na_rm = TRUE)

gcd2(x, y, tol = sqrt(.Machine$double.eps), na_rm = TRUE)

scm2(x, y, tol = sqrt(.Machine$double.eps), na_rm = TRUE)
}
\arguments{
\item{x}{A \link{numeric} vector.}

\item{tol}{Tolerance. This must
be a single positive number strictly less than 1.}

\item{na_rm}{If \code{TRUE} the default, \code{NA} values are ignored.}

\item{round}{If \code{TRUE} the output is rounded as
\code{round(gcd, digits)} where digits is
\code{ceiling(abs(log10(tol))) + 1}. \cr
This can potentially reduce floating point errors on
further calculations. \cr
The default is \code{TRUE}.}

\item{break_early}{This is experimental and
applies only to floating-point numbers.
When \code{TRUE} the algorithm will end once \code{gcd > 0 && gcd < 2 * tol}.
This can offer a tremendous speed improvement.
If \code{FALSE} the algorithm finishes once it has gone through all elements of \code{x}.
The default is \code{TRUE}. \cr
For integers, the algorithm always breaks early once \code{gcd > 0 && gcd <= 1}.}

\item{y}{A \link{numeric} vector.}
}
\value{
A number representing the GCD or SCM.
}
\description{
Fast greatest common divisor and smallest common multiple
using the Euclidean algorithm.

\code{gcd()} returns the greatest common divisor. \cr
\code{scm()} returns the smallest common multiple. \cr
\code{gcd2()} is a vectorised binary version of \code{gcd}. \cr
\code{scm2()} is a vectorised binary version of \code{scm}.
}
\details{
\subsection{Method}{
\subsection{GCD (Greatest Common Divisor)}{

The GCD is calculated using a binary function that takes input
\code{GCD(gcd, x[i + 1])} where the output of this function is passed as input
back into the same function iteratively along the length of \code{x}.
The first gcd value is \code{x[1]}.

Zeroes are handled in the following way: \cr
\code{GCD(0, 0) = 0} \cr
\code{GCD(a, 0) = a} \cr

This has the nice property that zeroes are essentially ignored.
}

\subsection{SCM (Smallest Common Multiple)}{

This is calculated using the GCD and the formula is: \cr
\code{SCM(x, y) = (abs(x) / GCD(x, y) ) * abs(y)}

If you want to calculate the gcd & lcm for 2 values
or across 2 vectors of values, use \code{gcd2} and \code{scm2}.
}

\subsection{A note on performance}{

A very common solution to finding the GCD of a vector of values is to use
\code{Reduce()} along with a binary function like \code{gcd2()}. \cr
e.g. \code{Reduce(gcd2, seq(5, 20, 5))}. \cr
This is exactly identical to \code{gcd(seq(5, 20, 5))}, with \code{gcd()} being much
faster and overall cheaper as it is written in C++ and heavily optimised.
Therefore it is recommended to always use \code{gcd()}.

For example we can compare the two approaches below, \cr
\code{x <- seq(5L, length = 10^6, by = 5L)} \cr
\code{bench::mark(Reduce(gcd2, x), gcd(x))} \cr
This example code shows \code{gcd()} being ~200x faster on my machine than
the \code{Reduce} + \code{gcd2} approach, even though \code{gcd2} itself is written in C++
and has little overhead.
}

}
}
\examples{
library(cheapr)
library(bench)

# Binary versions
gcd2(15, 25)
gcd2(15, seq(5, 25, 5))
scm2(15, seq(5, 25, 5))
scm2(15, 25)

# GCD across a vector
gcd(c(0, 5, 25))
mark(gcd(c(0, 5, 25)))

x <- rnorm(10^5)
gcd(x)
gcd(x, round = FALSE)
mark(gcd(x))

}
