% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brsmm.R
\name{brsmm}
\alias{brsmm}
\title{Fit a mixed-effects beta interval regression model}
\usage{
brsmm(
  formula,
  random = ~1 | id,
  data,
  link = "logit",
  link_phi = "logit",
  repar = 2L,
  ncuts = 100L,
  lim = 0.5,
  int_method = c("laplace", "aghq", "qmc"),
  n_points = 11L,
  qmc_points = 1024L,
  start = NULL,
  method = c("BFGS", "L-BFGS-B"),
  hessian_method = c("numDeriv", "optim"),
  control = list(maxit = 2000L)
)
}
\arguments{
\item{formula}{Model formula. Supports one- or two-part formulas:
\code{y ~ x1 + x2} or \code{y ~ x1 + x2 | z1 + z2}.}

\item{random}{Random-effects specification of the form
\code{~ terms | group}, e.g. \code{~ 1 | id} or \code{~ 1 + x | id}.}

\item{data}{Data frame.}

\item{link}{Mean link function.}

\item{link_phi}{Precision link function.}

\item{repar}{Beta reparameterization code (0, 1, 2).}

\item{ncuts}{Number of categories on the original scale.}

\item{lim}{Half-width used to construct interval endpoints.}

\item{int_method}{Integration method: \code{"laplace"} (default),
\code{"aghq"}, or \code{"qmc"}.}

\item{n_points}{Number of quadrature points for \code{int_method="aghq"}.
Ignored for other methods. Default is 11.}

\item{qmc_points}{Number of QMC points for \code{int_method="qmc"}.
Default is 1024.}

\item{start}{Optional numeric vector of starting values
(\code{beta}, \code{gamma}, and packed lower-Cholesky random parameters).}

\item{method}{Optimizer passed to \code{\link[stats]{optim}}.}

\item{hessian_method}{\code{"numDeriv"} (default) or \code{"optim"}.}

\item{control}{Control list for \code{\link[stats]{optim}}.}
}
\value{
An object of class \code{"brsmm"}.
}
\description{
Fits a beta interval-censored mixed model with Gaussian random
intercepts/slopes using marginal maximum likelihood. The implementation supports
random-effects formulas such as \code{~ 1 | group} and \code{~ 1 + x | group},
and offers three integration methods for the
random effects: Laplace approximation, Adaptive Gauss-Hermite Quadrature
(AGHQ), and Quasi-Monte Carlo (QMC).
}
\details{
The conditional contribution for each observation follows the same mixed
censoring likelihood used by \code{\link{brs}}:

\enumerate{
  \item \eqn{\delta=0}: exact contribution via beta density,
  \item \eqn{\delta=1}: left-censored contribution via beta CDF,
  \item \eqn{\delta=2}: right-censored contribution via survival CDF,
  \item \eqn{\delta=3}: interval contribution via CDF difference.
}

For group \eqn{i}, the random-effects vector
\eqn{\mathbf{b}_i \sim N(\mathbf{0}, D)} is integrated out numerically.

\itemize{
  \item \code{"laplace"}: Uses a second-order Laplace approximation at the
    conditional mode. Fast and generally accurate for \eqn{n_i} large.
  \item \code{"aghq"}: Adaptive Gauss-Hermite Quadrature. Uses \code{n_points}
    quadrature nodes centered and scaled by the conditional mode and curvature.
    More accurate than Laplace, especially for small \eqn{n_i}.
  \item \code{"qmc"}: Quasi-Monte Carlo integration using a Halton sequence.
    Uses \code{qmc_points} evaluation points. Suitable for high-dimensional
    integration (future proofing) or checking robustness.
}
}
\examples{
\donttest{
dat <- data.frame(
  y = c(
    0, 5, 20, 50, 75, 90, 100, 30, 60, 45,
    10, 40, 55, 70, 85, 25, 35, 65, 80, 15
  ),
  x1 = rep(c(1, 2), 10),
  id = factor(rep(1:4, each = 5))
)
prep <- brs_prep(dat, ncuts = 100)
fit_mm <- brsmm(y ~ x1, random = ~ 1 | id, data = prep)
fit_mm
}

}
\references{
Lopes, J. E. (2023). \emph{Modelos de regressao beta para dados de escala}.
Master's dissertation, Universidade Federal do Parana, Curitiba.
URI: \url{https://hdl.handle.net/1884/86624}.

Ferrari, S. L. P., and Cribari-Neto, F. (2004).
Beta regression for modelling rates and proportions.
\emph{Journal of Applied Statistics}, \bold{31}(7), 799--815.
\doi{10.1080/0266476042000214501}
}
