% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NonparTrawlEstimation.R
\name{trawl_est}
\alias{trawl_est}
\title{Bias-corrected nonparametric estimation of the trawl function}
\usage{
trawl_est(data, Delta)
}
\arguments{
\item{data}{Data to be used in the trawl function estimation.}

\item{Delta}{Width of the grid on which we observe the data}
}
\value{
trawlfct Returns the original nonparametric trawl function estimate
obtained from \code{\link{nonpar_trawlest}}.

trawlfct_biascor Returns the bias-corrected trawl function estimate,
computed as the original estimate minus the bias correction term
\eqn{0.5 \times \Delta \times \hat{a}'(t)}.
}
\description{
This function implements bias correction for the nonparametric trawl
estimation proposed in Sauri and Veraart (2022).
}
\details{
This function performs bias correction on the nonparametric trawl
function estimates obtained using the methodology of Sauri and Veraart (2022).
The bias correction is implemented by estimating the derivative of the trawl
function and applying the correction \eqn{0.5 \times \Delta \times
\hat{a}'(t)},
where \eqn{\hat{a}'(t)} is the estimated derivative at lag \eqn{t}.

Suppose the data is observed on the grid
0, Delta, 2Delta, ..., (n-1)Delta. Given the path contained in data,
the function returns both the original and bias-corrected estimates of the
trawl function.

The bias correction is based on the theoretical result that the leading
bias term in nonparametric trawl function estimation is proportional to
Delta times the derivative of the true trawl function.
}
\examples{
\donttest{
  #Simulation of a Gaussian trawl process with exponential trawl function
  n<-2000
  Delta<-0.5

  trawlfct_par <-0.5
  distr<-"Gauss"
  distr_par<-c(0,1) #mean 0, std 1
  set.seed(233)

  a <- function(x){exp(-trawlfct_par*x)}
  path <- sim_weighted_trawl_gen(n, Delta, a,
                                 distr, distr_par)$path
  #Plot the path
  library(ggplot2)
  df <- data.frame(time = seq(0,n,1), value=path)
  p <- ggplot(df, aes(x=time, y=path))+
    geom_line()+
    xlab("l")+
    ylab("Trawl process")
  p



  result <- trawl_est(path, Delta)

  original_estimate <- result$trawlfct[1:30]
  bias_corrected <- result$trawlfct_biascor[1:30]



  # True exponential values for the same lags
  lags <- 0:(length(original_estimate)-1)
  true_vals <- a(lags * Delta)

  # Plot all three
  plot(lags, original_estimate,
       type = "l", col = "blue", lwd = 2,
       xlab = "Lag index", ylab = "Trawl function estimate",
       main = "Trawl function: Original vs bias-corrected vs true",
       ylim = range(c(original_estimate, bias_corrected, true_vals), na.rm = TRUE))
  lines(lags, bias_corrected,
        col = "red", lwd = 2, lty = 2)
  lines(lags, true_vals,
        col = "black", lwd = 2, lty = 3)

  legend("topright",
         legend = c("Original", "Bias-corrected", "True exponential"),
         col = c("blue", "red", "black"),
         lty = c(1, 2, 3), lwd = 2)
}


}
\seealso{
\code{\link{nonpar_trawlest}}, \code{\link{trawl_deriv_mod}}
}
