% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_vars_cat.R
\name{derive_vars_cat}
\alias{derive_vars_cat}
\title{Derive Categorization Variables Like \code{AVALCATy} and \code{AVALCAyN}}
\usage{
derive_vars_cat(dataset, definition, by_vars = NULL)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} and \code{definition} arguments are expected to be in the dataset.

\describe{
\item{Default value}{none}
}}

\item{definition}{List of expressions created by \code{exprs()}.
Must be in rectangular format and specified using the same syntax as when creating
a \code{tibble} using the \code{tribble()} function.
The \code{definition} object will be converted to a \code{tibble} using \code{tribble()} inside this function.

Must contain:
\itemize{
\item the column \code{condition} which will be converted to a logical expression and
will be used on the \code{dataset} input.
\item at least one additional column with the new column name and
the category value(s) used by the logical expression.
\item the column specified in \code{by_vars} (if \code{by_vars} is specified)
}

e.g. if \code{by_vars} is not specified:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exprs(~condition,   ~AVALCAT1, ~AVALCA1N,
      AVAL >= 140, ">=140 cm",         1,
      AVAL < 140,   "<140 cm",         2)
}\if{html}{\out{</div>}}

e.g. if \code{by_vars} is specified as \code{exprs(VSTEST)}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exprs(~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N,
      "Height", AVAL >= 140, ">=140 cm",         1,
      "Height",  AVAL < 140,  "<140 cm",         2)
}\if{html}{\out{</div>}}

\describe{
\item{Default value}{none}
}}

\item{by_vars}{list of expressions with one element. \code{NULL} by default.
Allows for specifying by groups, e.g. \code{exprs(PARAMCD)}.
Variable must be present in both \code{dataset} and \code{definition}.
The conditions in \code{definition} are applied only to those records that match \code{by_vars}.
The categorization variables are set to \code{NA} for records
not matching any of the by groups in \code{definition}.

\describe{
\item{Default value}{\code{NULL}}
}}
}
\value{
The input dataset with the new variables defined in \code{definition} added
}
\description{
Derive Categorization Variables Like \code{AVALCATy} and \code{AVALCAyN}
}
\details{
If conditions are overlapping, the row order of \code{definitions} must be carefully considered.
The \strong{first} match will determine the category.
i.e. if

\code{AVAL = 155}

and the \code{definition} is:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition <- exprs(
  ~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N,
  "Height",  AVAL > 170,  ">170 cm",         1,
  "Height", AVAL <= 170, "<=170 cm",         2,
  "Height", AVAL <= 160, "<=160 cm",         3
)
}\if{html}{\out{</div>}}

then \code{AVALCAT1} will be \code{"<=170 cm"}, as this is the first match for \code{AVAL}.
If you specify:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition <- exprs(
  ~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N,
  "Height", AVAL <= 160, "<=160 cm",         3,
  "Height", AVAL <= 170, "<=170 cm",         2,
  "Height",  AVAL > 170,  ">170 cm",         1
)
}\if{html}{\out{</div>}}

Then \code{AVAL <= 160} will lead to \code{AVALCAT1 == "<=160 cm"},
\code{AVAL} in-between \code{160} and \code{170} will lead to \code{AVALCAT1 == "<=170 cm"},
and \code{AVAL <= 170} will lead to \code{AVALCAT1 == ">170 cm"}.

However, we suggest to be more explicit when defining the \code{condition}, to avoid overlap.
In this case, the middle condition should be:
\code{AVAL <= 170 & AVAL > 160}
}
\seealso{
General Derivation Functions for all ADaMs that returns variable appended to dataset:
\code{\link{derive_var_extreme_flag}()},
\code{\link{derive_var_joined_exist_flag}()},
\code{\link{derive_var_merged_ef_msrc}()},
\code{\link{derive_var_merged_exist_flag}()},
\code{\link{derive_var_obs_number}()},
\code{\link{derive_var_relative_flag}()},
\code{\link{derive_vars_computed}()},
\code{\link{derive_vars_joined}()},
\code{\link{derive_vars_joined_summary}()},
\code{\link{derive_vars_merged}()},
\code{\link{derive_vars_merged_lookup}()},
\code{\link{derive_vars_merged_summary}()},
\code{\link{derive_vars_transposed}()}
}
\concept{der_gen}
\keyword{der_gen}
\section{Examples}{
\subsection{Data setup}{

The following examples use the ADVS dataset below as a basis. It contains
vital signs data with some missing values (\code{NA}) that will demonstrate how the
function handles different scenarios.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(dplyr)
library(tibble)

advs <- tibble::tribble(
  ~USUBJID,       ~VSTEST,  ~AVAL,
  "01-701-1015", "Height", 147.32,
  "01-701-1015", "Weight",  53.98,
  "01-701-1023", "Height", 162.56,
  "01-701-1023", "Weight",     NA,
  "01-701-1028", "Height",     NA,
  "01-701-1028", "Weight",     NA,
  "01-701-1033", "Height", 175.26,
  "01-701-1033", "Weight",  88.45
)}\if{html}{\out{</div>}}}
\subsection{Derive categorization variables without \code{by_vars}}{

In this example, we derive \code{AVALCAT1}, \code{AVALCA1N}, and \code{NEWCOL} without
using \code{by_vars}. The conditions must include all necessary filtering logic,
such as checking both \code{VSTEST} and \code{AVAL}. Records that don't match any
condition will have \code{NA} values for the new variables.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition <- exprs(
  ~condition,                        ~AVALCAT1, ~AVALCA1N,  ~NEWCOL,
  VSTEST == "Height" & AVAL > 160,   ">160 cm",         1, "extra1",
  VSTEST == "Height" & AVAL <= 160, "<=160 cm",         2, "extra2"
)

derive_vars_cat(
  dataset = advs,
  definition = definition
)
#> # A tibble: 8 × 6
#>   USUBJID     VSTEST  AVAL AVALCAT1 AVALCA1N NEWCOL
#>   <chr>       <chr>  <dbl> <chr>       <dbl> <chr> 
#> 1 01-701-1015 Height 147.  <=160 cm        2 extra2
#> 2 01-701-1015 Weight  54.0 <NA>           NA <NA>  
#> 3 01-701-1023 Height 163.  >160 cm         1 extra1
#> 4 01-701-1023 Weight  NA   <NA>           NA <NA>  
#> 5 01-701-1028 Height  NA   <NA>           NA <NA>  
#> 6 01-701-1028 Weight  NA   <NA>           NA <NA>  
#> 7 01-701-1033 Height 175.  >160 cm         1 extra1
#> 8 01-701-1033 Weight  88.4 <NA>           NA <NA>  }\if{html}{\out{</div>}}}
\subsection{Derive categorization variables using \code{by_vars}}{

When using \code{by_vars}, the conditions are automatically scoped to records
matching each by group value. This simplifies the condition logic as you don't
need to include the by variable in each condition. Here we derive categories
for both Height and Weight measurements using \code{by_vars = exprs(VSTEST)}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition2 <- exprs(
  ~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N,
  "Height",  AVAL > 160,  ">160 cm",         1,
  "Height", AVAL <= 160, "<=160 cm",         2,
  "Weight",   AVAL > 70,   ">70 kg",         1,
  "Weight",  AVAL <= 70,  "<=70 kg",         2
)

derive_vars_cat(
  dataset = advs,
  definition = definition2,
  by_vars = exprs(VSTEST)
)
#> # A tibble: 8 × 5
#>   USUBJID     VSTEST  AVAL AVALCAT1 AVALCA1N
#>   <chr>       <chr>  <dbl> <chr>       <dbl>
#> 1 01-701-1015 Height 147.  <=160 cm        2
#> 2 01-701-1015 Weight  54.0 <=70 kg         2
#> 3 01-701-1023 Height 163.  >160 cm         1
#> 4 01-701-1023 Weight  NA   <NA>           NA
#> 5 01-701-1028 Height  NA   <NA>           NA
#> 6 01-701-1028 Weight  NA   <NA>           NA
#> 7 01-701-1033 Height 175.  >160 cm         1
#> 8 01-701-1033 Weight  88.4 >70 kg          1}\if{html}{\out{</div>}}}
\subsection{Using multiple conditions with explicit ranges}{

When you need more than two categories, you can define multiple conditions.
It's best practice to make conditions mutually exclusive using explicit range
definitions (e.g., \code{AVAL <= 170 & AVAL > 160}) to avoid ambiguity, even though
the function uses first-match logic.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition3 <- exprs(
  ~VSTEST,                ~condition,  ~AVALCAT1, ~AVALCA1N,
  "Height",               AVAL > 170,  ">170 cm",         1,
  "Height", AVAL <= 170 & AVAL > 160, "<=170 cm",         2,
  "Height",              AVAL <= 160, "<=160 cm",         3
)

derive_vars_cat(
  dataset = advs,
  definition = definition3,
  by_vars = exprs(VSTEST)
)
#> # A tibble: 8 × 5
#>   USUBJID     VSTEST  AVAL AVALCAT1 AVALCA1N
#>   <chr>       <chr>  <dbl> <chr>       <dbl>
#> 1 01-701-1015 Height 147.  <=160 cm        3
#> 2 01-701-1015 Weight  54.0 <NA>           NA
#> 3 01-701-1023 Height 163.  <=170 cm        2
#> 4 01-701-1023 Weight  NA   <NA>           NA
#> 5 01-701-1028 Height  NA   <NA>           NA
#> 6 01-701-1028 Weight  NA   <NA>           NA
#> 7 01-701-1033 Height 175.  >170 cm         1
#> 8 01-701-1033 Weight  88.4 <NA>           NA}\if{html}{\out{</div>}}}
\subsection{Deriving categories based on reference ranges (\code{MCRITy} variables)}{

This example demonstrates deriving laboratory measurement criteria
variables (\code{MCRITyML} and \code{MCRITyMN}). The conditions use reference range
variables (like \code{ANRHI}) to create categories relative to normal ranges,
which is common in laboratory data analysis.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{adlb <- tibble::tribble(
  ~USUBJID,     ~PARAM, ~AVAL, ~AVALU,  ~ANRHI,
  "01-701-1015", "ALT",   150,  "U/L",      40,
  "01-701-1023", "ALT",    70,  "U/L",      40,
  "01-701-1036", "ALT",   130,  "U/L",      40,
  "01-701-1048", "ALT",    30,  "U/L",      40,
  "01-701-1015", "AST",    50,  "U/L",      35
)

definition_mcrit <- exprs(
  ~PARAM,                      ~condition,    ~MCRIT1ML, ~MCRIT1MN,
  "ALT",                    AVAL <= ANRHI,    "<=ANRHI",         1,
  "ALT", ANRHI < AVAL & AVAL <= 3 * ANRHI, ">1-3*ANRHI",         2,
  "ALT",                 3 * ANRHI < AVAL,   ">3*ANRHI",         3
)

adlb \%>\%
  derive_vars_cat(
    definition = definition_mcrit,
    by_vars = exprs(PARAM)
  )
#> # A tibble: 5 × 7
#>   USUBJID     PARAM  AVAL AVALU ANRHI MCRIT1ML   MCRIT1MN
#>   <chr>       <chr> <dbl> <chr> <dbl> <chr>         <dbl>
#> 1 01-701-1015 ALT     150 U/L      40 >3*ANRHI          3
#> 2 01-701-1023 ALT      70 U/L      40 >1-3*ANRHI        2
#> 3 01-701-1036 ALT     130 U/L      40 >3*ANRHI          3
#> 4 01-701-1048 ALT      30 U/L      40 <=ANRHI           1
#> 5 01-701-1015 AST      50 U/L      35 <NA>             NA}\if{html}{\out{</div>}}}
\subsection{Handling missing values and partial by groups}{

When using \code{by_vars}, records that don't match any by group in the
\code{definition} will have \code{NA} for all derived variables. In this example,
records with \code{VSTEST == "Weight"} will have \code{NA} values because only
"Height" conditions are defined. Additionally, records with missing \code{AVAL}
will result in \code{NA} for the categorization variables since conditions
cannot be evaluated.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition4 <- exprs(
  ~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N,
  "Height",  AVAL > 160,  ">160 cm",         1,
  "Height", AVAL <= 160, "<=160 cm",         2
)

derive_vars_cat(
  dataset = advs,
  definition = definition4,
  by_vars = exprs(VSTEST)
)
#> # A tibble: 8 × 5
#>   USUBJID     VSTEST  AVAL AVALCAT1 AVALCA1N
#>   <chr>       <chr>  <dbl> <chr>       <dbl>
#> 1 01-701-1015 Height 147.  <=160 cm        2
#> 2 01-701-1015 Weight  54.0 <NA>           NA
#> 3 01-701-1023 Height 163.  >160 cm         1
#> 4 01-701-1023 Weight  NA   <NA>           NA
#> 5 01-701-1028 Height  NA   <NA>           NA
#> 6 01-701-1028 Weight  NA   <NA>           NA
#> 7 01-701-1033 Height 175.  >160 cm         1
#> 8 01-701-1033 Weight  88.4 <NA>           NA}\if{html}{\out{</div>}}}
\subsection{Deriving multiple categorization variables simultaneously}{

You can derive any number of categorization variables in a single call.
This example creates three different categorization schemes (\code{AVALCAT1},
\code{AVALCAT2}, and \code{AVALCAT3}) with their corresponding numeric flags, all
from the same set of conditions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{definition5 <- exprs(
  ~VSTEST,   ~condition,  ~AVALCAT1, ~AVALCA1N, ~AVALCAT2,      ~AVALCA2N, ~AVALCAT3,
  "Height",  AVAL > 160,  ">160 cm",         1,    "Tall",              1,   "Group A",
  "Height", AVAL <= 160, "<=160 cm",         2,   "Short",              2,   "Group B",
  "Weight",   AVAL > 70,   ">70 kg",         3,   "Heavy",              3,   "Group C",
  "Weight",  AVAL <= 70,  "<=70 kg",         4,   "Light",              4,   "Group D"
)

derive_vars_cat(
  dataset = advs,
  definition = definition5,
  by_vars = exprs(VSTEST)
)
#> # A tibble: 8 × 8
#>   USUBJID     VSTEST  AVAL AVALCAT1 AVALCA1N AVALCAT2 AVALCA2N AVALCAT3
#>   <chr>       <chr>  <dbl> <chr>       <dbl> <chr>       <dbl> <chr>   
#> 1 01-701-1015 Height 147.  <=160 cm        2 Short           2 Group B 
#> 2 01-701-1015 Weight  54.0 <=70 kg         4 Light           4 Group D 
#> 3 01-701-1023 Height 163.  >160 cm         1 Tall            1 Group A 
#> 4 01-701-1023 Weight  NA   <NA>           NA <NA>           NA <NA>    
#> 5 01-701-1028 Height  NA   <NA>           NA <NA>           NA <NA>    
#> 6 01-701-1028 Weight  NA   <NA>           NA <NA>           NA <NA>    
#> 7 01-701-1033 Height 175.  >160 cm         1 Tall            1 Group A 
#> 8 01-701-1033 Weight  88.4 >70 kg          3 Heavy           3 Group C }\if{html}{\out{</div>}}}}

