% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filters.R
\name{filter_wavelet}
\alias{filter_wavelet}
\title{Wavelet Multiresolution Analysis Filter}
\usage{
filter_wavelet(
  y,
  wf = "la8",
  J = 4,
  boundary = "periodic",
  levels_to_combine = c(3, 4),
  first_difference = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{y}{Numeric vector of the time series to decompose. Length must be at
least \code{2^J}.}

\item{wf}{Character string specifying the wavelet filter. Options include
"la8" (least asymmetric with 8 vanishing moments, 16 coefficients),
"la16", "la20", "haar", "d4", "d6", "d8", etc. Default is "la8".}

\item{J}{Integer specifying the decomposition depth (number of levels).
Default is 4, yielding D1-D4 detail levels plus S4 smooth level.}

\item{boundary}{Character string specifying boundary handling: "periodic"
(default) or "reflection".}

\item{levels_to_combine}{Integer vector specifying which detail levels to
combine for the trend estimate. Default is \code{c(3, 4)} for D3+D4.}

\item{first_difference}{Logical. If TRUE, applies wavelet to first
differences and reconstructs via cumulative sum. Default is FALSE.}

\item{verbose}{Logical indicating whether to print diagnostic messages.}
}
\value{
A list of class "signaly_wavelet" containing:
\describe{
\item{trend}{Numeric vector of the extracted trend component}
\item{mra}{Full multiresolution analysis object from waveslim::mra}
\item{detail_levels}{Data frame with all detail level coefficients}
\item{smooth_level}{Vector of the smooth (SJ) coefficients}
\item{combined_levels}{Character string indicating which levels were
combined}
\item{settings}{List of parameters used in the analysis}
\item{diagnostics}{List with variance decomposition and energy
distribution}
}
}
\description{
Performs wavelet-based signal decomposition using the Maximal Overlap
Discrete Wavelet Transform (MODWT) to extract trend components at specified
frequency scales. This method decomposes the signal into detail coefficients
(D1, D2, ..., DJ) capturing progressively lower frequencies and a smooth
coefficient (SJ) representing the underlying trend.
}
\details{
The MODWT (Maximal Overlap Discrete Wavelet Transform) is preferred over
the classical DWT for several reasons relevant to signal extraction:

\enumerate{
\item \strong{Translation invariance}: Unlike DWT, MODWT does not depend on
the starting point of the series, producing consistent results regardless
of circular shifts.
\item \strong{Any sample size}: MODWT can be applied to series of any length,
not just powers of 2.
\item \strong{Additive decomposition}: The MRA (multiresolution analysis)
coefficients sum exactly to the original series.
}

The choice of wavelet filter affects the trade-off between time and frequency
localization:
\itemize{
\item \strong{la8 (Daubechies least asymmetric, 8 vanishing moments)}: Good
balance of smoothness and localization, recommended for economic data.
\item \strong{Higher order (la16, la20)}: Better frequency resolution at cost
of temporal smearing.
\item \strong{haar}: Maximum time localization but poor frequency resolution.
}
}
\section{Frequency Interpretation}{

For a series with unit sampling interval, the detail levels correspond to
approximate frequency bands:
\itemize{
\item D1: periods 2-4 (highest frequency noise)
\item D2: periods 4-8 (short-term fluctuations)
\item D3: periods 8-16 (medium-term cycles)
\item D4: periods 16-32 (longer cycles)
\item S4: periods > 32 (smooth trend)
}

For annual economic data, D3+D4 typically captures business cycle dynamics
(8-32 year periods), while D1+D2 captures short-term noise.
}

\examples{
set.seed(123)
y <- cumsum(rnorm(100)) + sin(seq(0, 4*pi, length.out = 100))
result <- filter_wavelet(y, wf = "la8", J = 4)
plot(y, type = "l", col = "gray")
lines(result$trend, col = "red", lwd = 2)

}
\references{
Daubechies, I. (1992). Ten Lectures on Wavelets. SIAM.

Percival, D. B., & Walden, A. T. (2000). Wavelet Methods for Time Series
Analysis. Cambridge University Press.

Gencay, R., Selcuk, F., & Whitcher, B. (2002). An Introduction to Wavelets
and Other Filtering Methods in Finance and Economics. Academic Press.
}
\seealso{
\code{\link[waveslim]{mra}}, \code{\link{filter_emd}},
\code{\link{filter_hpgc}}
}
