% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_random_table_multi.R
\name{svem_random_table_multi}
\alias{svem_random_table_multi}
\title{Generate a Random Prediction Table from Multiple SVEMnet Models (no refit)}
\usage{
svem_random_table_multi(
  objects,
  n = 1000,
  mixture_groups = NULL,
  debias = FALSE,
  range_tol = 1e-08
)
}
\arguments{
\item{objects}{A list of fitted \code{svem_model} objects returned by \code{SVEMnet()}.
Each object must contain \code{$sampling_schema} produced by the updated \code{SVEMnet()}.
A single model is also accepted and treated as a length-one list.}

\item{n}{Number of random points to generate. Default is \code{1000}.}

\item{mixture_groups}{Optional list of mixture constraint groups. Each group is a list
with elements \code{vars}, \code{lower}, \code{upper}, \code{total}. The variables in
\code{vars} must be numeric-like predictors present in all models. The sampler uses a
truncated Dirichlet so that elementwise bounds are respected and \code{sum(vars) = total}.}

\item{debias}{Logical; if \code{TRUE}, apply each model's calibration during prediction
when available. Default is \code{FALSE}.}

\item{range_tol}{Numeric tolerance for comparing numeric ranges across models.
Default is \code{1e-8}.}
}
\value{
A list with three data frames:
\itemize{
\item \code{data}: the sampled predictor settings, one row per random point.
\item \code{pred}: one column per response, aligned to \code{data} rows.
\item \code{all}: \code{cbind(data, pred)} for convenience.
}
Each prediction column is named by the model's response (left-hand side). If a response
name would collide with a predictor name, \code{".pred"} is appended.
}
\description{
Samples the original predictor factor space cached in fitted \code{svem_model}
objects and computes predictions from each model at the same random points.
Intended for multiple responses built over the same factor space and a
deterministic factor expansion.
}
\details{
All models must share an identical predictor schema:
\itemize{
\item The same \code{predictor_vars} in the same order
\item The same \code{var_classes} for each predictor
\item Identical factor \code{levels} and level order
\item Numeric \code{ranges} that match within \code{range_tol}
}
The function stops with an informative error message if any of these checks fail.
}
\section{Sampling strategy}{

Non-mixture numeric variables are sampled with a maximin Latin hypercube over the
cached numeric ranges. Mixture variables are sampled jointly within each specified
group using a truncated Dirichlet so that elementwise bounds and the total sum
are satisfied. Categorical variables are sampled from the cached factor levels.
The same random predictor table is fed to each model so response columns are directly comparable.
}

\section{Notes on mixtures}{

Each mixture group should list only numeric-like variables. Bounds are interpreted
on the original scale of those variables. If \code{total} equals the sum of lower
bounds, the sampler returns the lower-bound corner for that group.
}

\examples{
\donttest{
set.seed(1)
n <- 60
X1 <- runif(n); X2 <- runif(n)
A <- runif(n); B <- runif(n); C <- pmax(0, 1 - A - B)
F <- factor(sample(c("lo","hi"), n, TRUE))
y1 <- 1 + 2*X1 - X2 + 3*A + 1.5*B + 0.5*C + (F=="hi") + rnorm(n, 0, 0.3)
y2 <- 0.5 + 0.8*X1 + 0.4*X2 + rnorm(n, 0, 0.2)
d  <- data.frame(y1, y2, X1, X2, A, B, C, F)

fit1 <- SVEMnet(y1 ~ X1 + X2 + A + B + C + F, d, nBoot = 40)
fit2 <- SVEMnet(y2 ~ X1 + X2 + A + B + C + F, d, nBoot = 40)

# Mixture constraint for A, B, C that sum to 1
mix <- list(list(vars=c("A","B","C"),
                 lower=c(0,0,0), upper=c(1,1,1), total=1))

res <- svem_random_table_multi(
  list(fit1, fit2), n = 200, mixture_groups = mix,
  debias = FALSE
)
head(res$all)

# Lipid screening example with composition group
data(lipid_screen)
spec <- bigexp_terms(
  Potency ~ PEG + Helper + Ionizable + Cholesterol +
    Ionizable_Lipid_Type + N_P_ratio + flow_rate,
  data = lipid_screen, factorial_order = 3
)
fP <- bigexp_formula(spec, "Potency")
fS <- bigexp_formula(spec, "Size")
fD <- bigexp_formula(spec, "PDI")
mP <- SVEMnet(fP, lipid_screen, nBoot = 30)
mS <- SVEMnet(fS, lipid_screen, nBoot = 30)
mD <- SVEMnet(fD, lipid_screen, nBoot = 30)

mixL <- list(list(
  vars  = c("Cholesterol","PEG","Ionizable","Helper"),
  lower = c(0.10, 0.01, 0.10, 0.10),
  upper = c(0.60, 0.05, 0.60, 0.60),
  total = 1
))

tab <- svem_random_table_multi(
  objects        = list(Potency = mP, Size = mS, PDI = mD),
  n              = 1000,
  mixture_groups = mixL,
  debias         = FALSE
)
head(tab$all)
}

}
\seealso{
\code{SVEMnet}, \code{predict.svem_model}
}
