#' Compute Softmax Plant Stress Response Index (PSRI_SM)
#'
#' Calculates the softmax-aggregated PSRI from time-series germination data.
#' This method handles zero-value components gracefully through adaptive
#' softmax reweighting, eliminating the zero-collapse failure mode of the
#' geometric mean approach.
#'
#' @param germination_counts Integer vector of cumulative germination counts
#'   at each timepoint. Must be non-decreasing and not exceed
#'   \code{total_seeds}.
#' @param timepoints Numeric vector of observation times (e.g., days).
#'   Must be the same length as \code{germination_counts} and strictly
#'   increasing.
#' @param total_seeds Integer. Total number of seeds in the replicate.
#' @param radicle_count Integer or \code{NULL}. Number of seeds with
#'   radicle emergence at the final observation. If \code{NULL}, radicle
#'   vigor is excluded (3-component PSRI). Default is \code{NULL}.
#' @param temperature Numeric. Softmax temperature parameter. Default is
#'   0.13, calibrated via perplexity targeting. See
#'   \code{\link{calibrate_temperature}}.
#' @param return_components Logical. If \code{TRUE}, returns a list
#'   containing the PSRI value, all component scores, and softmax weights.
#'   Default is \code{FALSE}.
#'
#' @return If \code{return_components = FALSE} (default), a single numeric
#'   PSRI value. If \code{return_components = TRUE}, a list with:
#'   \describe{
#'     \item{psri_sm}{The softmax PSRI value.}
#'     \item{components}{Named numeric vector of component scores.}
#'     \item{weights}{Named numeric vector of softmax weights.}
#'     \item{temperature}{The temperature parameter used.}
#'     \item{n_components}{Number of components used (3 or 4).}
#'   }
#'
#' @details
#' The softmax PSRI aggregates germination components using weighted
#' summation:
#' \deqn{PSRI_{SM} = \sum W_i \cdot C_i}
#' where \eqn{C = [MSG, MRG, cMTG]} (3-component) or
#' \eqn{C = [MSG, MRG, cMTG, RVS]} (4-component), and weights are
#' computed via:
#' \deqn{W_i = \frac{\exp(C_i / T)}{\sum_j \exp(C_j / T)}}
#'
#' \strong{Zero handling:} When any component \eqn{C_k = 0}, its softmax
#' weight approaches zero and the remaining components absorb its weight.
#' This preserves the information from non-zero components rather than
#' collapsing the entire index to zero.
#'
#' \strong{Components:}
#' \itemize{
#'   \item \code{MSG}: Maximum Stress-adjusted Germination (fraction
#'     germinated)
#'   \item \code{MRG}: Maximum Rate of Germination (scaled average rate)
#'   \item \code{cMTG}: Complementary Mean Time to Germination
#'     (\eqn{1 - t_{50}/t_{max}})
#'   \item \code{RVS}: Radicle Vigor Score (radicle count / total seeds,
#'     continuous 0-1; included only when \code{radicle_count} is provided)
#' }
#'
#' @examples
#' # Basic 3-component PSRI (no radicle data)
#' compute_psri_sm(
#'   germination_counts = c(5, 15, 20),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = 25
#' )
#'
#' # With radicle vigor (4-component)
#' compute_psri_sm(
#'   germination_counts = c(5, 15, 20),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = 25,
#'   radicle_count = 18
#' )
#'
#' # Zero germination: softmax preserves non-zero components
#' compute_psri_sm(
#'   germination_counts = c(0, 0, 5),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = 25,
#'   return_components = TRUE
#' )
#'
#' # Compare: geometric would collapse to zero if MRG = 0
#' result <- compute_psri_sm(
#'   germination_counts = c(0, 0, 10),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = 25,
#'   radicle_count = 3,
#'   return_components = TRUE
#' )
#' print(result$components)
#' print(result$weights)
#'
#' @seealso
#' \code{\link{compute_msg}}, \code{\link{compute_mrg}},
#' \code{\link{compute_cmtg}}, \code{\link{compute_rvs}} for individual
#' component calculations.
#' \code{\link{softmax_weights}} for the weighting function.
#' \code{\link{calibrate_temperature}} for temperature selection.
#'
#' @references
#' Walne, C.H., Gaudin, A., Henry, W.B., and Reddy, K.R. (2020).
#' In vitro seed germination response of corn hybrids to osmotic stress
#' conditions. \emph{Agrosystems, Geosciences & Environment}, 3(1), e20087.
#' \doi{10.1002/agg2.20087}
#'
#' @export
compute_psri_sm <- function(germination_counts,
                            timepoints,
                            total_seeds,
                            radicle_count = NULL,
                            temperature = 0.13,
                            return_components = FALSE) {

  # --- Input validation ---
  if (!is.numeric(germination_counts) || length(germination_counts) == 0) {
    stop("germination_counts must be a non-empty numeric vector.")
  }
  if (!is.numeric(timepoints) || length(timepoints) == 0) {
    stop("timepoints must be a non-empty numeric vector.")
  }
  if (length(germination_counts) != length(timepoints)) {
    stop("germination_counts and timepoints must have the same length.")
  }
  if (!is.numeric(total_seeds) || length(total_seeds) != 1 || total_seeds <= 0) {
    stop("total_seeds must be a single positive number.")
  }
  if (any(germination_counts < 0)) {
    stop("germination_counts must be non-negative.")
  }
  if (max(germination_counts) > total_seeds) {
    stop("germination_counts cannot exceed total_seeds.")
  }
  if (!is.numeric(temperature) || length(temperature) != 1 ||
      temperature <= 0) {
    stop("temperature must be a single positive number.")
  }

  # --- Compute components ---
  msg  <- compute_msg(germination_counts, total_seeds)
  mrg  <- compute_mrg(germination_counts, timepoints, total_seeds)
  cmtg <- compute_cmtg(germination_counts, timepoints, total_seeds)

  # Build component vector
  if (!is.null(radicle_count) && !is.na(radicle_count)) {
    rvs <- compute_rvs(radicle_count, total_seeds)
    components <- c(MSG = msg, MRG = mrg, cMTG = cmtg, RVS = rvs)
  } else {
    components <- c(MSG = msg, MRG = mrg, cMTG = cmtg)
  }

  # --- Softmax aggregation ---
  weights <- softmax_weights(components, temperature = temperature)
  psri_sm <- sum(weights * components)

  # --- Return ---
  if (return_components) {
    return(list(
      psri_sm      = psri_sm,
      components   = components,
      weights      = weights,
      temperature  = temperature,
      n_components = length(components)
    ))
  }

  psri_sm
}
