# Perks.R
#' Perks Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Perks distribution.
#'
#' The Perks distribution is parameterized by the parameters
#' \eqn{\alpha > 0} and \eqn{\beta > 0}.
#'
#' @details
#' The Perks distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta) =
#' \quad 1 - \left( {\frac{{1 + \alpha }}{{1 + \alpha {e^{\beta x}}}}} \right)
#' \, ; \quad x \ge 0.
#' }
#'
#' where \eqn{\alpha} and \eqn{\beta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dperks()} — Density function
#'   \item \code{pperks()} — Distribution function
#'   \item \code{qperks()} — Quantile function
#'   \item \code{rperks()} — Random generation
#'   \item \code{hperks()} — Hazard function
#' }
#'
#' @usage
#' dperks(x, alpha, beta, log = FALSE)
#' pperks(q, alpha, beta, lower.tail = TRUE, log.p = FALSE)
#' qperks(p, alpha, beta, lower.tail = TRUE, log.p = FALSE)
#' rperks(n, alpha, beta)
#' hperks(x, alpha, beta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dperks}: numeric vector of (log-)densities
#'   \item \code{pperks}: numeric vector of probabilities
#'   \item \code{qperks}: numeric vector of quantiles
#'   \item \code{rperks}: numeric vector of random variates
#'   \item \code{hperks}: numeric vector of hazard values
#' }
#'
#' @references  
#' Richards, S.J. (2008). 
#' Applying survival models to pensioner mortality data. 
#' \emph{Bra. Actuarial Journal}, \bold{14}, 257--303.
#'
#' Chaudhary, A.K., & Kumar, V. (2013). 
#' A Bayesian Analysis of Perks Distribution via Markov Chain Monte Carlo 
#' Simulation. \emph{Nepal Journal of Science and Technology}, \bold{14(1)},
#' 153--166.   \doi{10.3126/njst.v14i1.8936}
#'
#' Richards, S. J. (2012). 
#' A handbook of parametric survival models for actuarial use. 
#' \emph{Scandinavian Actuarial Journal}, 1--25.     
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.1)
#' dperks(x, 5.0, 1.5)
#' pperks(x, 5.0, 1.5)
#' qperks(0.5, 5.0, 1.5)
#' rperks(10, 5.0, 1.5)
#' hperks(x, 5.0, 1.5)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=4.5967e-4, beta=1.1077)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pperks, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qperks, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dperks, pfun=pperks, plot=TRUE)
#' print.gofic(out)
#' 
#' @name Perks
#' @aliases dperks pperks qperks rperks hperks
#' @export
NULL

#' @rdname Perks
#' @export   
dperks <- function (x, alpha, beta, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta) <= 0)) {
		stop("Parameters 'alpha' and 'beta' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- exp(beta * x[valid])
    pdf[valid] <- alpha * beta * u * (1.0 + alpha)/((1.0 + alpha * u) ^ 2)
    }
	if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname Perks
#' @export 
pperks <- function(q, alpha, beta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta) <= 0)) {
		stop("Parameters 'alpha' and 'beta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(beta * q[valid])
    cdf[valid] <- 1.0 - ((1.0 + alpha)/(1.0 + alpha * u))
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname Perks
#' @export 
qperks <- function(p, alpha, beta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta) <= 0))
		stop("Invalid parameters: alpha, and beta must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <- (1.0/beta)* log((1/alpha)* ((1+alpha)/(1-p[valid]) -1))    
    }
	return(qtl)   
}

#' @rdname Perks
#' @export 
rperks <- function(n, alpha, beta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta), is.numeric)))
		stop("'alpha', and 'beta' must be numeric.")
	if (any(c(alpha, beta) <= 0))
		stop("Invalid parameters: alpha, and beta must be positive.")

    u <- runif(n)
    q <- qperks(u, alpha, beta)
    return(q)
}

#' @rdname Perks
#' @export 
hperks <- function (x, alpha, beta)
{
    if (!all(sapply(list(x, alpha, beta), is.numeric)))
		stop("All arguments must be numeric.")
    if (any(c(alpha, beta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <-  dperks(x, alpha, beta, log = FALSE)
	surv <- 1.0 - pperks(x, alpha, beta, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf) 
}
