#' Probability_Best_Model_WAIC calculates the probability of models compared with loo being the best
#' @title Return the probability of models compared with loo being the best
#' @author Marc Girondot \email{marc.girondot@@gmail.com}
#' @return A data.frame with the result of loo:compare()
#' @param loos A named vector object
#' @param column.mean Name of the column of loos with means to compare.
#' @param column.se Name of the column of loos with se to compare.
#' @param order If TRUE, higher is better, if FALSE lower is better.
#' @param nreplicates Number of replicates.
#' @param print If TRUE, print the results
#' @description Calculate the probability of models compared with loo being the best.
#' @family AIC
#' @examples
#' \dontrun{
#' Probability_Best_Model_WAIC(llok)
#' Probability_Best_Model_WAIC(llok, column.mean="elpd_loo", column.se="se_elpd_loo")
#' Probability_Best_Model_WAIC(llok, column.mean="p_loo", column.se="se_p_loo")
#' Probability_Best_Model_WAIC(llok, column.mean="looic", column.se="se_looic", order=FALSE)
#' }
#' @export

Probability_Best_Model_WAIC <- function(loos=stop("Results from loo::loo_compare() must be provided"), 
                                        column.mean="elpd_diff", column.se="se_diff", order=TRUE, # TRUE higher is better
                                        nreplicates=10000, print=TRUE) {
  # loos <- loos[, c(mean, se)]
  loos <- cbind(loos, Probability_Best=NA)
  
  rd <- matrix(data = NA, ncol=nreplicates, nrow = nrow(loos))
  for (i in 1:nrow(loos)) {
    rd[i, ] <- rnorm(nreplicates, mean=loos[i, column.mean], sd=loos[i, column.se])
  }
  best <- matrix(data = FALSE, ncol=nreplicates, nrow = nrow(loos))
  for (i in 1:nrow(loos)) {
    if (order) {
      for (j in 1:nreplicates) best[i, j] <- all(rd[i, j] > rd[-i, j])
    } else {
      for (j in 1:nreplicates) best[i, j] <- all(rd[i, j] < rd[-i, j])
    }
  }
  
  for (i in 1:nrow(loos)) loos[i, "Probability_Best"] <- sum(best[i, ])/nreplicates
  
  loos <- loos[order(loos[, "Probability_Best"], decreasing = TRUE), ]
  
  if (print) {
    message("Model ordered by better first")
    print(loos)
  }
  return(invisible(loos))
}


