#' @title estimate_parameters_on
#'
#' @description Online maximum likelihood estimation of the statistical measures of the output-differenced version of the single-output cumulative structural equation model with an additive-noise output measurement equation and assumptions of normality and independence.
#'
#' @param dat An (n + 1) x (m + 1) data frame of finite numeric elements (possibly except for row 1, columns 1 to m) containing observed input (columns 1 to m) and output (column m + 1) data of the original model.
#' @param s Initial value of parameter s (a vector of length 1 containing a finite numeric element that belongs to the interval [0, 1]).
#'
#' @return A list containing n sublists, each representing a progressively larger data sample (with 1 to n observation points), where each sublist consists of three elements: 1) estimate of the covariance at lag 0 of the data that result from the output-differenced model (an (m + 1) x (m + 1) matrix of numeric elements); 2) estimate of the only non-zero element of the negative covariance at lag 1 of the data that result from the output-differenced model (a vector of length 1 containing a numeric element); 3) estimate of the mean of the data that result from the output-differenced model (an (m + 1) x 1 matrix of numeric elements).
#'
#' @examples
#' set.seed(1)
#'
#' m <- 4
#' k <- 2
#'
#' L <- matrix(runif((m + 1) * k, min = -10, max = 10), nrow = m + 1)
#' sigma <- matrix(runif(m + 2, min = 0, max = 10), nrow = m + 2)
#' mu <- matrix(runif(m + 1, min = -10, max = 10), nrow = m + 1)
#'
#' data <- generate_data(100, L, sigma, mu)
#'
#' estimate_parameters_on(data, 0.35)
#' 
#' @references
#' Vytautas Dulskis & Leonidas Sakalauskas (2025). Toward Efficient Online Estimation of Dynamic Structural Equation Models: A Case Study. Journal of Statistical Computation and Simulation, 1-24. DOI: 10.1080/00949655.2025.2515955
#'
#' @export

estimate_parameters_on <- function(dat, s) {
  if (is.data.frame(dat) && is.vector(s)) {
    m <- ncol(dat) - 1
    if (nrow(dat) >= 2 && m >= 1 && length(s) == 1) {
      if (all(sapply(dat, is.numeric)) && is.numeric(s)) {
        if (all(sapply(dat[-1, 1:m], is.finite)) &&
            all(is.finite(dat[, m + 1])) && is.finite(s)) {
          if (s >= 0 && s <= 1) {
            xdy <- cbind(as.matrix(dat)[-1, 1:m], diff(as.matrix(dat[, m + 1])))
            n <- nrow(xdy)
            
            result <- list()
            
            c1 <- 1
            c2 <- xdy[1, 1:m]
            c4 <- xdy[1, m + 1]
            a1 <- c1
            a2 <- c2
            a3 <- c2 %*% t(c2)
            a4 <- c4
            a5 <- c4 ^ 2
            a6 <- c2 * c4
            d1 <- c1
            d2 <- c2
            d4 <- c4
            b1 <- d1 * (2 * s) / (s ^ 2 + 1)
            b2 <- d2 * (2 * s) / (s ^ 2 + 1)
            b3 <- d2 %*% t(d2) * (2 * s) / (s ^ 2 + 1)
            b4 <- d4 * (2 * s) / (s ^ 2 + 1)
            b5 <- d4 ^ 2 * (2 * s) / (s ^ 2 + 1)
            b6 <- d2 * d4 * (2 * s) / (s ^ 2 + 1)
            
            mu_x <- xdy[1, 1:m]
            u <- xdy[1, 1:m] %*% t(xdy[1, 1:m])
            Sigma_x <- u - mu_x %*% t(mu_x)
            tilde_Sigma_xdy <- rep(0, m)
            mu_dy <-
              (a4 - t(tilde_Sigma_xdy) %*% (a2 - mu_x * a1)) / a1
            f_a <-
              a1 * mu_dy ^ 2 - 2 * a4 * mu_dy + a5 + 2 * (mu_dy %*% t(a2) - t(a6) + (a4 - mu_dy * a1) %*% t(mu_x)) %*% tilde_Sigma_xdy + t(tilde_Sigma_xdy) %*% (a3 + (mu_x * a1 - 2 * a2) %*% t(mu_x)) %*% tilde_Sigma_xdy
            sigma_y_squared <- s * f_a
            Sigma_xdy <- Sigma_x %*% tilde_Sigma_xdy
            Sigma_dy <-
              f_a * (s ^ 2 + 1) + t(Sigma_xdy) %*% tilde_Sigma_xdy
            
            Sigma <-
              rbind(cbind(Sigma_x, Sigma_xdy), cbind(t(Sigma_xdy), Sigma_dy))
            colnames(Sigma)[m + 1] <- "dy"
            rownames(Sigma) <- colnames(Sigma)
            sigma_y_squared <-
              c("sigma_y_squared" = as.numeric(sigma_y_squared))
            mu <- as.matrix(append(mu_x, mu_dy))
            rownames(mu)[m + 1] <- "dy"
            colnames(mu) <- "mu"
            result[[1]] <- list(Sigma, sigma_y_squared, mu)
            names(result[[1]]) <-
              c(
                "Estimated covariance matrix at lag 0",
                "Estimated non-zero element of the negative covariance matrix at lag 1",
                "Estimated mean vector"
              )
            
            for (i in 2:n) {
              f_b <-
                b1 * mu_dy ^ 2 - 2 * b4 * mu_dy + b5 + 2 * (mu_dy %*% t(b2) - t(b6) + (b4 - mu_dy * b1) %*% t(mu_x)) %*% tilde_Sigma_xdy + t(tilde_Sigma_xdy) %*% (b3 + (mu_x * b1 - 2 * b2) %*% t(mu_x)) %*% tilde_Sigma_xdy
              if (f_a == 0) {
                s <- s
              } else {
                s <- (1 - 1 / i) * f_b / f_a
                s <- min(1, max(0, s))
              }
              
              c1 <- s * c1 + 1
              c2 <- s * c2 + xdy[i, 1:m]
              c4 <- s * c4 + xdy[i, m + 1]
              a1 <- (1 - 1 / i) * a1 + (c1 ^ 2) / i
              a2 <- (1 - 1 / i) * a2 + (c2 * c1) / i
              a3 <- (1 - 1 / i) * a3 + (c2 %*% t(c2)) / i
              a4 <- (1 - 1 / i) * a4 + (c4 * c1) / i
              a5 <- (1 - 1 / i) * a5 + (c4 ^ 2) / i
              a6 <- (1 - 1 / i) * a6 + (c2 * c4) / i
              b1 <-
                (1 - 1 / i) * b1 + (s * c1 ^ 2 - (1 - s ^ 2) * d1 * c1) / i
              b2 <-
                (1 - 1 / i) * b2 + (s * c1 * c2 - (1 - s ^ 2) * (d1 * c2 + d2 * c1) / 2) / i
              b3 <-
                (1 - 1 / i) * b3 + (s * c2 %*% t(c2) - (1 - s ^ 2) * (d2 %*% t(c2) + c2 %*% t(d2)) / 2) / i
              b4 <-
                (1 - 1 / i) * b4 + (s * c4 * c1 - (1 - s ^ 2) * (d4 * c1 + d1 * c4) / 2) / i
              b5 <-
                (1 - 1 / i) * b5 + (s * c4 ^ 2 - (1 - s ^ 2) * d4 * c4) / i
              b6 <-
                (1 - 1 / i) * b6 + (s * c2 * c4 - (1 - s ^ 2) * (d2 * c4 + d4 * c2) / 2) / i
              d1 <- s * d1 + c1
              d2 <- s * d2 + c2
              d4 <- s * d4 + c4
              
              mu_x <- (1 - 1 / i) * mu_x + xdy[i, 1:m] / i
              u <-
                (1 - 1 / i) * u + (xdy[i, 1:m] %*% t(xdy[i, 1:m])) / i
              Sigma_x <- u - mu_x %*% t(mu_x)
              tmp1 <-
                a3 - a2 %*% t(mu_x) - mu_x %*% t(a2) + mu_x %*% t(mu_x) * a1
              tmp2 <- a2 - mu_x * a1
              try(tilde_Sigma_xdy <-
                    solve(tmp1 - tmp2 %*% t(tmp2) / a1) %*% (a6 - mu_x * a4 - a4 * tmp2 / a1),
                  silent = TRUE)
              mu_dy <- (a4 - t(tilde_Sigma_xdy) %*% tmp2) / a1
              f_a <-
                a1 * mu_dy ^ 2 - 2 * a4 * mu_dy + a5 + 2 * (mu_dy %*% t(a2) - t(a6) + (a4 - mu_dy * a1) %*% t(mu_x)) %*% tilde_Sigma_xdy + t(tilde_Sigma_xdy) %*% (a3 + (mu_x * a1 - 2 * a2) %*% t(mu_x)) %*% tilde_Sigma_xdy
              sigma_y_squared <- s * f_a
              Sigma_xdy <- Sigma_x %*% tilde_Sigma_xdy
              Sigma_dy <-
                f_a * (s ^ 2 + 1) + t(Sigma_xdy) %*% tilde_Sigma_xdy
              
              Sigma <-
                rbind(cbind(Sigma_x, Sigma_xdy), cbind(t(Sigma_xdy), Sigma_dy))
              colnames(Sigma)[m + 1] <- "dy"
              rownames(Sigma) <- colnames(Sigma)
              sigma_y_squared <-
                c("sigma_y_squared" = as.numeric(sigma_y_squared))
              mu <- as.matrix(append(mu_x, mu_dy))
              rownames(mu)[m + 1] <- "dy"
              colnames(mu) <- "mu"
              result[[i]] <- list(Sigma, sigma_y_squared, mu)
              names(result[[i]]) <-
                c(
                  "Estimated covariance matrix at lag 0",
                  "Estimated non-zero element of the negative covariance matrix at lag 1",
                  "Estimated mean vector"
                )
            }
            
            return(result)
          }
        }
      }
    }
  }
  warning("The input is not valid. Consider checking it against the requirements of the function.")
  return(NULL)
}