% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_cp.R
\name{detect_cp}
\alias{detect_cp}
\title{Detect change points on time series.}
\usage{
detect_cp(
  data,
  n_iterations,
  n_burnin = 0,
  q = 0.5,
  params = list(),
  kernel,
  print_progress = TRUE,
  user_seed = 1234
)
}
\arguments{
\item{data}{if kernel = "ts" a vector or a matrix. If kernel = "epi" a matrix.}

\item{n_iterations}{number of MCMC iterations.}

\item{n_burnin}{number of iterations that must be excluded when computing the posterior estimate.}

\item{q}{probability of performing a split at each iteration.}

\item{params}{a list of parameters:

If data is an univariate time series the following must be specified:

\itemize{
\item \code{a}, \code{b}, \code{c} parameters of the Normal-Gamma prior for \eqn{\mu} and \eqn{\lambda}.
\item \code{prior_var_phi} variance for the proposal in the \eqn{N(0,\sigma^2_\phi)} posterior estimate of \eqn{\delta}.
\item \code{prior_delta_c} parameter of the shifted Gamma prior of \eqn{\delta}.
\item \code{prior_delta_d} parameter of the shifted Gamma prior of \eqn{\delta}.
}

If the time series is multivariate the following must be specified:

\itemize{
\item \code{m_0}, \code{k_0}, \code{nu_0}, \code{S_0} parameters for the Normal-Inverse-Wishart prior for \eqn{(\mu,\lambda)}.
\item \code{prior_var_phi} variance for the proposal in the \eqn{N(0,\sigma^2_\phi)} posterior estimate of \eqn{\delta}.
\item \code{prior_delta_c} parameter of the shifted Gamma prior of \eqn{\delta}.
\item \code{prior_delta_d} parameter of the shifted Gamma prior of \eqn{\delta}.
}

If data are epidemic diffusions:

\itemize{
\item \code{M} number of Monte Carlo iterations when computing the likelihood of the epidemic diffusion.
\item \code{xi} recovery rate fixed constant for each population at each time.
\item \code{a0},\code{b0} parameters for the computation of the integrated likelihood of the epidemic diffusions.
\item \code{I0_var} variance for the Metropolis-Hastings estimation of the proportion of infected at time 0.
}}

\item{kernel}{can be "ts" if data are time series or "epi" if data are epidemic diffusions.}

\item{print_progress}{If TRUE (default) print the progress bar.}

\item{user_seed}{seed for random distribution generation.}
}
\value{
A \code{DetectCpObj} object containing:

\itemize{
\item \code{$data} Vector or matrix with the input data.
\item \code{$n_iterations} Total number of MCMC iterations.
\item \code{$n_burnin} Number of burn-in iterations removed from posterior summaries.
\item \code{$orders} Matrix of change–point allocations. Each row corresponds to an iteration; columns correspond to time indices.
\item \code{$time} Computational time (in seconds).

\item \code{$entropy_MCMC} A \code{coda::mcmc} object containing the MCMC samples of the entropy measure.
\item \code{$lkl_MCMC} A \code{coda::mcmc} object containing the MCMC samples of the log-likelihood.
\item \code{$phi_MCMC} A \code{coda::mcmc} object storing MCMC draws of \eqn{\gamma}.
\item \code{$sigma_MCMC} A \code{coda::mcmc} object storing MCMC draws of \eqn{\sigma}.
\item \code{$delta_MCMC} A \code{coda::mcmc} object storing MCMC draws of \eqn{\delta}.
\item \code{$I0_MCMC} A \code{coda::mcmc} object storing MCMC draws of \eqn{I_0}.

\item \code{$kernel_ts} Logical; TRUE if the data are time series.
\item \code{$kernel_epi} Logical; TRUE if the data follow an epidemic diffusion process.
\item \code{$univariate_ts} Logical; TRUE if the time series is univariate, FALSE for multivariate.
}
}
\description{
The \code{detect_cp} function detect change points on univariate and multivariate time series.
}
\examples{

## Univariate time series

data("eu_inflation")

params_uni <- list(a = 1, b = 1, c = 1, prior_var_phi = 0.1,
                   prior_delta_c = 1, prior_delta_d = 1)

out <- detect_cp(data = eu_inflation[1,], n_iterations = 2000,
                 n_burnin = 250, q = 0.5, params = params_uni,
                 kernel = "ts")

print(out)

## Multivariate time series

data("eu_inflation")

params_multi <- list(m_0 = rep(0,3),
                     k_0 = 1,
                     nu_0 = 10,
                     S_0 = diag(0.1,3,3),
                     prior_var_phi = 0.1,
                     prior_delta_c = 1,
                     prior_delta_d = 1)

out <- detect_cp(data = eu_inflation[1:3,], n_iterations = 2000,
                 n_burnin = 250, q = 0.5, params = params_multi, kernel = "ts")

print(out)

\donttest{
## Epidemic diffusions

data("epi_synthetic")

params_epi <- list(M = 100, xi = 1/8, a0 = 4, b0 = 10, I0_var = 0.1)

out <- detect_cp(data = epi_synthetic, n_iterations = 2000, n_burnin = 250,
                 q = 0.25, params = params_epi, kernel = "epi")

print(out)
}

}
\references{
Martínez, A. F., & Mena, R. H. (2014). On a Nonparametric Change Point Detection Model in Markovian Regimes. \emph{Bayesian Analysis}, 9(4), 823–858. \doi{10.1214/14-BA878}

Corradin, R., Danese, L., & Ongaro, A. (2022). Bayesian nonparametric change point detection for multivariate time series with missing observations. \emph{International Journal of Approximate Reasoning}, 143, 26--43. \doi{10.1016/j.ijar.2021.12.019}
}
