#' Set YAML for Scheduled Emails in RStudio Connect
#'
#' RStudio Connect allows you to schedule emails to send using R Markdown. It
#' uses a special type of YAML using the top-level field `rmd_output_metadata`
#' that tells RStudio Connect about the email output. Several `rsc_*` fields
#' exist to specify different components of the email, which can be set in the
#' YAML header or programmatically using `rmarkdown::output_metadata()`. See the
#' [RStudio Connect
#' documentation](https://docs.rstudio.com/connect/1.7.2/user/r-markdown.html)
#' for more. `yml_output_metadata()` allows you to add any type of content to
#' the `rmd_output_metadata` field.
#'
#' @template describe_yml_param
#' @param rsc_email_subject The subject of the email. A report without an
#'   `rsc_email_subject` entry uses its published document name.
#' @param rsc_email_body_html,rsc_email_body_text The body of the email, either
#'   in plain text or HTML. A report with neither entry uses an automatically
#'   generated, plain-text body with a link to the report’s URL.
#' @param rsc_email_images Images to embed in the email. The embedded image must
#'   have a Content ID that is used in the body of the HTML and when providing
#'   the image to `rsc_email_images`, and the image itself must be
#'   base64-encoded, e.g. with the base64enc package.
#' @param rsc_output_files A vector of file names that should be available after
#'   the report has rendered. If you list a file that does not exist after
#'   rendering your report, Connect will log a message but continue trying to
#'   processing the other files listed. If the output files are not generated
#'   during the rendering of your report, then you will also need to list them
#'   in `resource_files` when you upload your report to Connect.
#' @param rsc_email_attachments A vector of file names that should be attached
#'   to the email.
#' @param rsc_email_suppress_scheduled Logical. Should the email schedule be
#'   suppressed? Default is `FALSE`.
#' @param rsc_email_suppress_report_attachment Logical. Should the rendered
#'   document be included as an attachment? Default is `TRUE`.
#' @param resource_files A file or files to host on RStudio Connect that is
#'   *not* generated by your report, e.g. an existing file.
#' @template describe_dots_param
#'
#' @template describe_yml_output
#' @export
#'
#' @examples
#'
#' yml() %>%
#'   yml_rsconnect_email(
#'     rsc_email_subject = "Quarterly report",
#'     rsc_output_files = "data.csv",
#'     rsc_email_attachments = c("attachment_1.csv", "attachment_2.csv")
#'   )
yml_rsconnect_email <- function(
  .yml,
  rsc_email_subject = yml_blank(),
  rsc_email_body_html = yml_blank(),
  rsc_email_body_text = yml_blank(),
  rsc_email_images = yml_blank(),
  rsc_output_files = yml_blank(),
  rsc_email_attachments = yml_blank(),
  rsc_email_suppress_scheduled = yml_blank(),
  rsc_email_suppress_report_attachment = yml_blank(),
  resource_files = yml_blank(),
  ...
) {
  rsconnect_email_opts <- list(
    rmd_output_metadata = list(
      rsc_email_subject = rsc_email_subject,
      rsc_email_body_html = rsc_email_body_html,
      rsc_email_body_text = rsc_email_body_text,
      rsc_email_images = rsc_email_images,
      rsc_output_files = rsc_output_files,
      rsc_email_attachments = rsc_email_attachments,
      rsc_email_suppress_scheduled = rsc_email_suppress_scheduled,
      rsc_email_suppress_report_attachment = rsc_email_suppress_report_attachment,
      ...
    ) %>% purrr::discard(is_yml_blank),
    resource_files = resource_files
  ) %>% purrr::discard(is_yml_blank)

  warn_if_duplicate_fields(.yml, rsconnect_email_opts)
  .yml[names(rsconnect_email_opts)] <- rsconnect_email_opts

  .yml
}

#' @export
#' @rdname yml_rsconnect_email
yml_output_metadata <- function(
  .yml,
  ...
) {
  rmd_output_metadata_list <- list(
    rmd_output_metadata = list(
      ...
    ) %>% purrr::discard(is_yml_blank)
  )

  warn_if_duplicate_fields(.yml, rmd_output_metadata_list)
  .yml[names(rmd_output_metadata_list)] <- rmd_output_metadata_list

  .yml
}
