#' Subset selection data frames based on manually filtered image files
#' 
#' \code{filtersels} subsets selection data frames based on image files that have been manually filtered.
#' @usage filtersels(X, path = NULL, lspec = FALSE, img.suffix = NULL, it = "jpeg",
#'  incl.wav = TRUE, missing = FALSE, index = FALSE)
#' @param X 'selection.table' object or data frame with the following columns: 1) "sound.files": name of the .wav 
#' files, 2) "sel": number of the selections. The ouptut of \code{\link{manualoc}} or \code{\link{autodetec}} can 
#' be used as the input data frame.
#' @param path Character string containing the directory path where the sound files are located. 
#' If \code{NULL} (default) then the current working directory is used.
#' @param lspec A logical argument indicating if the image files to be use for filtering were produced by the function \code{\link{lspec}}. 
#' All the image files that correspond to a sound file must be deleted in order to be 
#' filtered out.
#' @param img.suffix A character vector of length 1 with the suffix (label) at the end 
#' of the names of the image files. Default is \code{NULL} (i.e. no suffix as in the images
#' produced by \code{\link{specreator}}). Ignored if \code{lspec = TRUE}.
#' @param it A character vector of length 1 giving the image type ("tiff", "jpeg" or "pdf") Default is "jpeg". Note that pdf files can only be generated by \code{\link{lspec2pdf}}.
#' @param incl.wav Logical. To indicate if sound files extensions (".wav") are included ( \code{TRUE}, default) or not in the image
#' file names.
#' @param missing Logical. Controls whether the output data frame (or row index if is \code{index = TRUE}) 
#' contains the selections with images in the working directory (Default, \code{missing = FALSE}) 
#' or the ones with no image.
#' @param index Logical. If \code{TRUE} and \code{missing = FALSE} the row index for the selections with images in the working directory is returned. If \code{missing = TRUE}) then the row index of the ones with no image is returned instead. Default is \code{FALSE}. 
#' @return If all .wav files are ok, returns message "All files are ok!".
#'   Otherwise returns "These file(s) cannot be read" message with names of the
#'   corrupted .wav files.
#' @details This function subsets selections (or sound files if \code{lspec} is \code{TRUE}) listed in a data frame
#'  based on the image files from spectrogram-creating functions (e.g. \code{\link{specreator}}) in the 
#'  working directory. Only the selections/sound files with and image in the working directory will remain. 
#'  This is useful for excluding selections from undesired signals. Note that the
#'  image files should be in the working directory (or the directory provided in 'path').
#' @export
#' @name filtersels
#' @examples{ 
#' # First set temporary folder
#' setwd(tempdir())
#' 
#' # save wav file examples
#' data(list = c("Phae.long1", "Phae.long2", "Phae.long3", "selec.table"))
#' writeWave(Phae.long1,"Phae.long1.wav")
#' writeWave(Phae.long2,"Phae.long2.wav")
#' writeWave(Phae.long3,"Phae.long3.wav")
#' 
#' specreator(selec.table, flim = c(0, 11), inner.mar = c(4,4.5,2,1), outer.mar = c(4,2,2,1), 
#' picsize = 2, res = 300, cexlab = 2, mar = 0.05, wl = 300)
#' 
#' #go to the working directory and delete some images
#' 
#' #filter selection data frame
#' fmloc <- filtersels(X = selec.table)
#' 
#' #this data frame does not have the selections corresponding to the images that were deleted
#' fmloc
#' 
#' #now using lspec images
#' lspec(sxrow = 2, rows = 8, pal = reverse.heat.colors, wl = 300, ovlp = 10)
#' 
#' #go to the working directory and delete lspec images (the ones with several rows of spectrograms)
#' 
#' #filter selection data frame
# fmloc2 <- filtersels(X = selec.table, lspec = TRUE)
#' 
#' }
#' @author Marcelo Araya-Salas (\email{araya-salas@@cornell.edu})
#last modification on feb-6-2017 (MAS)

filtersels <- function(X, path = NULL, lspec = FALSE, img.suffix = NULL, it = "jpeg", 
                       incl.wav = TRUE, missing = FALSE, index = FALSE)
  {

  # reset working directory 
  wd <- getwd()
  on.exit(setwd(wd))
  
  #check path to working directory
  if(is.null(path)) path <- getwd() else {if(!file.exists(path)) stop("'path' provided does not exist") else
    setwd(path)
  }  

    #if X is not a data frame
    if(!class(X) %in% c("data.frame", "selection.table")) stop("X is not of a class 'data.frame' or 'selection table")
    
    


  #if it argument is not "jpeg" or "tiff" 
  if(!any(it == "jpeg", it == "tiff", it == "pdf")) stop(paste("Image type", it, "not allowed"))  
  
if(!all(c("sound.files", "selec") %in% colnames(X))) 
  stop(paste(paste(c("sound.files", "selec")[!(c("sound.files", "selec") %in% colnames(X))], collapse=", "), "column(s) not found in data frame"))

if(it != "pdf")
{
  #check if files are in working directory
  imgs <- list.files(pattern = "\\.jpeg$|\\.tiff$", ignore.case = FALSE)
  if (length(imgs) == 0) 
    stop("No image files in working directory")
  
  # if not long spec
  if(!lspec){
  
    #if img suffix not provided
  if(is.null(img.suffix)){
    # if .wav extension is included
    if(incl.wav)  imgn <- paste(paste(X$sound.files, X$selec, sep =  "-"), it, sep = ".") else 
      imgn <- paste(paste(gsub(".wav$", "", X$sound.files, ignore.case = TRUE), X$selec, sep =  "-"), it, sep = ".") 
    } else
  {      if(incl.wav)  imgn <- paste(paste(X$sound.files, X$selec, img.suffix, sep =  "-"), it, sep = ".") else
    imgn <- paste(paste(gsub(".wav$", "", X$sound.files, ignore.case = TRUE), X$selec, img.suffix, sep =  "-"), it, sep = ".")

  }
          #subset data frame X
  miss.index <- imgn %in% imgs
    
    if(!index)
    {
      if(missing) Y <- X[!miss.index, ] else Y <- X[miss.index, ]
      
    } else if(missing)  Y <- which(!miss.index) else  Y <- which(miss.index)
    
    
    
     
  } else {

  #   #remove the ones with no pX.it ending
    imgs <- grep("p\\d+\\.jpeg|p\\d+\\.tiff" ,imgs, value = TRUE)
  if(length(imgs) == 0) stop("Images have not been produced by 'lspec' function")
  
  #subset selection table
    miss.index <- gsub(".wav$", "", X$sound.files, ignore.case = TRUE) %in% gsub("-p\\d+\\.jpeg$|-p\\d+\\.tiff$" , "", imgs)
    
    if(!index)
    {
      if(missing) Y <- X[!miss.index, ] else Y <- X[miss.index, ]
    
    } else if(missing)  Y <- which(!miss.index) else Y <- which(miss.index)

  }
  
} else {
  #check if pdf files are in working directory
  imgs <- list.files(pattern = ".pdf$", ignore.case = FALSE)
  if (length(imgs) == 0) 
    stop("No pdf files in working directory")
  
  #subset selection table
  miss.index <- gsub(".wav$", "", X$sound.files, ignore.case = TRUE) %in% gsub(".pdf$" , "", imgs)
  
  if(!index)
  {
    if(missing) Y <- droplevels(X[miss.index, ]) else Y <- droplevels(X[!miss.index, ])
    
  } else if(missing)  Y <- which(!miss.index) else  Y <- which(miss.index)
  
}
if(!index)
{if(nrow(Y) == 0) stop("Image files in working directory do not match sound file names in X (wrong working directory?)")
  return(Y)
  } else 
  {
    if(length(Y) == 0) message("Index vector is of length 0")
 return(Y)
    }

}
