#' @title Create an igraph graph from network
#'
#' @param net A named list of dataframes \code{nodes} and \code{edges} generated by \code{Create}.
#' @param directed Logical. Create a directed graph. Default is \code{TRUE}.
#' @param writeToFile Logical. Save graph to a file in the current working directory. Default is \code{FALSE}.
#' @param verbose Logical. Output additional information. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#'
#' @return An igraph object.
#'
#' @aliases Graph
#' @name Graph
#' @export
Graph <- function(net,
                  directed = TRUE,
                  writeToFile = FALSE,
                  verbose = FALSE,
                  ...) {

  msg <- f_verbose(verbose)

  prompt_and_stop("igraph", "Graph")

  msg("Creating igraph network graph...\n")

  # igraph warning converting a dataframe with an POSIXct column to an igraph object
  # https://github.com/igraph/rigraph/pull/250
  # https://github.com/igraph/rigraph/issues/251
  if (is.data.frame(net$nodes) && is.data.frame(net$edges)) {
    net$nodes <- net$nodes |> dplyr::mutate_if(lubridate::is.POSIXt, as.character)
    net$edges <- net$edges |> dplyr::mutate_if(lubridate::is.POSIXt, as.character)
  }

  if (any(c("twitter", "youtube", "reddit", "web") %in% class(net)) &&
      any(c("activity", "actor", "twomode") %in% class(net))) {
    g <- igraph::graph_from_data_frame(
      d = net$edges,
      directed = directed,
      vertices = net$nodes
    )
  }

  # searches the class list of net for matching method
  UseMethod("Graph", net)
}

#' @noRd
#' @export
Graph.default <- function(...) {
  stop("Unknown network type passed to graph.", call. = FALSE)
}

#' @noRd
#' @method Graph activity
#' @export
Graph.activity <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {
    UseMethod("Graph.activity", net)
  }

#' @noRd
#' @export
Graph.activity.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.activity.twitter <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "twitter")

    graphOutputFile(g, "graphml", writeToFile, "TwitterActivity", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.activity.youtube <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "youtube")

    graphOutputFile(g, "graphml", writeToFile, "YoutubeActivity", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.activity.reddit <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "reddit")

    graphOutputFile(g, "graphml", writeToFile, "RedditActivity", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.activity.web <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "web")

    graphOutputFile(g, "graphml", writeToFile, "WebActivity", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @method Graph actor
#' @export
Graph.actor <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {
    UseMethod("Graph.actor", net)
  }

#' @noRd
#' @export
Graph.actor.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.actor.twitter <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "twitter")

    graphOutputFile(g, "graphml", writeToFile, "TwitterActor", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.actor.youtube <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "youtube")

    graphOutputFile(g, "graphml", writeToFile, "YoutubeActor", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.actor.reddit <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "reddit")

    graphOutputFile(g, "graphml", writeToFile, "RedditActor", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @export
Graph.actor.web <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "web")

    graphOutputFile(g, "graphml", writeToFile, "WebActor", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @method Graph semantic
#' @export
Graph.semantic <-
  function(net,
           directed = FALSE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {
    UseMethod("Graph.semantic", net)
  }

#' @noRd
#' @export
Graph.semantic.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.semantic.twitter <-
  function(net,
           directed = FALSE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    # create igraph object from dataframes
    g <- igraph::graph_from_data_frame(
      d = net$edges,
      directed = directed,
      vertices = net$nodes
    )

    g <- igraph::set_graph_attr(g, "type", "twitter")

    graphOutputFile(g, "graphml", writeToFile, "TwitterSemantic", verbose = verbose)
    msg("Done.\n")

    g
  }

#' @noRd
#' @method Graph twomode
#' @export
Graph.twomode <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {
    UseMethod("Graph.twomode", net)
  }

#' @noRd
#' @export
Graph.twomode.default <- function(...) {
  stop("Unknown social media type passed to graph.", call. = FALSE)
}

#' @noRd
#' @export
Graph.twomode.twitter <-
  function(net,
           directed = TRUE,
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {

    g <- igraph::set_graph_attr(g, "type", "twitter")

    graphOutputFile(g, "graphml", writeToFile, "Twitter2mode", verbose = verbose)
    msg("Done.\n")

    g
  }

# set output file name
# if wtof is logical use def as file name
# if character use wtof as file name
graphOutputFile <- function(g, type, wtof, def, verbose = FALSE) {

  if (is.logical(wtof) && wtof) {
    write_output_file(g, "graphml", def, verbose = verbose)

  } else if (is.character(wtof)) {
    write_output_file(g, "graphml", wtof, verbose = verbose)
  }
}
