% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bind.R
\name{vec_bind}
\alias{vec_bind}
\alias{vec_rbind}
\alias{vec_cbind}
\title{Combine many data frames into one data frame}
\usage{
vec_rbind(
  ...,
  .ptype = NULL,
  .names_to = rlang::zap(),
  .name_repair = c("unique", "universal", "check_unique"),
  .name_spec = NULL
)

vec_cbind(
  ...,
  .ptype = NULL,
  .size = NULL,
  .name_repair = c("unique", "universal", "check_unique", "minimal")
)
}
\arguments{
\item{...}{Data frames or vectors.

When the inputs are named:
\itemize{
\item \code{vec_rbind()} assigns names to row names unless \code{.names_to} is
supplied. In that case the names are assigned in the column
defined by \code{.names_to}.
\item \code{vec_cbind()} creates packed data frame columns with named
inputs.
}

\code{NULL} inputs are silently ignored. Empty (e.g. zero row) inputs
will not appear in the output, but will affect the derived \code{.ptype}.}

\item{.ptype}{If \code{NULL}, the default, the output type is determined by
computing the common type across all elements of \code{...}.

Alternatively, you can supply \code{.ptype} to give the output known type.
If \code{getOption("vctrs.no_guessing")} is \code{TRUE} you must supply this value:
this is a convenient way to make production code demand fixed types.}

\item{.names_to}{This controls what to do with input names supplied in \code{...}.
\itemize{
\item By default, input names are \link[rlang:zap]{zapped}.
\item If a string, specifies a column where the input names will be
copied. These names are often useful to identify rows with
their original input. If a column name is supplied and \code{...} is
not named, an integer column is used instead.
\item If \code{NULL}, the input names are used as row names.
}}

\item{.name_repair}{One of \code{"unique"}, \code{"universal"}, or
\code{"check_unique"}. See \code{\link[=vec_as_names]{vec_as_names()}} for the meaning of these
options.

With \code{vec_rbind()}, the repair function is applied to all inputs
separately. This is because \code{vec_rbind()} needs to align their
columns before binding the rows, and thus needs all inputs to
have unique names. On the other hand, \code{vec_cbind()} applies the
repair function after all inputs have been concatenated together
in a final data frame. Hence \code{vec_cbind()} allows the more
permissive minimal names repair.}

\item{.name_spec}{A name specification (as documented in \code{\link[=vec_c]{vec_c()}})
for combining the outer inputs names in \code{...} and the inner row
names of the inputs. This only has an effect when \code{.names_to} is
set to \code{NULL}, which causes the input names to be assigned as row
names.}

\item{.size}{If, \code{NULL}, the default, will determine the number of
rows in \code{vec_cbind()} output by using the standard recycling rules.

Alternatively, specify the desired number of rows, and any inputs
of length 1 will be recycled appropriately.}
}
\value{
A data frame, or subclass of data frame.

If \code{...} is a mix of different data frame subclasses, \code{vec_ptype2()}
will be used to determine the output type. For \code{vec_rbind()}, this
will determine the type of the container and the type of each column;
for \code{vec_cbind()} it only determines the type of the output container.
If there are no non-\code{NULL} inputs, the result will be \code{data.frame()}.
}
\description{
This pair of functions binds together data frames (and vectors), either
row-wise or column-wise. Row-binding creates a data frame with common type
across all arguments. Column-binding creates a data frame with common length
across all arguments.
}
\section{Invariants}{


All inputs are first converted to a data frame. The conversion for
1d vectors depends on the direction of binding:
\itemize{
\item For \code{vec_rbind()}, each element of the vector becomes a column in
a single row.
\item For \code{vec_cbind()}, each element of the vector becomes a row in a
single column.
}

Once the inputs have all become data frames, the following
invariants are observed for row-binding:
\itemize{
\item \code{vec_size(vec_rbind(x, y)) == vec_size(x) + vec_size(y)}
\item \code{vec_ptype(vec_rbind(x, y)) = vec_ptype_common(x, y)}
}

Note that if an input is an empty vector, it is first converted to
a 1-row data frame with 0 columns. Despite being empty, its
effective size for the total number of rows is 1.

For column-binding, the following invariants apply:
\itemize{
\item \code{vec_size(vec_cbind(x, y)) == vec_size_common(x, y)}
\item \code{vec_ptype(vec_cbind(x, y)) == vec_cbind(vec_ptype(x), vec_ptype(x))}
}
}

\section{Dependencies}{

\subsection{vctrs dependencies}{
\itemize{
\item \code{\link[=vec_cast_common]{vec_cast_common()}}
\item \code{\link[=vec_proxy]{vec_proxy()}}
\item \code{\link[=vec_init]{vec_init()}}
\item \code{\link[=vec_assign]{vec_assign()}}
\item \code{\link[=vec_restore]{vec_restore()}}
}
}

\subsection{base dependencies of \code{vec_rbind()}}{
\itemize{
\item \code{\link[base:c]{base::c()}}
}

If columns to combine inherit from a common class,
\code{vec_rbind()} falls back to \code{base::c()} if there exists a \code{c()}
method implemented for this class hierarchy.
}
}

\examples{
# row binding -----------------------------------------

# common columns are coerced to common class
vec_rbind(
  data.frame(x = 1),
  data.frame(x = FALSE)
)

# unique columns are filled with NAs
vec_rbind(
  data.frame(x = 1),
  data.frame(y = "x")
)

# null inputs are ignored
vec_rbind(
  data.frame(x = 1),
  NULL,
  data.frame(x = 2)
)

# bare vectors are treated as rows
vec_rbind(
  c(x = 1, y = 2),
  c(x = 3)
)

# default names will be supplied if arguments are not named
vec_rbind(
  1:2,
  1:3,
  1:4
)

# column binding --------------------------------------

# each input is recycled to have common length
vec_cbind(
  data.frame(x = 1),
  data.frame(y = 1:3)
)

# bare vectors are treated as columns
vec_cbind(
  data.frame(x = 1),
  y = letters[1:3]
)

# if you supply a named data frame, it is packed in a single column
data <- vec_cbind(
  x = data.frame(a = 1, b = 2),
  y = 1
)
data

# Packed data frames are nested in a single column. This makes it
# possible to access it through a single name:
data$x

# since the base print method is suboptimal with packed data
# frames, it is recommended to use tibble to work with these:
if (rlang::is_installed("tibble")) {
  vec_cbind(x = tibble::tibble(a = 1, b = 2), y = 1)
}

# duplicate names are flagged
vec_cbind(x = 1, x = 2)

}
\seealso{
\code{\link[=vec_c]{vec_c()}} for combining 1d vectors.
}
