% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_taxes.R
\name{taxsim_calculate_taxes}
\alias{taxsim_calculate_taxes}
\title{Calculate state and federal taxes using TASXSIM 35.}
\usage{
taxsim_calculate_taxes(
  .data,
  marginal_tax_rates = "Wages",
  return_all_information = FALSE,
  interface = "wasm"
)
}
\arguments{
\item{.data}{Data frame containing the information that will be used to calculate taxes.
This data set will be sent to TAXSIM. Data frame must have specified column names and data types.}

\item{marginal_tax_rates}{Variable to use when calculating marginal tax rates. One of 'Wages', 'Long Term Capital Gains',
'Primary Wage Earner', or 'Secondary Wage Earner'. Default is 'Wages'.}

\item{return_all_information}{Boolean (TRUE or FALSE). Whether to return all information from TAXSIM (TRUE),
or only key information (FALSE). Returning all information returns 42 columns of output, while only
returning key information returns 9 columns. It is faster to download results with only key information.}

\item{interface}{String indicating which NBER TAXSIM interface to use. Should
be one of: "wasm", "ssh", or "http".
\itemize{
\item "wasm" uses a compiled WebAssembly version of the TAXSIM app. Details
about generating the wasm file can be found here:
https://github.com/tmm1/taxsim.js
\item "ssh" uses SSH to connect to taxsimssh.nber.org. Your system must already
have SSH installed.
\item "http" uses CURL to connect to
https://taxsim.nber.org/uptest/webfile.cgi. 'http" does not work for dataset with more than
1,000 rows.
}}
}
\value{
The output data set contains all the information returned by \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35},
using the same column names. Descriptions of these columns can be found at the bottom of the page
containing \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35's documentation}.
}
\description{
This function calculates state and federal income taxes using the TAXSIM 35 tax simulator.
See \url{http://taxsim.nber.org/taxsim35/} for more information on TAXSIM 35.
}
\section{Formatting your data}{


In the input data set, \code{.data}, each column is a tax characteristic (year, filing status, income, etc.)
and each row is a tax filing unit.

Columns should take the same names, and fulfill the same requirements, as those needed for \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35}.
Potential columns, with there names and descriptions, can be found at: \href{http://taxsim.nber.org/taxsim35/}{http://taxsim.nber.org/taxsim35/}.

The following columns are required: \code{taxsimid}, \code{year}, \code{mstat}, and \code{state}.

There are two points where \code{\link{taxsim_calculate_taxes}} departs from \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35}.
\enumerate{
\item For filing status, \code{mstat}, users can either enter the number allowed by \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35}
or one of the following descriptions:
}
\itemize{
\item "single"
\item "married, jointly"
\item "married, separately"
\item "dependent child"
\item "head of household"
}
\enumerate{
\item For \code{state}, users can either enter the SOI code, as required by \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35},
the two-letter state abbreviation, or the full name of the state.
}

It is OK if the input data set, \code{.data}, contains columns in addition to the ones that are used by \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35}.
}

\section{Giving credit where it is due}{


The NBER's \href{http://taxsim.nber.org/taxsim35/}{TAXSIM 35} tax simulator does all tax
calculations. This package simply lets users interact with the tax simulator through R. Therefore,
users should cite the TASXSIM 35 tax simulator when they use this package in their work:

Feenberg, Daniel Richard, and Elizabeth Coutts, An Introduction to the TAXSIM Model,
Journal of Policy Analysis and Management vol 12 no 1, Winter 1993, pages 189-194.
}

\examples{

\dontrun{
family_income <- data.frame(
    taxsimid = c(1, 2),
    state = c('North Carolina', 'NY'),
    year = c(2015, 2015),
    mstat = c('single', 'married, jointly'),
    pwages = c(10000, 100000),
    page = c(26, 36)
)


family_taxes <- taxsim_calculate_taxes(family_income)

merge(family_income, family_taxes, by = 'taxsimid')
}

}
