#' Time-Varying Coefficient Arrays of the Lagged Endogenous Variables of a tv-VAR (no intercept).
#'
#' Returns the estimated coefficients of the lagged endogenous variables as an array.
#'  Given an estimated time varying VAR of the form:
#' \deqn{\hat{\bold{y}}_t = \hat{A}_{1t} \bold{y}_{t-1} + \ldots + \hat{A}_{pt} \bold{y}_{t-p} + \hat{C}_tD_t}
#' the function returns a list for each equation with \eqn{\hat{A}_{1t} | \ldots | \hat{A}_{pt} | \hat{C}_t)}
#' set of arrays
#'
#' @rdname tvAcoef
#' @param x An object of class \code{tvvar} generated by \code{\link{tvVAR}}.
#' @return A list object with coefficient arrays for the lagged endogenous variables.
#' @export

tvAcoef <- function (x) UseMethod("tvAcoef", x)

#' @rdname tvAcoef
#' @inheritParams tvAcoef
#'
#' @examples
#' data(Canada, package="vars")
#' var.2p <- vars::VAR(Canada, p = 2, type = "const")
#' tvvar.2p <- tvVAR(Canada, p = 2, type = "const")
#' A <- vars::Acoef(var.2p)
#' tvA <- tvAcoef(tvvar.2p)
#'
#' @method tvAcoef tvvar
#' @export
tvAcoef.tvvar <- function (x)
{
  if (!(class(x) == "tvvar"))
    stop("\nPlease provide an object of class 'tvvar', generated by 'tvVAR()'.\n")
  neq  <-  x$neq
  p  <-  x$p
  obs <- x$obs
  A  <-  tvBcoef(x)[,, 1:(neq * p)]
  As  <-  list()
  start  <-  seq(1, p * neq, neq)
  end  <-  seq(neq, p * neq, neq)
  for (i in 1:p)
  {
    As[[i]]  <-  as.array(A[,, start[i]:end[i]], dim=c(obs,neq,neq),
                        dimnames=list(NULL, dimnames(A)[[2]],
                                      dimnames(A)[[3]][start[i]:end[i]]))
  }
  return(As)
}
