% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visualization.R
\name{plot_cluster_mapping}
\alias{plot_cluster_mapping}
\title{Create Clustered Mapping Plots}
\usage{
plot_cluster_mapping(
  df_coords,
  ndim,
  dim_config = new_dim_reduction_config(),
  aesthetic_config = new_aesthetic_config(),
  layout_config = new_layout_config(),
  annotation_config = new_annotation_config(),
  output_dir,
  show_shape_legend = TRUE,
  cluster_legend_title = "Cluster",
  draw_arrows = FALSE,
  annotate_arrows = TRUE,
  phylo_tree = NULL,
  sigma_t = NULL,
  sigma_x = NULL,
  clade_node_depth = NULL,
  show_one_arrow_per_cluster = FALSE,
  cluster_legend_order = NULL
)
}
\arguments{
\item{df_coords}{Data frame containing:
- V1, V2, ... Vn: Coordinate columns
- antigen: Binary indicator for antigen points
- antiserum: Binary indicator for antiserum points
- cluster: Factor or integer cluster assignments}

\item{ndim}{Number of dimensions in input coordinates}

\item{dim_config}{Dimension reduction configuration object specifying method and parameters}

\item{aesthetic_config}{Aesthetic configuration object controlling plot appearance}

\item{layout_config}{Layout configuration object controlling plot dimensions and style.
Use x_limits and y_limits in layout_config to set axis limits.}

\item{annotation_config}{Annotation configuration object for labeling notable points}

\item{output_dir}{Character. Directory for output files. Required if \code{layout_config$save_plot} is \code{TRUE}.}

\item{show_shape_legend}{Logical. Whether to show the shape legend (default: TRUE)}

\item{cluster_legend_title}{Character. Custom title for the cluster legend (default: "Cluster")}

\item{draw_arrows}{logical; if TRUE, compute and draw antigenic drift vectors}

\item{annotate_arrows}{logical; if TRUE, show names of the points having arrows}

\item{phylo_tree}{Optional; phylo object in Newick format. Does not need to be rooted. If provided, used to compute antigenic velocity arrows.}

\item{sigma_t}{Optional; numeric; bandwidth for the Gaussian kernel discounting on time in years or the time unit of the data. If NULL, uses Silverman's rule of thumb.}

\item{sigma_x}{Optional; numeric; bandwidth for the Gaussian kernel discounting on antigenic distance in antigenic units. If NULL, uses Silverman's rule of thumb.}

\item{clade_node_depth}{Optional; integer; number of levels of parent nodes to define clades. Antigens from different clades will be excluded from the calculation antigenic velocity arrows. (Default: Automatically calculated mode of leaf-to-backbone distance of the tree)}

\item{show_one_arrow_per_cluster}{Shows only the largest antigenic velocity arrow in each cluster}

\item{cluster_legend_order}{in case you prefer a certain order for clusters in the legend,
provide a list with that order here; e.g., c("cluster 2", "cluster 1")}
}
\value{
A \code{ggplot} object containing the cluster mapping visualization.
}
\description{
Creates a visualization of points colored by cluster assignment using dimension
reduction, with optional antigenic velocity arrows. Points are colored by cluster with different shapes for antigens and
antisera.
}
\details{
The function performs these steps:
\enumerate{
\item Validates input data structure and types
\item Applies dimension reduction if ndim > 2
\item Creates visualization with cluster-based coloring
\item Applies specified aesthetic and layout configurations
\item Applies custom axis limits if specified in layout_config
}

Different shapes distinguish between antigens and antisera points, while
color represents cluster assignment. The color palette can be customized
through the aesthetic_config.
}
\examples{
# Basic usage with default configurations
data <- data.frame(
  V1 = rnorm(100), V2 = rnorm(100), V3 = rnorm(100), name = 1:100,
  antigen = rep(c(0,1), 50), antiserum = rep(c(1,0), 50),
  cluster = rep(1:5, each=20)
)
p1 <- plot_cluster_mapping(data, ndim=3)

# Save plot to a temporary directory
temp_dir <- tempdir()
# Custom configurations with specific color palette and axis limits
aesthetic_config <- new_aesthetic_config(
  point_size = 4,
  point_alpha = 0.7,
  color_palette = c("red", "blue", "green", "purple", "orange"),
  show_labels = TRUE,
  label_size = 3
)

layout_config_save <- new_layout_config(save_plot = TRUE,
  width = 10,
  height = 8,
  coord_type = "fixed",
  show_grid = TRUE,
  grid_type = "major",
  x_limits = c(-10, 10),
  y_limits = c(-8, 8)
)

p_saved <- plot_cluster_mapping(data, ndim=3, 
  layout_config = layout_config_save, 
  aesthetic_config = aesthetic_config,
  output_dir = temp_dir
)

list.files(temp_dir)
unlink(temp_dir, recursive = TRUE)

}
\seealso{
\code{\link{plot_temporal_mapping}} for temporal visualization
\code{\link{plot_3d_mapping}} for 3D visualization
\code{\link{new_dim_reduction_config}} for dimension reduction options
\code{\link{new_aesthetic_config}} for aesthetic options
\code{\link{new_layout_config}} for layout options
\code{\link{new_annotation_config}} for annotation options
}
