% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tipping_point_model_free.R
\name{tipping_point_model_free}
\alias{tipping_point_model_free}
\title{Tipping Point Analysis (Model-Free)}
\usage{
tipping_point_model_free(
  dat,
  reason,
  impute,
  J = 10,
  tipping_range = seq(5, 95, by = 5),
  cox_fit = NULL,
  verbose = FALSE,
  method = c("random sampling", "deterministic sampling"),
  seed = 12345
)
}
\arguments{
\item{dat}{data.frame containing at least 5 columns: TRT01P (treatment arm as factor), AVAL (survival time), EVENT (event indicator), CNSRRS (censoring reason) and MAXAVAL (maximum potential survival time, duration between randomization to data cut-off)}

\item{reason}{Vector specifying censoring reasons to be imputed.}

\item{impute}{a string specifying the treatment arm(s) which require imputation. It must be one of the arms from variable TRT01P, the first level of TRT01P is considered as the control arm.}

\item{J}{numeric indicating number of imputations.}

\item{tipping_range}{Numeric vector. Percentiles to use when \code{method = "random sampling"}. Number of patients to impute when \code{method = "deterministic sampling"}.}

\item{cox_fit}{A Cox model that will be used to calculate HRs on imputed datasets.
In case of inclusion of stratification factors or covariates, conditional HR will be used.}

\item{verbose}{Logical. If \code{TRUE}, prints progress and analysis details.}

\item{method}{Character. Either \code{"random sampling"} or \code{"deterministic sampling"}.}

\item{seed}{Integer. Random seed for reproducibility.}
}
\value{
A \code{tipse} object containing:
\describe{
\item{original data}{Input argument from 'data'.}
\item{imputation_results}{A data frame of combined pooled model results across tipping points}
\item{original_HR}{The original hazard ratio.}
\item{reason_to_impute}{Input argument from 'reason'.}
\item{arm_to_impute}{Input argument from 'impute'.}
\item{method_to_impute}{Input argument from 'method'.}
\item{imputation_data}{A list of imputed datasets for each tipping point value.}
}
}
\description{
Performs a model-free tipping point analysis on time-to-event data by
repeatedly imputing censored observations under varying assumptions. The model-free
framework assumes that censored patients share similar survival
behavior with those from whom they are sampled, without fitting any parametric
survival model.
}
\details{
The \strong{model-free tipping point analysis} provides a reproducible and intuitive
framework for exploring the robustness of treatment effects in time-to-event
(survival) endpoints when censoring may differ between study arms.

Two sampling modes are supported:
\itemize{
\item \code{method = "random sampling"} - performs re-sampling of event
times from the best or worst percentile of observed patients ranked by their event or censoring time.
The \code{tipping_range} specifies the percentiles of the observed data from which
event times will be sampled to impute censored patients.
For the treatment arm, use the worst percentiles (shortest survival times) from the
observed data of both arms. For the control arm, use the best percentiles (longest survival times).
\item \code{method = "deterministic sampling"} - imputes a fixed number of censored
patients deterministically. The \code{tipping_range} specifies the number of patients to be imputed.
For the treatment arm, it defines the number of patients that will be assumed to
have an event at their time of censoring. For the control arm, it defines the
number of patients that will be assumed to be event-free at data cut-off, their maximum potential follow-up time.
}

This function iteratively applies the random- or deterministic-sampling
imputation procedure across a range of
tipping point parameters \code{tipping_range}. For each parameter value:
\enumerate{
\item Multiple imputed datasets are generated (\code{J} replicates), where censored
observations in the selected arm are replaced by sampled or reassigned
event times according to the imputation method.
\item A Cox proportional hazards model is fitted to each imputed dataset.
\item Model estimates are pooled using \strong{Rubin’s rules} to obtain a combined
hazard ratio and confidence interval for that tipping point parameter.
}

The process yields a series of results showing how the treatment effect changes
as increasingly conservative or optimistic assumptions are made about censored
observations. The \emph{tipping point} is defined as the smallest value of the
sensitivity parameter (percentile or number of imputed patients) for which the upper
bound of the hazard ratio confidence interval crosses 1 - i.e., where the
apparent treatment benefit is lost.
}
\examples{
cox1 <- survival::coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200)
result <- tipping_point_model_free(
  dat = codebreak200,
  reason = "Early dropout",
  impute = "docetaxel",
  J = 10,
  tipping_range = seq(5, 95, by = 5),
  cox_fit = cox1,
  verbose = TRUE,
  method = "random sampling",
  seed = 12345
)

result2 <- tipping_point_model_free(
  dat = codebreak200,
  reason = "Early dropout",
  impute = "docetaxel",
  J = 10,
  tipping_range = seq(1, 21, by = 2),
  cox_fit = cox1,
  verbose = TRUE,
  method = "deterministic sampling",
  seed = 12345
)
}
