% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sampleUniNormMixture.R
\name{sampleUniNormMixture}
\alias{sampleUniNormMixture}
\title{Telescoping sampling of a Bayesian finite univariate Gaussian mixture where a prior
on the number of components K is specified.}
\usage{
sampleUniNormMixture(
  y,
  S,
  mu,
  sigma2,
  eta,
  c0,
  g0,
  G0,
  C0_0,
  b0,
  B0,
  M,
  burnin,
  thin,
  Kmax,
  G = c("MixDynamic", "MixStatic"),
  priorOnK,
  priorOnWeights,
  verbose = FALSE
)
}
\arguments{
\item{y}{A numeric matrix; containing the data.}

\item{S}{A numeric matrix; containing the initial cluster
assignments.}

\item{mu}{A numeric matrix; containing the initial cluster-specific
mean values.}

\item{sigma2}{A numeric matrix; containing the initial cluster-specific
variance values.}

\item{eta}{A numeric vector; containing the initial cluster sizes.}

\item{c0}{A numeric vector; hyperparameter of the prior on \eqn{\sigma^2_k}.}

\item{g0}{A numeric vector; hyperparameter of the prior on \eqn{\sigma^2_k}}

\item{G0}{A numeric vector; hyperparameter of the prior on \eqn{\sigma^2_k}}

\item{C0_0}{A numeric vector; initial value of hyperparameter \eqn{C_0}.}

\item{b0}{A numeric vector; hyperparameter of the prior on \eqn{\mu_k}.}

\item{B0}{A numeric vector; hyperparameter of the prior on \eqn{\mu_k}.}

\item{M}{A numeric scalar; specifying the number of recorded
iterations.}

\item{burnin}{A numeric scalar; specifying the number of burn-in
iterations.}

\item{thin}{A numeric scalar; specifying the thinning used for the
iterations.}

\item{Kmax}{A numeric scalar; the maximum number of components.}

\item{G}{A character string; either \code{"MixDynamic"} or \code{"MixStatic"}.}

\item{priorOnK}{A named list; providing the prior on the number of
components K, see \code{\link[=priorOnK_spec]{priorOnK_spec()}}.}

\item{priorOnWeights}{A named list; providing the prior on the mixture weights.}

\item{verbose}{A logical; indicating if some intermediate clustering
results should be printed.}
}
\value{
A named list containing:
\itemize{
\item \code{"Mu"}: sampled component means.
\item \code{"Sigma2"}: sampled component component variances.
\item \code{"Eta"}: sampled weights.
\item \code{"S"}: sampled assignments.
\item \code{"Nk"}: number of observations assigned to the different components, for each iteration.
\item \code{"K"}: sampled number of components.
\item \code{"Kplus"}: number of filled, i.e., non-empty components, for each iteration.
\item \code{"e0"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{e_0} is random).
\item \code{"alpha"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{\alpha} is random).
\item \code{"acc"}: logical vector indicating acceptance in the Metropolis-Hastings step when sampling either \eqn{e_0} or \eqn{\alpha}.
}
}
\description{
\itemize{
\item The MCMC scheme is implemented as suggested in Frühwirth-Schnatter et al (2021).
\item The priors on the model parameters are specified as in Frühwirth-Schnatter et al (2021),
see the vignette for details and notation.
\item The parametrizations of the gamma and inverse gamma distribution are used as in
Frühwirth-Schnatter et al (2021), see also the vignette.
}
}
\examples{
if (requireNamespace("mclust", quietly = TRUE)) {
    data("acidity", package = "mclust")
    y <- acidity
    
    N <- length(y)
    r <- 1
    
    M <- 200
    thin <- 1
    burnin <- 100
    Kmax <- 50  
    Kinit <- 10
    
    G <- "MixStatic" 
    priorOnE0 <- priorOnE0_spec("e0const", 0.01)
    priorOnK <- priorOnK_spec("Pois_1", 50)
    
    R <- diff(range(y))
    c0 <- 2 + (r-1)/2
    C0 <- diag(c(0.02*(R^2)), nrow = r)
    g0 <- 0.2 + (r-1) / 2
    G0 <- diag(10/(R^2), nrow = r)
    B0 <- diag((R^2), nrow = r)
    b0 <- as.matrix((max(y) + min(y))/2, ncol = 1)  
    
    cl_y <- kmeans(y, centers = Kinit, nstart = 100)
    S_0 <- cl_y$cluster
    mu_0 <- t(cl_y$centers)
    eta_0 <- rep(1/Kinit, Kinit)
    sigma2_0 <- array(0, dim = c(1, 1, Kinit))
    sigma2_0[1, 1, ] <- 0.5 * C0

    result <- sampleUniNormMixture(
        y, S_0, mu_0, sigma2_0, eta_0,
        c0, g0, G0, C0, b0, B0,
        M, burnin, thin, Kmax,
        G, priorOnK, priorOnE0)
    
    K <- result$K
    Kplus <- result$Kplus  
    
    plot(K, type = "l", ylim = c(0, max(K)),
         xlab = "iteration", main = "",
         ylab = expression("K" ~ "/" ~ K["+"]), col = 1)
    lines(Kplus, col = 2)
    legend("topright", legend = c("K", expression(K["+"])),
           col = 1:2, lty = 1, box.lwd = 0)
}

}
