#' Generate a chord set
#'
#' Generate a chord set for a music score.
#'
#' The chord set list returned by \code{chord_set} is only used for top center placement of a full set of chord fretboard diagrams for a music score.
#' \code{chord_set} returns a named list. The names are the chords and the list elements are strings defining string and fret fingering readable by LilyPond.
#' Multiple chord positions can be defined for the same chord name.
#' Instruments with a number of strings other than six are not currently supported.
#'
#' When defining chords, you may also wish to define rests or silence for chords added to a score for placement above the staff in time, where no chord is to be played.
#' Therefore, there are occasions where you may pass chord names and positions that happen to include entries \code{r} and/or \code{s} as \code{NA} as shown in the example.
#' These two special cases are passed through by \code{chord_set} but are ignored when the chord chart is generated.
#'
#' @param x character, n-string chord description from lowest to highest pitch, strings n through 1. E.g., \code{"xo221o"}. See details.
#' @param id character, the name of the chord in LilyPond readable format, e.g., \code{"a:m"}. Ignored if \code{x} is already a named vector.
#'
#' @return a named list.
#' @export
#'
#' @examples
#' chord_names <- c("e:m", "c", "d", "e:m", "d", "r", "s")
#' chord_positions <- c("xx997x", "x5553x", "x7775x", "ooo22o", "232oxx", NA, NA)
#' chord_set(chord_positions, chord_names)
chord_set <- function(x, id = NULL){
  if(!is.null(names(x))) id <- names(x)
  idx <- which(is.na(x))
  if(length(idx)) x2 <- x[idx]
  x <- x[!is.na(x)]
  id <- id[!id %in% c("r", "s")]
  f <- function(x) strsplit(gsub("\\(", " \\(", gsub("\\)", " ", x)), " ")[[1]] %>%
    purrr::map(~({
      if(substr(.x, 1, 1) == "(") substring(.x, 2) else strsplit(.x, "")[[1]]
    }
    )) %>% unlist() %>% paste(collapse = ";") %>% paste0(";")
  x <- purrr::map_chr(x, f)
  names(x) <- id
  if(length(idx)) x <- c(x, x2)
  x
}

#' Create a music track
#'
#' Create a music track from a collection of musical phrases.
#'
#' Musical phrases generated by \code{\link{phrase}} are fortified in a track table. All tracks are stored as track tables, one per row, even if that table consists of a single track.
#' \code{track} creates a single-entry track table. See \code{\link{trackbind}} for merging single tracks into a multi-track table. This is simply row binding that properly preserves phrase and track classes.
#'
#' The default for an additional music staff is \code{"treble_8"} for 8va treble clef, which is commonly displayed in quality guitar tablature above the tablature staff to include precise rhythm and timing information.
#' Note that guitar is a transposing instrument. For this reason, the default ID is \code{"treble_8"}, not \code{"treble"}.
#' Set \code{music_staff = NA} to suppress the additional music staff above the tablature staff.
#' This is appropriate for simple patterns where there are already multiple tracks and the additional space required for two staves per instrument is unnecessary and wasteful.
#'
#' The arguments \code{ms_transpose} and \code{ms_key} pertain to the transposition of the music staff relative to the tab staff if \code{music_staff} is not \code{NA}. These arguments default to 0 and \code{NA}, respectively.
#' The transposition and new key are simply stored in the \code{ms_transpose} and \code{ms_key} columns in the resulting track table.
#' This information is used by \code{lilypond} or \code{tab} to transpose the music staff relative to the tab staff at the time of LilyPond file generation.
#' Non-zero semitone transposition works without providing an explicit new key signature, but it is recommended to specify because it helps ensure the correct selection of accidentals in the output.
#' As with the \code{\link{transpose}} function, you can simply specify \code{key = "flat"} or \code{key = "sharp"}. The exact key signature is not required; it is merely more clear and informative for the user.
#'
#' @param phrase a phrase object.
#' @param tuning character, space-delimited pitches describing the instrument string tuning or a predefined tuning ID (see \code{\link{tunings}}). Defaults to standard guitar tuning. Tick or integer octave numbering accepted for custom tuning entries.
#' @param voice integer, ID indicating the unique voice \code{phrase} belongs to within a single track (another track may share the same tab/music staff but have a different voice ID).
#' @param music_staff add a standard sheet music staff above the tablature staff. See details.
#' @param ms_transpose integer, positive or negative number of semitones to transpose an included music staff relative to the tablature staff. See details.
#' @param ms_key character, specify the new key signature for a transposed music staff. See details.
#'
#' @return a track table.
#' @export
#'
#' @examples
#' x <- phrase("c ec'g' ec'g'", "4 4 2", "5 432 432")
#' track(x)
track <- function(phrase, tuning = "standard", voice = 1L, music_staff = "treble_8",
                  ms_transpose = 0, ms_key = NA){
  if(!"phrase" %in% class(phrase)) stop("`phrase` is not a phrase object.")
  x <- tibble::data_frame(phrase, tuning = .map_tuning(tuning), voice = as.integer(voice),
                          staff = as.character(music_staff),
                          ms_transpose = as.integer(ms_transpose),
                          ms_key = as.character(ms_key))
  x$phrase <- purrr::map(x$phrase, ~as.phrase(.x))
  class(x) <- unique(c("track", class(x)))
  x
}

#' Bind track tables
#'
#' Bind together track tables by row.
#'
#' This function appends multiple track tables into a single track table for preparation of generating a multi-track score.
#' \code{tabstaff} is used to separate music staves in the sheet music/tablature output. A track's \code{voice} is used to separate distinct voices within a common music staff.
#'
#' If not provided, the \code{tabstaff} ID automatically propagates \code{1:n} for \code{n} tracks passed to \code{...} when binding these tracks together. This expresses the default assumption of one tab staff per track.
#' This is the typical use case where each single \code{track} object being bound into a multi-\code{track} object is a fully separated track on its own staff.
#'
#' However, some tracks represent different voices that share the same staff. These should be assigned the same staff ID value, in which case you must provide the \code{tabstaff} argument.
#' An error will be thrown if any two tracks have both the same \code{voice} and the same \code{tabstaff}. The pair must be unique. E.g., provide \code{tabstaff = c(1, 1)} when you have two tracks with \code{voice} equal to 1 and 2. See examples.
#'
#' Note that the actual ID values assigned to each track do not matter; only the order in which tracks are bound, first to last.
#'
#' @param ... track tables.
#' @param tabstaff integer, ID vector indicating the tablature staff for each track. See details.
#'
#' @return a track table.
#' @export
#'
#' @examples
#' x <- phrase("c ec'g' ec'g'", "4 4 2", "5 432 432")
#' x1 <- track(x)
#' x2 <- track(x, voice = 2)
#' trackbind(x1, x1)
#' trackbind(x1, x2, tabstaff = c(1, 1))
trackbind <- function(..., tabstaff){
  x <- list(...)
  if(!all(purrr::map_lgl(x, ~any(class(.x) == "track")))) stop("All arguments must be `track` tables.")
  y <- if(missing(tabstaff)) seq_along(x) else tabstaff
  x <- purrr::map2(x, y, ~dplyr::mutate(.x, tabstaff = as.integer(.y)))
  x <- suppressWarnings(dplyr::bind_rows(x))
  if(nrow(dplyr::distinct(x, .data[["voice"]], .data[["tabstaff"]])) < nrow(x))
    stop("track `voice` and `tabstaff` ID combination must be unique across track rows.")
  x$phrase <- purrr::map(x$phrase, ~as.phrase(.x))
  class(x) <- unique(c("track", class(x)))
  x
}

as.phrase <- function(x){
  class(x) <- c("phrase", class(x))
  x
}

#' Create a music score
#'
#' Create a music score from a collection of tracks.
#'
#' Score takes track tables generated by \code{\link{track}} and fortifies them as a music score. It optionally binds tracks with a set of chord diagrams.
#' There may be only one track in \code{track} as well as no chord information passed, but for consistency \code{score} is still required to fortify the
#' single track as a score object that can be rendered by \code{\link{tab}}.
#'
#' @param track a track table consisting of one or more tracks.
#' @param chords an optional named list of chords and respective fingerings generated by \code{chord_set}, for inclusion of a top center chord diagram chart.
#' @param chord_seq an optional named vector of chords and their durations, for placing chord diagrams above staves in time.
#'
#' @return a score table.
#' @export
#'
#' @examples
#' x <- phrase("c ec'g' ec'g'", "4 4 2", "5 432 432")
#' x <- track(x)
#' score(x)
score <- function(track, chords = NULL, chord_seq = NULL){
  cl <- class(track)
  if(!"track" %in% cl) stop("`track` is not a `track` table.")
  if(!"tabstaff" %in% names(track)) track <- dplyr::mutate(track, tabstaff = 1L)
  class(track) <- unique(c("score", cl))
  attr(track, "chords") <- chords
  attr(track, "chord_seq") <- chord_seq
  track
}
