#'Construct Subspace Stream Clusterers
#'
#'This function allows you to combine a micro clustering algorithm and a macro 
#'clustering algorithm into a single object that can then be used as a normal 
#'DSC object. This object can then be used with e.g. \link{update} to produce a
#'micro clustering of a stream. 
#'@param micro a dsc object representing a micro clustering, which can be
#'  obtained by calling e.g. \link{DSC_subspaceCluStream}.
#'@param macro a dsc object representing a macroclustering
#'
#'@details
#'The microclustering component is implemented as usual, i.e. a
#'\emph{DSC_ThreeStage} object with a \link{DSC_subspaceCluStream}
#'microclustering component will produce the same microclusters that a normal
#'\emph{CluStream} would have produced. This is the first of the three stages.
#'
#'Whenever a macro clustering is requested, Multivariate Gaussian Distributions 
#'around the positions of the microclusters are used to simulate the original 
#'stream. This is the second stage.
#'
#'Then the macro clustering is performed on the points generated by these
#'distributions using the selected macro clustering algorithm. This is the third
#'stage.
#'
#'Possible choices for the micro clusterer are \link{DSC_subspaceDenStream} and 
#'\link{DSC_subspaceCluStream}. Possible macro clusterers are \link{DSC_clique},
#'\link{DSC_p3c}, \link{DSC_proclus} and \link{DSC_subclu}. Other clusterers are
#'currently not supported.
#'@export
#'@import methods
DSC_ThreeStage <- function(macro,micro) {
  if(!("DSC_SubspaceMOA_macro" %in% class(macro))) stop("Macroclusterer must have class DSC_SubspaceMOA_macro")
  if(!("DSC_SubspaceMOA_micro" %in% class(micro))) stop("Microclusterer must have class DSC_SubspaceMOA_micro")
  javaObj <- rJava::.jcall("moa/r_interface/ThreeStageClusterer",
                           "Lmoa/r_interface/RCompatibleDataStreamClusterer;",
                           "threeStage",
                           micro$javaObj,
                           macro$javaObj)
  if(rJava::is.jnull(javaObj)) stop("Creation of ThreeStageClusterer has failed")
  res <- structure(list(javaObj = javaObj),class = c("DSC_ThreeStage","DSC_SubspaceMOA","DSC"))
  return(res)
}
