/**
 * [PreDeConMicroCluster.java] for Subspace MOA
 * 
 * SCStream: PreDeCon microcluster class
 * 
 * @author Stephan Wels
 * Data Management and Data Exploration Group, RWTH Aachen University
 */

package moa.clusterers.scstream;

import java.util.Arrays;
import java.util.List;
import java.util.Vector;

import moa.cluster.CFCluster;

public class PreDeConMicroCluster {

	private static int id = 0;

	protected CFCluster mCluster;
	protected boolean mVisited;
	protected boolean mIsClustered;

	protected List<PreDeConMicroCluster> mEpsilonNeighborhood;

	protected List<PreDeConMicroCluster> mWeightedNeighborhood;

	// protected double[] mPrefDimVector;

	protected double[] mVariances;

	protected double[] mDimensions;
	protected int mNumDim;

	protected double mDelta;

	protected int mLambda;

	protected int mMu;

	protected double mEpsilon;

	protected double mOfflineEpsilon;

	protected int mKappa;

	protected boolean mCoreChanges;
	private int mNumRelDim;
	int myId = id++;

	// store to which cluster this microcluster belong, default none ==> null
	private PreDeConCluster mDenseCluster;

	public PreDeConMicroCluster(CFCluster mc, double delta, int lambda, int mu,
			double epsilon, int kappa) {
		mCluster = mc;
		mVisited = false;
		mNumDim = mc.LS.length;
		mDimensions = new double[mNumDim];
		mVariances = new double[mNumDim];
		// mPrefDimVector = new double[mNumDim];
		mEpsilonNeighborhood = new Vector<PreDeConMicroCluster>();
		mWeightedNeighborhood = new Vector<PreDeConMicroCluster>();
		mMu = mu;
		mEpsilon = epsilon;
		mDelta = delta;
		mKappa = kappa;
		mLambda = lambda;
		mDenseCluster = null;
	}

	public void preprocessing(List<PreDeConMicroCluster> microClustering) {
		double[] oldDimensions = mDimensions.clone();

		mEpsilonNeighborhood = epsNeighborhood(microClustering);
		mVariances = variances();
		subspacePrefVector();

		if (!Arrays.equals(mDimensions, oldDimensions)) {
			mCoreChanges = true;
		} else {
			mCoreChanges = false;
		}
		epsilonWeightedNeighborhood(microClustering);
	}

	public boolean updatePrefVector() {

		double[] oldDimensions = mDimensions.clone();
		mVariances = variances();
		subspacePrefVector();

		if (!Arrays.equals(mDimensions, oldDimensions))
			mCoreChanges = true;
		else
			mCoreChanges = false;

		return mCoreChanges;
	}

	// @Deprecated
	// public void preprocessing() {
	// double[] oldDimensions = mDimensions.clone();
	// mVariances = variances();
	// subspacePrefVector();
	//
	// if (!Arrays.equals(mDimensions, oldDimensions)) {
	// mCoreChanges = true;
	// } else {
	// mCoreChanges = false;
	// }
	// }

	private List<PreDeConMicroCluster> epsNeighborhood(
			List<PreDeConMicroCluster> microClustering) {
		List<PreDeConMicroCluster> inRange = new Vector<PreDeConMicroCluster>();
		for (PreDeConMicroCluster mc : microClustering) {
			if (this.distance(mc) <= mEpsilon)
				inRange.add(mc);
		}
		return inRange;
	}

	/**
	 * Calculates the variance for every dimension
	 */
	private double[] variances() {
		double dist = 0;

		double[] center = getCenter();
		double[] variances = new double[mNumDim];
		// for every dimension...
		for (int i = 0; i < mNumDim; i++) {
			dist = 0;
			double diff = 0;
			for (int j = 0; j < mEpsilonNeighborhood.size(); j++) {
				PreDeConMicroCluster neighbor = mEpsilonNeighborhood.get(j);
				diff = center[i] - neighbor.getCenter()[i];
				dist += Math.pow(diff, 2);
			}
			variances[i] = dist / mEpsilonNeighborhood.size();
		}
		return variances;
	}

	private void subspacePrefVector() {
		mNumRelDim = 0;
		for (int i = 0; i < mVariances.length; i++) {
			if (mVariances[i] <= mDelta) {
				mNumRelDim++;
				mDimensions[i] = 1;
			} else
				mDimensions[i] = 0;
		}
	}

	protected double asymmetricDistance(PreDeConMicroCluster first,
			PreDeConMicroCluster second) {
		double distance = 0d;
		double[] center1 = first.getCenter();
		double[] center2 = second.getCenter();

		for (int i = 0; i < first.mNumDim; i++) {
			double diff = center1[i] - center2[i];
			// weight the distances
			if (mVariances[i] > mDelta)
				distance += diff * diff;
			else
				distance += mKappa * diff * diff;
		}
		return Math.sqrt(distance);
	}

	public double distance_pref(PreDeConMicroCluster object) {
		double first, second;
		first = asymmetricDistance(this, object);
		second = asymmetricDistance(object, this);
		return Math.max(first, second);
	}

	/**
	 * Creates the preference weighted epsilon-neighborhood
	 */
	public void epsilonWeightedNeighborhood(
			List<PreDeConMicroCluster> microClusters) {
		mWeightedNeighborhood = new Vector<PreDeConMicroCluster>();
		// m_weighted_neighborhood.add(this);
		for (int i = 0; i < microClusters.size(); i++) {
			double debug = distance_pref(microClusters.get(i));
			if (distance_pref(microClusters.get(i)) > mEpsilon)
				continue;
			else {
				mWeightedNeighborhood.add(microClusters.get(i));
			}
		}
	}

	// optimazation: compute the weighted epsilon neighborhood with the epsilon
	// neighborhood
	public void epsilonWeightedNeighborhood() {
		mWeightedNeighborhood = new Vector<PreDeConMicroCluster>();
		// m_weighted_neighborhood.add(this);
		for (int i = 0; i < mEpsilonNeighborhood.size(); i++) {
			double debug = distance_pref(mEpsilonNeighborhood.get(i));
			if (distance_pref(mEpsilonNeighborhood.get(i)) > mEpsilon)
				continue;
			else {
				mWeightedNeighborhood.add(mEpsilonNeighborhood.get(i));
			}
		}
	}

	// @Deprecated
	// // optimazation: visit all my neighbours to tell them that I'm their new
	// // neighbor
	// public List<PreDeConMicroCluster> updateEpsNeighborhoodOfOthers() {
	// List<PreDeConMicroCluster> inRange = mEpsilonNeighborhood;
	// // inRange.add(m_inst);
	// for (int i = 0; i < inRange.size(); i++) {
	// PreDeConMicroCluster mc = inRange.get(i);
	// if (!mc.equals(this)) {
	// mc.mEpsilonNeighborhood.add(this);
	// }
	// }
	// return inRange;
	// }

	public boolean isCore() {

		List<PreDeConMicroCluster> mc = getWeightedNeighborhood();
		if (mc.size() < mMu)
			return false;
		if (mNumRelDim > mLambda)
			return false;
		return true;
	}

	public boolean isDirReachable(PreDeConMicroCluster mc) {
		if (!isCore())
			return false;
		if (mc.mNumRelDim > mLambda)
			return false;
		boolean found = false;
		for (int i = 0; i < mWeightedNeighborhood.size(); i++) {
			double[] center1 = mWeightedNeighborhood.get(i).getCenter();
			double[] center2 = mc.getCenter();

			if (Arrays.equals(center1, center2)) {
				found = true;
				break;
			}
		}
		return found;
	}

	public List<PreDeConMicroCluster> getEpsilonNeighborhood() {
		return mEpsilonNeighborhood;
	}

	public String toString() {
		String s = "[";
		for (int i = 0; i < mEpsilonNeighborhood.size(); i++) {
			s += mEpsilonNeighborhood.get(i).myId + ", ";
		}
		return myId + " " + s + "]";
	}

	public void setVisited() {
		mVisited = true;
	}

	public boolean isVisited() {
		return mVisited;
	}

	public void setClustered() {
		mIsClustered = true;
	}

	public boolean isClustered() {
		return mIsClustered;
	}

	public CFCluster getCFCluster() {
		return mCluster;
	}

	public List<PreDeConMicroCluster> getWeightedNeighborhood() {
		return mWeightedNeighborhood;
	}

	// compute euclidean distance between the two centers
	private double distance(PreDeConMicroCluster o) {
		return distance(this.getCenter(), o.getCenter());
	}

	private double distance(double[] center, double[] center2) {
		double d = 0D;
		for (int i = 0; i < center.length; i++) {
			d += Math.pow((center[i] - center2[i]), 2);
		}
		return Math.sqrt(d);
	}

	public double[] getCenter() {
		return mCluster.getCenter();
	}

	public double getRadius() {
		return mCluster.getRadius();
	}

	public void setCluster(PreDeConCluster cluster) {
		mDenseCluster = cluster;
	}

	public PreDeConCluster getCluster() {
		return mDenseCluster;
	}
	
	public boolean equals(Object o){
		PreDeConMicroCluster s = (PreDeConMicroCluster)o;
		return s.myId == myId;
	}
}
