\name{parametricFamily}
\alias{parametricFamily}
\alias{parametricfamily}

\title{Parametric families}

\description{
Overview about the supported parametric families (models). More details are given in section 5 of the vignette.
}

\details{
The following parametric families (models and fitting methods) are available. Some of them have additional parameters that have to / can be specified in \code{...}.

\describe{
\item{\code{"gauss"}}{independent normal distributed variables with unknown mean but known, constant standard deviation given by the optional argument \code{sd}. Fits are obtained by the method \acronym{SMUCE} (\cite{Frick et al.}, 2014) for independent normal distributed observations. Argument \code{sd} has to be a single, positive, finite \code{\link{numeric}}. If omitted it will be estimated by \code{\link{sdrobnorm}}. For \code{\link{monteCarloSimulation}} \code{sd == 1} will be used always. The observations argument \code{y} has to be a numeric vector with finite entries. The default \link[=intervalSystem]{interval system} is \code{"all"} up to 1000 observations and \code{"dyaLen"} for more observations. Possible lengths are \code{1:length(y)}. The default \link{penalty} is \code{"sqrt"}. In \code{\link{monteCarloSimulation}} by default \code{n} random observations will be generated by \code{\link{rnorm}}.}

\item{\code{"hsmuce"}}{independent normal distributed variables with unknown mean and also unknown piecewise constant standard deviation as a nuisance parameter. Fits are obtained by the method \acronym{HSMUCE} (\cite{Pein et al.}, 2017). No additional argument has to be given. The observations argument \code{y} has to be a numeric vector with finite entries. The default \link[=intervalSystem]{interval system} is \code{"dyaPar"} and possible lengths are \code{2:length(y)}. The default \link{penalty} is \code{"weights"}  which will automatically be converted to \code{"none"} in \code{\link{computeStat}} and \code{\link{monteCarloSimulation}}. In \code{\link{monteCarloSimulation}} by default \code{n} random observations will be generated by \code{\link{rnorm}}.}

\item{\code{"mDependentPS"}}{normal distributed variables with unknown mean and m-dependent errors with known covariance structure given either by the argument \code{covariances}, \code{correlations} or \code{filter}. Fits are obtained by the method \acronym{SMUCE} (\cite{Frick et al.}, 2014) for m-dependent normal distributed observations using partial sum tests and minimizing the least squares distance (\cite{Pein et al.}, 2017, (7) and (8)). If \code{correlations} or \code{filter} is used to specify the covariances an additional optional argument \code{sd} can be used to specify the constant standard deviation. If \code{covariances} is specified the arguments \code{correlations}, \code{filter} and \code{sd} will be ignored and if \code{correlations} is specified the argument \code{filter} will be ignored. The argument \code{covariances} has to be a finite numeric vector, m will be defined by \code{m = length(covariances) - 1}, giving the vector of covariances, i.e. the first element must be positive, the absolute value of every other element must be smaller than or equal to the first one and the last element should not be zero. The argument \code{correlation} has to be a finite numeric vector, m will be defined by \code{m = length(correlations) - 1}, giving the vector of correlations, i.e. the first element must be 1, the absolute value of every other element must be smaller than or equal to the first one and the last element should not be zero. Covariances will be calculated by \code{correlations * sd^2}. The argument \code{filter} has to be an object of class \code{lowpassFilter} from which the correlation vector will be obtained. A definition of class \code{lowpassFilter} and a function to obtain objects of this class will be published in a separate package. The argument \code{sd} has to be a single, positive, finite \code{\link{numeric}}. If omitted it will be estimated by \code{\link{sdrobnorm}} with \code{lag = m + 1}. For \code{\link{monteCarloSimulation}} \code{sd == 1} will be used always. The observations argument \code{y} has to be a numeric vector with finite entries. The default \link[=intervalSystem]{interval system} is \code{"dyaLen"} and possible lengths are \code{1:length(y)}. The default \link{penalty} is \code{"sqrt"}. In \code{\link{monteCarloSimulation}} by default \code{n} random observations will be generated by calculating the coefficients of the corresponding moving average process and generating random observations from it.}
}

The family is selected via the \code{family} argument, providing the corresponding string, while additional parameters have to / can be specified in \code{\dots} if any.
}

\encoding{utf8}
\references{
Frick, K., Munk, A., Sieling, H. (2014) Multiscale change-point inference. With discussion and rejoinder by the authors. \emph{Journal of the Royal Statistical Society, Series B} \bold{76}(3), 495--580.

Pein, F., Sieling, H., Munk, A. (2017) Heterogeneous change point inference. \emph{Journal of the Royal Statistical Society, Series B}, \bold{79}(4), 1207--1227.

Pein, F., Tecuapetla-Gómez, I., \enc{Schütte}{Schuette}, O., Steinem, C., Munk, A. (2017) Fully-automatic multiresolution idealization for filtered ion channel recordings: flickering event detection. \emph{arXiv}:1706.03671.
}

\seealso{\link{Distributions}, \code{\link{sdrobnorm}}, \code{\link{rnorm}}}

\examples{
# parametric family "gauss": independent gaussian errors with constant variance
set.seed(1)
x <- seq(1 / 100, 1, 1 / 100)
y <- c(rnorm(50), rnorm(50, 2))
plot(x, y, pch = 16, col = "grey30", ylim = c(-3, 5))

# computation of SMUCE and its confidence statements
fit <- stepFit(y, x = x, alpha = 0.5, family = "gauss",
               jumpint = TRUE, confband = TRUE)
lines(fit, lwd = 3, col = "red", lty = "22")

# confidence intervals for the change-point locations
points(jumpint(fit), col = "red")
# confidence band
lines(confband(fit), lty = "22", col = "darkred", lwd = 2)

# "gauss" is default for family
identical(stepFit(y, x = x, alpha = 0.5, jumpint = TRUE, confband = TRUE), fit)
# missing sd is estimated by sdrobnorm
identical(stepFit(y, x = x, alpha = 0.5, family = "gauss", sd = sdrobnorm(y),
                  jumpint = TRUE, confband = TRUE), fit)

# parametric family "hsmuce": independent gaussian errors with also
# piecewise constant variance
# estimaton that is robust against variance changes
set.seed(1)
y <- c(rnorm(50, 0, 1), rnorm(50, 1, 0.2))
plot(x, y, pch = 16, col = "grey30", ylim = c(-2.5, 2))

# computation of HSMUCE and its confidence statements
fit <- stepFit(y, x = x, alpha = 0.5, family = "hsmuce",
               jumpint = TRUE, confband = TRUE)
lines(fit, lwd = 3, col = "red", lty = "22")

# confidence intervals for the change-point locations
points(jumpint(fit), col = "red")
# confidence band
lines(confband(fit), lty = "22", col = "darkred", lwd = 2)

# for comparison SMUCE
lines(stepFit(y, x = x, alpha = 0.5, jumpint = TRUE, confband = TRUE),
      lwd = 3, col = "blue", lty = "22")


# parametric family "mDependentPS": m dependent observations with known covariances
# observations are generated from a moving average process
set.seed(1)
y <- c(rep(0, 50), rep(2, 50)) +
  as.numeric(arima.sim(n = 100, list(ar = c(), ma = c(0.8, 0.5, 0.3)), sd = 0.5))
correlations <- as.numeric(ARMAacf(ar = c(), ma = c(0.8, 0.5, 0.3), lag.max = 3))
covariances <- 0.5^2 * correlations
plot(x, y, pch = 16, col = "grey30", ylim = c(-2, 4))

# computation of SMUCE for dependent observations with given covariances
fit <- stepFit(y, x = x, alpha = 0.5, family = "mDependentPS",
               covariances = covariances, jumpint = TRUE, confband = TRUE)
lines(fit, lwd = 3, col = "red", lty = "22")

# confidence intervals for the change-point locations
points(jumpint(fit), col = "red")
# confidence band
lines(confband(fit), lty = "22", col = "darkred", lwd = 2)

# for comparison SMUCE for independent gaussian errors
lines(stepFit(y, x = x, alpha = 0.5, jumpint = TRUE, confband = TRUE),
      lwd = 3, col = "blue", lty = "22")

# covariance structure can also be given by correlations and sd
identical(stepFit(y, x = x, alpha = 0.5, family = "mDependentPS",
                  correlations = correlations, sd = 0.5,
                  jumpint = TRUE, confband = TRUE), fit)

# if sd is missing it will be estimated by sdrobnorm
identical(stepFit(y, x = x, alpha = 0.5,family = "mDependentPS",
                  correlations = correlations, jumpint = TRUE, confband = TRUE),
          stepFit(y, x = x, alpha = 0.5, family = "mDependentPS",
                  correlations = correlations,
                  sd = sdrobnorm(y, lag = length(correlations)),
                  jumpint = TRUE, confband = TRUE))
}

\keyword{distribution}
