% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/impactspsur.R
\name{impactspsur}
\alias{impactspsur}
\title{Direct, indirect and total effects estimated for a spatial SUR model}
\usage{
impactspsur (obj, ..., tr = NULL, R = NULL, listw = NULL, 
                      evalues = NULL,tol = 1e-06, 
                      empirical = FALSE, Q = NULL)
}
\arguments{
\item{obj}{An \code{spsur} object created by \code{\link{spsurml}},
\code{\link{spsur3sls}} or \code{\link{spsurtime}}.}

\item{...}{Arguments passed through to methods in the \pkg{coda} package}

\item{tr}{A vector of traces of powers of the spatial weights matrix created using \code{trW}, for approximate impact measures; if not given, \code{listw} must be given for exact measures (for small to moderate spatial weights matrices); the traces must be for the same spatial weights as were used in fitting the spatial regression, and must be row-standardised}

\item{R}{If given, simulations are used to compute distributions for the impact measures, returned as \code{mcmc} objects; the objects are used for convenience but are not output by an MCMC process}

\item{listw}{If \code{tr} is not given, a spatial weights object as created by \code{nb2listw}; they must be the same spatial weights as were used in fitting the spatial regression, but do not have to be row-standardised}

\item{evalues}{vector of eigenvalues of spatial weights matrix for impacts calculations}

\item{tol}{Argument passed to \code{mvrnorm}: tolerance (relative to largest variance) for numerical lack of positive-definiteness in the coefficient covariance matrix}

\item{empirical}{Argument passed to \code{mvrnorm} (default FALSE): if true, the coefficients and their covariance matrix specify the empirical not population mean and covariance matrix}

\item{Q}{default NULL, else an integer number of cumulative power series impacts to calculate if \code{tr} is given}
}
\value{
A list of \emph{G} objects either of class \code{lagImpact} 
or \code{WXImpact}.

For each of the G objects of the list, if no simulation is carried out 
the object returned is a list with:
\tabular{ll}{
  \code{direct} \tab numeric vector \cr
  \code{indirect} \tab numeric vector \cr
  \code{total} \tab numeric vector \cr
} 
and a matching \code{Qres} list attribute if {Q} was given.

On the other hand, for each of the G objects of the list, if 
simulation is carried out the object returned is a list with:
\tabular{ll}{  
  \code{res} \tab	a list with three components as for the non-simulation 
  case, with a matching \code{Qres} list attribute if \code{Q} was given. \cr
  \code{sres} \tab a list with three \code{mcmc} matrices, for the direct, 
  indirect and total impacts with a matching \code{Qmcmc}list attribute 
  if \code{Q} was given. \cr
}
}
\description{
This function is a wrapper for \code{\link[spatialreg]{impacts}} method
  used in \pkg{spatialreg} package. Nevertheless, in this case the same 
  method is used for both \code{\link[spatialreg]{lagsarlm}} and 
  \code{\link[spatialreg]{lmSLX}} objects. For details of implementation, 
  see the documentation of \code{\link[spatialreg]{impacts}} function in 
  \pkg{spatialreg} package. \cr
  The function obtains the multiplier effects, on the explained variable, 
  of a change in a regressor for the model that has been estimated. 
  For reasons given below, this function only applies to models with an 
  autoregressive structure ("slm", "sdm", "sarar" and "gnm") or with spatial lags 
  of the regressors ("slx", "sdem"). \cr
  The measurement of the multiplier effects is a bit more complicated than 
  in a pure time series context because, due to the spatial structure of 
  the model, part of the impacts spills over non uniformly over the space. 
  Using the notation introduced by LeSage and Pace (2009) we distinguish 
  between:
  \itemize{
    \item \strong{Average Direct effects}: The average over the \emph{N} 
    spatial units and \emph{Tm} time periods of the effect of a unitary 
    change in the value of a explanatory variable on the contemporaneous 
    value of the corresponding explained variable, located in the same 
    point of the intervened regressor. This calculus is solved for all the 
    regressors that appear in the \emph{G} equations of the model.
    \item \strong{Average Indirect effects}: The average over the \emph{N} 
    spatial units and \emph{Tm} time periods of the effects of a unitary 
    change in the value of a explanatory variable on the contemporaneous 
    value of the corresponding explained variable, located in a different 
    spatial unit that that of the intervened regressor. This calculus is 
    solved for all the regressors that appear in the \emph{G} equations of 
    the model.
    \item \strong{Average total effects}: The sum of Direct and 
    Indirect effects.
  }

   The information on the three estimated effects is supplement with an 
   indirect measure of statistical significance obtained from the 
   randomization approach introduced in LeSage and Pace (2009).
}
\details{
LeSage and Pace (2009) adapt the classical notion of 
 \emph{'economic multiplier'} to the problem of measuring the impact that 
 a unitary change in the value of a regressor, produced in a certain point 
 in space, has on the explained variable. The question is interesting 
 because, due to the spatial structure of the model, the impacts of such 
 change spill non uniformly over the space. In fact, the reaction of the 
 explained variable depends on its relative location in relation to the 
 point of intervention. \cr
 To simplify matters, LeSage and Pace (2009) propose to obtain aggregated 
 multipliers for each regressor, just averaging the \eqn{N^{2}} impacts 
 that results from intervening the value of each regressor on each of the
 N points in Space, on the explained variable, measured also in each of 
 the \eqn{N} points in space. This aggregated average is the so-called 
 \emph{Total effect}. \cr
 Part of this impact will be absorved by the explained variable located in 
 the same point of the regressor whose value has been changed (for example, 
 the k-th regresor in the g-th equation, in the n-th spatial unit) or,
 in other words, we expect that \eqn{[d y_{tgn}]/[d x_{ktgn}] ne 0}. The 
 aggregated average for the \emph{N} points in space (n=1,2,...,N) and 
 \emph{Tm} time periods is the so-called \emph{Direct effect}.
 The difference between the \emph{Total effect} and the \emph{Direct effect} 
 measures the portion of the impact on the explained variable that leakes 
 to other points in space, \eqn{[d y_{tgn}]/[d x_{ktgm}] for n ne m};
 this is the \emph{Indirect effect}.

 \code{\link{impacts}} obtains the three multipliers together with an 
 indirect measure of statistical significance, according to the 
 randomization approach described in Lesage and Pace (2009). Briefly, they 
 suggest to obtain a sequence of \emph{nsim} random matrices of order 
 \emph{(NTmxG)} from a multivariate normal distribution 
 N(0; \emph{Sigma}), being \emph{Sigma} the estimated covariance matrix 
 of the \emph{G} equations in the SUR model. These random matrices, 
 combined with the observed values of the regressors and the estimated 
 values of the parameters of the corresponding spatial SUR model, are used 
 to obtain simulated values of the explained variables. Then, for each one 
 of the \emph{nsim} experiments, the SUR model is estimated, and the effects
 are evaluated. The function \code{\link{impacts}} obtains the standard 
 deviations of the \emph{nsim} estimated effects in the randomization 
 procedure, which are used to test the significance of the estimated 
 effects for the original data.

 Finally, let us note that this is a SUR model where the \emph{G} equations 
 are connected only through the error terms. This means that if we 
 intervene a regressor in equation \emph{g}, in any point is space, only 
 the explained variable of the same equation \emph{g} should react. The 
 impacts do not spill over equations. Moreover, the impact of a regressor, 
 intervened in the spatial unit \emph{n}, will cross the borders of this 
 spatial unit only if in the right hand side of the equation there are 
 spatial lags of the explained variables or of the regressors. In other 
 words, the \emph{Indirect effect} is zero for the "sim" and "sem" models. 
 \code{\link{impacts}} produces no output for these two models.
 Lastly, it is clear that all the impacts are contemporaneous because the 
 equations in the SUR model have no time dynamics.
}
\examples{

## VIP: The output of the whole set of the examples can be examined 
## by executing demo(demo_impactspsur, package="spsur")

\donttest{
###############################################
### PURE CROSS SECTIONAL DATA(G>1; Tm=1) ######
###############################################

#### Example 1: Spatial Phillips-Curve. Anselin (1988, p. 203)
 rm(list = ls()) # Clean memory
 data(spc)
 lwspc <- spdep::mat2listw(Wspc, style = "W")
 Tformula <- WAGE83 | WAGE81 ~ UN83 + NMR83 + SMSA | UN80 + NMR80 + SMSA
## For SLM, SDM and SARAR models the output is a list of "lagImpact" objects
## See spatialreg::impacts for details.
 spcsur_slm <-spsurml(formula = Tformula, data = spc, 
                      type = "slm", listw = lwspc)
 summary(spcsur_slm)
 impacts_slm <- impactspsur(spcsur_slm, listw = lwspc, R = 1000)
## Impacts equation 1
 summary(impacts_slm[[1]], zstats = TRUE, short = TRUE)
## Impacts equation 2
 summary(impacts_slm[[2]], zstats = TRUE, short = TRUE)
## For SLX and SDEM models the output is a list of "WXImpact" objects
## See spatialreg::impacts for details.
## A SUR-SLX model
 spcsur_slx <-spsurml(formula = Tformula, data = spc, 
                      type = "slx", listw = lwspc)
 summary(spcsur_slx)
 impacts_slx <- impactspsur(spcsur_slx, listw = lwspc)
 summary(impacts_slx[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_slx[[2]], zstats = TRUE, short = TRUE)

## A SUR-SDM model
 spcsur_sdm <-spsurml(formula = Tformula, data = spc, 
                      type = "sdm", listw = lwspc)
 impacts_sdm <- impactspsur(spcsur_sdm, listw = lwspc, R = 1000)
 summary(impacts_sdm[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_sdm[[2]], zstats = TRUE, short = TRUE)
## A SUR-SDM model with different spatial lags in each equation
 TformulaD <- ~ UN83 + NMR83 + SMSA | UN80
 spcsur_sdm2 <-spsurml(formula = Tformula, data = spc, type = "sdm", 
                       listw = lwspc, Durbin = TformulaD)
 summary(spcsur_sdm2)                       
 impacts_sdm2 <- impactspsur(spcsur_sdm2, listw = lwspc, R = 1000)
 summary(impacts_sdm2[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_sdm2[[2]], zstats = TRUE, short = TRUE)
 ## A SUR-SLX model with different spatial lags in each equation
 spcsur_slx2 <-spsurml(formula = Tformula, data = spc, 
                      type = "slx", listw = lwspc, Durbin = TformulaD)
 summary(spcsur_slx2)
 impacts_slx2 <- impactspsur(spcsur_slx2, listw = lwspc)
 summary(impacts_slx2[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_slx2[[2]], zstats = TRUE, short = TRUE)
### A SUR-SDEM model
 spcsur_sdem <-spsurml(formula = Tformula, data = spc, 
                      type = "sdem", listw = lwspc)
 impacts_sdem <- impactspsur(spcsur_sdem, listw = lwspc)
 summary(impacts_sdem[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_sdem[[2]], zstats = TRUE, short = TRUE)

### A SUR-SARAR model
 spcsur_sarar <-spsurml(formula = Tformula, data = spc, 
                      type = "sarar", listw = lwspc,
                      control = list(tol = 0.01))
 impacts_sarar <- impactspsur(spcsur_sarar, listw = lwspc, R = 1000)
 summary(impacts_sarar[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_sarar[[2]], zstats = TRUE, short = TRUE)
 
## A SUR-GNM model
 spcsur_gnm <-spsurml(formula = Tformula, data = spc, 
                      type = "gnm", listw = lwspc,
                      control = list(tol = 0.1))
 impacts_gnm <- impactspsur(spcsur_gnm, listw = lwspc, R = 1000)
 summary(impacts_gnm[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_gnm[[2]], zstats = TRUE, short = TRUE)
## A SUR-GNM model with different spatial lags in each equation
 TformulaD <- ~ UN83 + NMR83 + SMSA | UN80
 spcsur_gnm2 <-spsurml(formula = Tformula, data = spc, type = "gnm", 
                       listw = lwspc, Durbin = TformulaD,
                       control = list(tol = 0.1))
 summary(spcsur_gnm2)                       
 impacts_gnm2 <- impactspsur(spcsur_gnm2, listw = lwspc, R = 1000)
 summary(impacts_gnm2[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_gnm2[[2]], zstats = TRUE, short = TRUE)
 
# ####################################
# ######## G=1; Tm>1               ###
# ####################################
#
 rm(list = ls()) # Clean memory
 data(NCOVR, package="spsur")
 nbncovr <- spdep::poly2nb(NCOVR.sf, queen = TRUE)
### Some regions with no links...
 lwncovr <- spdep::nb2listw(nbncovr, style = "W", zero.policy = TRUE)
 Tformula <- HR80  | HR90 ~ PS80 + UE80 | PS90 + UE90
### A SUR-SLM model
 NCOVRSUR_slm <-spsurml(formula = Tformula, data = NCOVR.sf, 
                        type = "slm", listw = lwncovr, 
                        method = "Matrix", zero.policy = TRUE, 
                        control = list(fdHess = TRUE))
 summary(NCOVRSUR_slm)
### Use of trW to compute.
 Wncovr <- as(spdep::listw2mat(lwncovr), "CsparseMatrix")
 trwncovr <- spatialreg::trW(Wncovr, type = "MC")
 impacts_NCOVRSUR_slm <- impactspsur(NCOVRSUR_slm, tr = trwncovr,
                                 R = 1000)
 summary(impacts_NCOVRSUR_slm[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_NCOVRSUR_slm[[2]], zstats = TRUE, short = TRUE)
### A SUR-SDM model
 NCOVRSUR_sdm <-spsurml(formula = Tformula, data = NCOVR.sf, 
                        type = "sdm", listw = lwncovr, 
                        method = "Matrix", zero.policy = TRUE, 
                        control = list(fdHess = TRUE))
 impacts_NCOVRSUR_sdm <- impactspsur(NCOVRSUR_sdm, tr = trwncovr,
                                 R = 1000)
 summary(impacts_NCOVRSUR_sdm[[1]], zstats = TRUE, short = TRUE)
 summary(impacts_NCOVRSUR_sdm[[2]], zstats = TRUE, short = TRUE)
}

}
\references{
\itemize{
    \item Bivand, R.S. and  Piras G. (2015). Comparing Implementations of 
       Estimation Methods for Spatial Econometrics. \emph{Journal of 
       Statistical Software}, 63(18), 1-36. 
       https://www.jstatsoft.org/v63/i18/.
     \item LeSage, J., and Pace, R. K. (2009). \emph{Introduction to spatial
       econometrics}. Chapman and Hall/CRC.
     \item López, F.A., Mur, J., and Angulo, A. (2014). Spatial model
       selection strategies in a SUR framework. The case of regional
       productivity in EU. \emph{Annals of Regional Science}, 53(1), 197-220.
     \item Mur, J., López, F., and Herrera, M. (2010). Testing for spatial
       effects in seemingly unrelated regressions. 
       \emph{Spatial Economic Analysis}, 5(4), 399-440.
  }
}
\seealso{
\code{\link[spatialreg]{impacts}}, \code{\link{spsurml}}, 
\code{\link{spsur3sls}}
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Jesús Mur  \tab \email{jmur@unizar.es} \cr
  }
}
