% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/localC-dev.R
\name{localC}
\alias{localC}
\alias{localC.default}
\alias{localC.formula}
\alias{localC.list}
\alias{localC.matrix}
\alias{localC.data.frame}
\alias{localC_perm}
\alias{localC_perm.default}
\alias{localC_perm.formula}
\title{Compute Local Geary statistic}
\usage{
localC(x, ..., zero.policy=NULL)

\method{localC}{default}(x, listw, ..., zero.policy=NULL)

\method{localC}{formula}(formula, listw, data, ..., zero.policy=NULL)

\method{localC}{list}(x, listw, ..., zero.policy=NULL)

\method{localC}{matrix}(x, listw, ..., zero.policy=NULL)

\method{localC}{data.frame}(x, listw, ..., zero.policy=NULL)

localC_perm(x, ..., conditional=FALSE, zero.policy=NULL)

\method{localC_perm}{default}(x, listw, nsim = 499, alternative = "less", ...,
 conditional=FALSE, zero.policy=NULL)

\method{localC_perm}{formula}(formula, listw, data, nsim = 499,
 alternative = "less", ..., conditional=FALSE, zero.policy=NULL)
}
\arguments{
\item{x}{a numeric vector, numeric matrix, or list. See details for more.}

\item{formula}{A one-sided formula determining which variables to be used.}

\item{listw}{a \code{listw} object created for example by \code{nb2listw}.}

\item{data}{Used when a formula is provided. A matrix or data frame containing the variables in the formula \code{formula}.}

\item{nsim}{The number of simulations to be used for permutation test.}

\item{alternative}{A character defining the alternative hypothesis. Must be one of \code{"less"} or \code{"greater"}.}
\item{...}{other arguments passed to methods.}
\item{zero.policy}{default NULL, use global option value; if TRUE assign zero to the lagged value of zones without neighbours, if FALSE assign NA.}
\item{conditional}{default FALSE, sample from the whole data set; if TRUE, sample from all observations but the one which is being tested.}
}
\description{
The Local Geary is a local adaptation of Geary's C statistic of spatial autocorrelation. The Local Geary uses squared differences to measure dissimilarity unlike the Local Moran. Low values of the Local Geary indicate positive spatial autocorrelation and large refers to negative spatial autocorrelation.

Inference for the Local Geary is based on a permutation approach which compares the observed value to the reference distribution under spatial randomness. \code{localC_perm()} returns a pseudo p-value. This is not an analytical p-value and is based on the number of permutations and as such should be used with care.
}
\details{
The Local Geary can be extended to a multivariate context. When \code{x} is a numeric vector, the univariate Local Geary will be calculated. To calculate the multivariate Local Moran provide either a list or a matrix. When \code{x} is a list, each element must be a numeric vector of the same length and of the same length as the neighbours in \code{listw}. In the case that \code{x} is a matrix the number of rows must be the same as the length of the neighbours in \code{listw}.

While not required in the univariate context, the standardized Local Geary is calculated. The multivariate Local Geary is \emph{always} standardized.

The univariate Local Geary is calculated as \eqn{c_i = \sum_j w_{ij}(x_i - x_j)^2} and the multivariate Local Geary is calculated as \eqn{c_{k,i} = \sum_{v=1}^{k} c_{v,i}} as described in Anselin (2019).
}
\examples{
orig <- spData::africa.rook.nb
listw <- nb2listw(orig)
x <- spData::afcon$totcon

(A <- localC(x, listw))
listw1 <- nb2listw(droplinks(sym.attr.nb(orig), 3, sym=TRUE), zero.policy=TRUE)
(A1 <- localC(x, listw1, zero.policy=FALSE))
(A2 <- localC(x, listw1, zero.policy=TRUE))
run <- FALSE
if (require(rgeoda, quietly=TRUE)) run <- TRUE
if (run) {
  W <- create_weights(as.numeric(length(x)))
  for (i in 1:length(listw$neighbours)) {
    set_neighbors_with_weights(W, i, listw$neighbours[[i]], listw$weights[[i]])
    update_weights(W)
  }
  set.seed(1)
  B <- local_geary(W, data.frame(x))
  all.equal(A, lisa_values(B))
}
if (run) {
  set.seed(1)
  C <- localC_perm(x, listw, nsim = 499, conditional=TRUE,
    alternative="two.sided")
  cor(ifelse(lisa_pvalues(B) < 0.5, lisa_pvalues(B), 1-lisa_pvalues(B)),
    attr(C, "pseudo-p")[,6])
}
# pseudo-p values probably wrongly folded https://github.com/GeoDaCenter/rgeoda/issues/28
\dontrun{
library(reticulate)
use_python("/usr/bin/python", required = TRUE)
gp <- import("geopandas")
ps <- import("libpysal")
tf <- tempfile(fileext=".gal")
write.nb.gal(orig, tf)
fl <- ps$io$open(tf)
w$transform <- "R"
esda <- import("esda")
lM <- esda$Moran_Local(x, w)
all.equal(unname(localmoran(x, listw, mlvar=FALSE)[,1]), c(lM$Is))
# confirm x and w the same
lC <- esda$Geary_Local(connectivity=w)$fit(scale(x))
# np$std missing ddof=1 
n <- length(x)
D01 <- spdep:::geary.intern((x - mean(x)) / sqrt(var(x)*(n-1)/n), listw, n=n)
# lC components probably wrongly ordered https://github.com/pysal/esda/issues/192
o <- match(round(D0, 6), round(lC$localG, 6))
all.equal(c(lC$localG)[o], D0)
# simulation order not retained
lC$p_sim[o]
attr(C, "pseudo-p")[,6]
}
}
\references{
{Anselin, L. (1995), Local Indicators of Spatial Association—LISA. Geographical Analysis, 27: 93-115. \doi{10.1111/j.1538-4632.1995.tb00338.x}}

{Anselin, L. (2019), A Local Indicator of Multivariate Spatial Association: Extending Geary's c. Geogr Anal, 51: 133-150. \doi{10.1111/gean.12164}}
}
\author{
Josiah Parry, \email{josiah.parry@gmail.com}
}
