% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_polys.R
\name{build_polys}
\alias{build_polys}
\title{Build Polygons}
\usage{
build_polys(
  DT = NULL,
  projection = NULL,
  hrType = NULL,
  hrParams = NULL,
  id = NULL,
  coords = NULL,
  splitBy = NULL,
  spPts = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{projection}{character string defining the projection to be passed to
\code{sp::CRS}. For example, for UTM zone 36S (EPSG 32736),
the projection argument is 'EPSG:32736'. See details.}

\item{hrType}{type of HR estimation, either 'mcp' or 'kernel'}

\item{hrParams}{a named list of parameters for \code{adehabitatHR} functions}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}

\item{spPts}{alternatively, provide solely a SpatialPointsDataFrame with one
column representing the ID of each point.}
}
\value{
\code{build_polys} returns a \code{SpatialPolygons} object
with a polyon for each individual (and optionally \code{splitBy}
combination).

An error is returned when \code{hrParams} do not match the arguments
of the \code{hrType} \code{adehabitatHR} function.
}
\description{
\code{build_polys} creates a \code{SpatialPolygons} object from a
\code{data.table}. The function accepts a \code{data.table} with
relocation data, individual identifiers, a \code{projection},
\code{hrType} and \code{hrParams}. The relocation data is transformed
into \code{SpatialPolygons} for each individual and optionally, each
\code{splitBy}. Relocation data should be in two columns representing
the X and Y coordinates.
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using
\code{\link[data.table:setDT]{data.table::setDT}}.

The \code{id}, \code{coords} (and optional \code{splitBy}) arguments
expect the names of respective columns in \code{DT} which correspond
to the individual identifier, X and Y coordinates, and additional
grouping columns.

The \code{projection} argument expects a character string defining
the EPSG code. For example, for UTM zone 36N (EPSG 32736), the projection
argument is "EPSG:32736". See \url{https://spatialreference.org}
for a list of EPSG codes. Please note, R spatial has followed updates
to GDAL and PROJ for handling projections, see more at
\url{https://www.r-spatial.org/r/2020/03/17/wkt.html}. It is likely
that \code{build_polys} will return "Warning in proj4string(xy) :
CRS object has comment, which is lost in output" due to these changes.

The \code{hrType} must be either one of "kernel" or "mcp". The
\code{hrParams} must be a named list of arguments matching those
of \code{adehabitatHR::kernelUD} and \code{adehabitatHR::getverticeshr}
or \code{adehabitatHR::mcp}.

The \code{splitBy} argument offers further control building
\code{SpatialPolygons}. If in your \code{DT}, you have multiple
temporal groups (e.g.: years) for example, you can provide the
name of the column which identifies them and build \code{SpatialPolygons}
for each individual in each year.

\code{group_polys} uses \code{build_polys} for grouping overlapping
polygons created from relocations.
}
\examples{
# Load data.table
library(data.table)

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# EPSG code for example data
utm <- 'EPSG:32736'

# Build polygons for each individual using kernelUD and getverticeshr
build_polys(DT, projection = utm, hrType = 'kernel',
            hrParams = list(grid = 60, percent = 95),
            id = 'ID', coords = c('X', 'Y'))

# Build polygons for each individual by year
DT[, yr := year(datetime)]
build_polys(DT, projection = utm, hrType = 'mcp', hrParams = list(percent = 95),
            id = 'ID', coords = c('X', 'Y'), splitBy = 'yr')

# Build polygons from SpatialPointsDataFrame
library(sp)
pts <- SpatialPointsDataFrame(coords = DT[, .(X, Y)],
                              proj4string = CRS(utm),
                              data = DT[, .(ID)]
)

build_polys(spPts = pts, hrType = 'mcp', hrParams = list(percent = 95))

}
\seealso{
\code{\link{group_polys}}

Other Build functions: 
\code{\link{build_lines}()}
}
\concept{Build functions}
