#' Summarises results of runs for parameter sets generated by a latin-hypercube
#'
#' Only to be applied for simulations that are stochastic, and responses are
#' supplied in the folder structure detailed in the R Journal paper, useful for
#' cases where the simulation is agent-based. Takes each parameter value set
#' generated by the hypercube in turn, and analyses the replicate simulation
#' results for that set. Produces a CSV file containing the parameters of the
#' run and the median of each simulation response for each run. In cases where,
#' for example, 300 runs have been performed for a parameter set, this file
#' will contain 300 rows for that set, each accompanied by the median of each
#' simulation response for that run. This file will be named as specified by
#' parameter LHC_ALL_SIM_RESULTS_FILE. This method can be performed for a number
#' of simulation timepoints, producing CSV files for each timepoint taken.
#'
#' @param FILEPATH Directory where the simulation runs of single CSV file can
#' be found
#' @param SPARTAN_PARAMETER_FILE Location of the file output by the
#' latin-hypercube sampling method. Note if providing a single CSV file
#' with parameter/response pairings, you do not need to provide this file,
#' and can thus enter this parameter as NULL.
#' @param PARAMETERS Array containing the names of the parameters of which
#' parameter samples will be generated
#' @param NUMSAMPLES The number of parameter subsets that were generated in the
#'  LHC design. Only required if analysing results provided within Folder
#'  structure setup.
#' @param NUMRUNSPERSAMPLE The number of runs performed for each parameter
#' subset. This figure is generated through Aleatory Analysis. Only required
#' if analysing results provided within Folder structure setup.
#' @param MEASURES Array containing the names of the output measures which are
#'  used to analyse the simulation
#' @param RESULTFILENAME Name of the simulation results file. In the current
#' version, XML and CSV files can be processed. If performing this analysis
#' over multiple timepoints, it is assumed that the timepoint follows the
#' file name, e.g. trackedCells_Close_12.csv.
#' @param ALTERNATIVEFILENAME In some cases, it may be relevant to read from
#' a further results file if the initial file contains no results. This
#' filename is set here. In the current version, XML and CSV files can be
#' processed.
#' @param OUTPUTCOLSTART Column number in the simulation results file where
#' output begins - saves (a) reading in unnecessary data, and (b) errors where
#' the first column is a label, and therefore could contain duplicates.
#' @param OUTPUTCOLEND Column number in the simulation results file where the
#' last output measure is.
#' @param LHC_ALL_SIM_RESULTS_FILE Name to be given to the CSV file that
#' summarises all simulation runs for all parameter sets
#' @param TIMEPOINTS Implemented so this method can be used when analysing
#' multiple simulation timepoints. If only analysing one timepoint, this
#' should be set to NULL. If not, this should be an array of timepoints,
#' e.g. c(12,36,48,60)
#' @param TIMEPOINTSCALE Implemented so this method can be used when analysing
#'  multiple simulation timepoints. Sets the scale of the timepoints being
#'  analysed, e.g. "Hours"
#'
#' @export
lhc_process_sample_run_subsets <- function(FILEPATH, SPARTAN_PARAMETER_FILE,
                                           PARAMETERS, NUMSAMPLES,
                                           NUMRUNSPERSAMPLE, MEASURES,
                                           RESULTFILENAME, ALTERNATIVEFILENAME,
                                           OUTPUTCOLSTART, OUTPUTCOLEND,
                                           LHC_ALL_SIM_RESULTS_FILE,
                                           TIMEPOINTS = NULL,
                                           TIMEPOINTSCALE = NULL) {

  # CREATE THE MEDIAN DISTRIBUTION OVER THE SET OF RUNS FOR EACH PARAMETER SET
  # CREATED IN LHC SAMPLING. SPARTAN VERSION 2: THESE MEDIANS ARE NOW STORED
  # IN ONE SINGLE FILE, NOT PER PARAMETER VALUE AS IN SPARTAN 1.0-1.3.
  # LATER PROCESSING NOW DEALS WITH THESE FILES ONLY. NOTE FROM SPARTAN 2,
  # THIS FILE CAN ONLY BE A CSV FILE - XML FILES OF THIS SIZE TAKE A LARGE
  # AMOUNT OF TIME TO PROCESS

  if (is.null(TIMEPOINTS)) {
    if (file.exists(FILEPATH)) {
      print("Generating Simulation Median Responses (process_sample_run_subsets)")

      # READ IN THE SPARTAN PARAMETER FILE
      LHCTABLE <- read.csv(paste(FILEPATH, "/", SPARTAN_PARAMETER_FILE,
                                 sep = ""), header = TRUE, check.names = FALSE)

      # NOW ALL THE MEDIANS ARE HELD TOGETHER,
      # ACCOMPANIED BY THEIR SIMULATION PARAMETERS BEING ANALYSED
      ALL_SIM_MEDIAN_RESULTS <- NULL

      for (k in 1:NUMSAMPLES) {
        print(paste("Summarising Responses for Parameter Set ", k, sep = ""))

        # GET THE PARAMETER VALUES FOR THIS SET
        PARAMROW <- as.numeric(LHCTABLE[k, ])

        MEDIAN_RESULTS <- getMediansSubset(paste(FILEPATH, "/", k, "/",
                                                 sep = ""),
                                           NUMRUNSPERSAMPLE, MEASURES,
                                           RESULTFILENAME,
                                           ALTERNATIVEFILENAME,
                                           OUTPUTCOLSTART, OUTPUTCOLEND)

        RUN_PARAMS <- NULL
        for (p in 1:length(PARAMROW)) {
          RUN_PARAMS <- cbind(RUN_PARAMS, array(as.numeric(PARAMROW[p]),
                                                dim = c(nrow(MEDIAN_RESULTS))))
        }
        # NOW BIND THESE PARAMS TO THE RESULTS
        PARAMRESULT <- cbind(RUN_PARAMS, MEDIAN_RESULTS)

        # ADD THIS TO THE LIST OF ALL MEDIANS BEING PROCESSED IN THIS ANALYSIS
        ALL_SIM_MEDIAN_RESULTS <- rbind(ALL_SIM_MEDIAN_RESULTS, PARAMRESULT)
      }

      # NOW OUTPUT ALL THE MEDIAN RESULTS TO THE SPECIFIED FILEPATH
      colnames(ALL_SIM_MEDIAN_RESULTS) <- cbind(t(PARAMETERS), t(MEASURES))

      # OUTPUT IF THE RESULTS ARE NOT BLANK
      if (!is.null(ALL_SIM_MEDIAN_RESULTS)) {
        RESULTSFILE <- paste(FILEPATH, "/", LHC_ALL_SIM_RESULTS_FILE, sep = "")
        print(paste("Writing Median Results to CSV File: ", RESULTSFILE,
                    sep = ""))
        write.csv(ALL_SIM_MEDIAN_RESULTS, RESULTSFILE, quote = FALSE,
                  row.names = FALSE)
      }
    } else {
      print("The directory specified in FILEPATH does not exist.")
      print("No analysis performed")
    }
    } else {
      # PROCESS EACH TIMEPOINT, AMEND FILENAMES AND RECALL THIS FUNCTION
      for (n in 1:length(TIMEPOINTS)) {

        current_time <- TIMEPOINTS[n]
        print(paste("PROCESSING TIMEPOINT: ", current_time, sep = ""))

        resultfileformat <- check_file_extension(RESULTFILENAME)
        SIMRESULTFILENAME <- paste(substr(RESULTFILENAME, 0,
                                          nchar(RESULTFILENAME) - 4),
                                   "_", current_time, ".",
                                   resultfileformat, sep = "")

        if (!is.null(ALTERNATIVEFILENAME)) {
          ALTERNATIVEFILENAMEFULL <- paste(substr(ALTERNATIVEFILENAME, 0,
                                                  nchar(ALTERNATIVEFILENAME) - 4),
                                           "_", current_time, ".",
                                           resultfileformat, sep = "")
        } else {
          ALTERNATIVEFILENAMEFULL <- ALTERNATIVEFILENAME
        }

        MEDIANSFILEFORMAT <- substr(LHC_ALL_SIM_RESULTS_FILE,
                                    (nchar(LHC_ALL_SIM_RESULTS_FILE) + 1) - 3,
                                    nchar(LHC_ALL_SIM_RESULTS_FILE))

        LHC_ALL_SIM_RESULTS_FILEFULL <- paste(substr(
          LHC_ALL_SIM_RESULTS_FILE, 0, nchar(LHC_ALL_SIM_RESULTS_FILE) - 4),
          "_", current_time, ".", MEDIANSFILEFORMAT, sep = "")

        # NOW CALL THIS FUNCTION AGAIN TO DO THE TIMEPOINTS - WE SET THE
        # TIMEPOINTS AND TIMEPOINTSCALE TO NULL NOW SO WE DONT END UP
        # BACK IN THIS ELSE

        lhc_process_sample_run_subsets(FILEPATH, SPARTAN_PARAMETER_FILE,
                                       PARAMETERS, NUMSAMPLES,
                                       NUMRUNSPERSAMPLE, MEASURES,
                                       SIMRESULTFILENAME,
                                       ALTERNATIVEFILENAME,
                                       OUTPUTCOLSTART, OUTPUTCOLEND,
                                       LHC_ALL_SIM_RESULTS_FILEFULL,
                                       NULL, NULL)
      }
  }
}

#' Summarises simulation behaviour for each parameter set, by median of
#' distribution of replicate runs
#'
#' Processes either the CSV file generated by lhc_process_sample_run_subsets
#' or one that has been supplied, going through each line of that file and
#' generating a file that summarises simulation responses under each parameter
#' set. This CSV file, named as specified by parameter LHCSUMMARYFILENAME,
#' will contain one row for each parameter set, accompanied by the median of
#' all the responses contained in the LHC_ALL_SIM_RESULTS_FILE. This method
#' can also be performed for a number of simulation timepoints
#' @param FILEPATH Directory where the simulation runs of single CSV file can
#' be found
#' @param PARAMETERS Array containing the names of the parameters of which
#' parameter samples will be generated
#' @param MEASURES Array containing the names of the output measures which are
#' used to analyse the simulation
#' @param LHC_ALL_SIM_RESULTS_FILE If lhc_process_sample_run_subsets is used
#' (i.e. results processed by folder structure), this will contain the output
#' of that method. If specifying responses using a single CSV file, this will
#' contain the name of that file (which should be in the FILEPATH folder).
#' @param LHCSUMMARYFILENAME Name of the LHC Summary file to be generated.
#' Contains each parameter set alongside the result gained when the simulation
#' was run under that criteria.
#' @param SPARTAN_PARAMETER_FILE Location of the file output by the
#' latin-hypercube sampling method (list of parameters). Note if providing a
#' single CSV file with parameter/response pairings, you do not need to provide
#' this file, and can thus enter this parameter as NULL.
#' @param TIMEPOINTS Implemented so this method can be used when analysing
#' multiple simulation timepoints. If only analysing one timepoint, this should
#' be set to NULL. If not, this should be an array of timepoints,
#' e.g. c(12,36,48,60)
#' @param TIMEPOINTSCALE Implemented so this method can be used when analysing
#'  multiple simulation timepoints. Sets the scale of the timepoints being
#'  analysed, e.g. "Hours"
#'
#' @export
lhc_generateLHCSummary <- function(FILEPATH, PARAMETERS, MEASURES,
                                   LHC_ALL_SIM_RESULTS_FILE,
                                   LHCSUMMARYFILENAME,
                                   SPARTAN_PARAMETER_FILE = NULL,
                                   TIMEPOINTS = NULL, TIMEPOINTSCALE = NULL) {

  # SPARTAN 2.0 - READS THIS INFORMATION FROM A SINGLE FILE, NOT ONE PER
  # PARAMETER SET AS IN PREVIOUS VERSIONS THEN PRODUCES A SUMMARY FILE
  # WITH THE PARAMETERS USED AND THE MEDIAN OF EACH MEASURE OVER THE
  # NUMBER OF REPLICATE RUNS

  if (is.null(TIMEPOINTS)) {
    if (file.exists(FILEPATH)) {
      # LHCSUMMARYFILENAME IS LHCSummary.csv for 1 timepoint
      # READ IN THE LHC DESIGN TABLE - NEED TO REFER TO THIS LATER AS
      # PARAMETERS ARE LISTED WITH THE MEDIAN RESULT SET
      print(join_strings_space(c("Generating LHC summary file from median",
            "simulation results (lhc_generateLHCSummary)")))

      # SUMMARY TABLE WILL STORE THE PARAMETERS USED IN THE RUN SET,
      # AND THE MEDIAN OUTPUT MEASURES, FOR EACH SET
      SUMMARYTABLE <- NULL

      # READ IN THE RESULT FILE - EITHER PROVIDED OR CONSTRUCTED BY THE FIRST
      # LHC METHOD IN SPARTAN
      LHC_ALL_SIM_RESULTS <- read.csv(paste(FILEPATH, "/",
                                            LHC_ALL_SIM_RESULTS_FILE,
                                            sep = ""),
                                      header = TRUE, check.names = FALSE)

      # NOW WE ARE PROCESSING A FILE WITH MULTIPLE RUNS OF THE SAME PARAMETER
      # SET. TO SAVE IMPORTING THE PARAMETER FILE (AS THIS MAY NOT ALWAYS BE
      # AVAILABLE), THIS READS THE PARAMETERS IN. THUS WE PUT A CHECK IN TO
      # MAKE SURE WE DO NOT PROCESS THE SAME SET OF PARAMETERS TWICE (WHICH
      # WE ASSUME ARE IN ORDER). WE DO THIS BY COMPARING THE SET WE HAVE JUST
      # PROCESSED TO THE ONE IN THE NEXT ROW, THUS IT IS IMPORTANT THIS FILE
      # IS IN ORDER
      string_last_params_seen <- ""

      # NOW PROCESS EACH ROW OF THE RESULTS FILE
      for (row in 1:nrow(LHC_ALL_SIM_RESULTS)) {
        # GET THE PARAMETERS FROM THE RESULT FILE - length parameters
        # put in such that the output measures are not included
        SIM_PARAMS <- LHC_ALL_SIM_RESULTS[row, 1:length(PARAMETERS)]
        # CONVERT TO A STRING TO DO THE COMPARISON DISCUSSED ABOVE
        STRING_SIM_PARAMS <- paste(SIM_PARAMS, collapse = " ")

        if (STRING_SIM_PARAMS != string_last_params_seen) {
          string_last_params_seen <- STRING_SIM_PARAMS

          # CONSTRUCTING A ROW, SO ADD THE PARAMETERS TO BEGIN THIS
          SUMMARY_SIM_ROW <- SIM_PARAMS

          # NOW TO SUBSET THE RESULTS (WHICH CONTAIN MULTIPLE SIM RESULTS
          # FOR THIS SET OF PARAMETERS) TO CALC MEDIANS
          PARAM_RESULT <- subset_results_by_param_value_set(
            PARAMETERS, LHC_ALL_SIM_RESULTS, SIM_PARAMS)

          # NOW WE CAN CALCULATE MEDIANS FOR EACH MEASURE
          for (l in 1:length(MEASURES)) {
            SUMMARY_SIM_ROW <- cbind(SUMMARY_SIM_ROW,
                                     median(PARAM_RESULT[[MEASURES[l]]]))

          }
          # NOW ADD THE ROW TO THE SET FOR ALL SIMULATIONS
          SUMMARYTABLE <- rbind(SUMMARYTABLE, SUMMARY_SIM_ROW)
        }
      }
      # NOW ADD HEADERS TO THIS INFORMATION AND WRITE TO FILE
      colnames(SUMMARYTABLE) <- c(PARAMETERS, MEASURES)

      SUMMARYRESULTSFILE <- paste(FILEPATH, "/", LHCSUMMARYFILENAME, sep = "")
      write.csv(SUMMARYTABLE, SUMMARYRESULTSFILE, quote = FALSE,
                row.names = FALSE)

      print(paste("LHC Summary file output to ", SUMMARYRESULTSFILE, sep = ""))
    } else {
      print("The directory specified in FILEPATH does not exist.
            No analysis performed")
    }
  } else {
    # PROCESS EACH TIMEPOINT, AMENDING FILENAMES AND RECALLING THIS FUNCTION
    for (n in 1:length(TIMEPOINTS)) {
      current_time <- TIMEPOINTS[n]
      print(paste("PROCESSING TIMEPOINT: ", current_time, sep = ""))

      lhc_all_sim_results_format <- check_file_extension(
        LHC_ALL_SIM_RESULTS_FILE)
      LHC_ALL_SIM_RESULTS_FULL <- paste(substr(
        LHC_ALL_SIM_RESULTS_FILE, 0, nchar(LHC_ALL_SIM_RESULTS_FILE) - 4),
        "_", current_time, ".", lhc_all_sim_results_format, sep = "")

      lhc_summary_format <- check_file_extension(LHCSUMMARYFILENAME)
      LHCSUMMARYFILENAME_FULL <- paste(substr(LHCSUMMARYFILENAME, 0,
                                              nchar(LHCSUMMARYFILENAME) - 4),
                                       "_", current_time, ".",
                                       lhc_summary_format, sep = "")

      lhc_generateLHCSummary(FILEPATH, PARAMETERS, MEASURES,
                             LHC_ALL_SIM_RESULTS_FULL,
                             LHCSUMMARYFILENAME_FULL,
                             SPARTAN_PARAMETER_FILE,
                             NULL, NULL)

    }
  }
}


#' Generate Partial Rank Correlation Coefficients for parameter/response pairs
#'
#' For each parameter, and each simulation output measure, calculates the
#' Partial Rank Correlation Coefficient between the parameter value and the
#' simulation results, giving a statistical measurement of any effect that
#' is present.  This is output to a CSV file. Can be performed for a
#' number of timepoints if required.
#'
#' @inheritParams lhc_generateLHCSummary
#' @param CORCOEFFSOUTPUTFILE Name of the generated CSV file generated
#'
#' @export
#'
lhc_generatePRCoEffs <- function(FILEPATH, PARAMETERS, MEASURES,
                                 LHCSUMMARYFILENAME, CORCOEFFSOUTPUTFILE,
                                 TIMEPOINTS = NULL, TIMEPOINTSCALE = NULL) {
  if (is.null(TIMEPOINTS)) {
    if (file.exists(FILEPATH)) {
      # LHCSUMMARYFILENAME IS LHCSummary.csv FOR 1 TIMEPOINT
      # CORCOEFFSOUTPUTFILE IS corCoefs.csv FOR 1 TIMEPOINT
      if (file.exists(paste(FILEPATH, "/", LHCSUMMARYFILENAME, sep = ""))) {
        LHCRESULTFILE <- read.csv(paste(FILEPATH, "/", LHCSUMMARYFILENAME,
                                        sep = ""),
                                  header = TRUE, check.names = FALSE)
        COEFFRESULTS <- NULL
        print("Generating Partial Rank Correlation Coefficients")
        print("(lhc_generatePRCoEffs)")

        # NEED TO GENERATE A COEFFICIENT FOR EACH PARAMETER BEING EXAMINED
        for (k in 1:length(PARAMETERS)) {
          PARAMNAME <- PARAMETERS[k]

          # GET COEFFICIENT SET
          COEFFDATA <- lhc_constructcoeff_dataset(LHCRESULTFILE, PARAMNAME,
                                                 PARAMETERS)
          # GET PARAMETER RESULT
          COEFFPARAMCOL <- LHCRESULTFILE[, PARAMETERS[k]]

          PARAMRESULTS <- NULL
          # GET MEASURE RESULTS AND CALCULATE COEFFICIENTS FOR EACH PARAMETER
          for (l in 1:length(MEASURES)) {
            COEFFMEASURERESULT <- LHCRESULTFILE[, MEASURES[l]]
            PARAMCOEFF <- pcor.test(COEFFPARAMCOL, COEFFMEASURERESULT,
                                    COEFFDATA, calc_method = c("s"))
            PARAMRESULTS <- cbind(PARAMRESULTS, PARAMCOEFF$estimate,
                                  PARAMCOEFF$p.value)
          }

          COEFFRESULTS <- rbind(COEFFRESULTS, PARAMRESULTS)
        }

        # NAME THE COLUMNS FOR EASE OF REFERENCE LATER
        COEFFRESULTSHEAD <- NULL
        for (l in 1:length(MEASURES)) {
          COEFFRESULTSHEAD <- cbind(COEFFRESULTSHEAD,
                                  (paste(MEASURES[l], "_Estimate", sep = "")),
                                  (paste(MEASURES[l], "_PValue", sep = "")))
        }

        # OUTPUT THE RESULTS FOR ALL PARAMETERS
        colnames(COEFFRESULTS) <- c(COEFFRESULTSHEAD)
        rownames(COEFFRESULTS) <- PARAMETERS

        COEFFSRESULTSFILE <- paste(FILEPATH, "/", CORCOEFFSOUTPUTFILE,
                                   sep = "")
        write.csv(COEFFRESULTS, COEFFSRESULTSFILE, quote = FALSE)

        print("File of Partial Rank Correlation Coefficients Generated.")
        print(paste("Output to ", COEFFSRESULTSFILE, sep = ""))
      } else {
        print("LHC Summary file cannot be found.")
        print("Are you sure you have run the method to generate it?")
      }
    } else {
      print("The directory specified in FILEPATH does not exist.")
      print("No Partial Rank Correlation Coefficients Generated")
    }
  } else {
    # PROCESS EACH TIMEPOINT, BY AMENDING THE FILENAMES AND
    # RECALLING THIS FUNCTION
    for (n in 1:length(TIMEPOINTS)) {
      current_time <- TIMEPOINTS[n]
      print(paste("Processing Timepoint: ", current_time, sep = ""))

      lhcsummaryfile_format <- check_file_extension(LHCSUMMARYFILENAME)
      LHCSUMMARYFILENAME_FULL <- paste(substr(LHCSUMMARYFILENAME, 0,
                                              nchar(LHCSUMMARYFILENAME) - 4),
                                       "_", current_time, ".",
                                       lhcsummaryfile_format, sep = "")

      corcoeffsfile_format <- check_file_extension(CORCOEFFSOUTPUTFILE)
      CORCOEFFSOUTPUTFILE_FULL <- paste(substr(CORCOEFFSOUTPUTFILE, 0,
                                             nchar(CORCOEFFSOUTPUTFILE) - 4),
                                      "_", current_time, ".",
                                      corcoeffsfile_format, sep = "")

      lhc_generatePRCoEffs(FILEPATH, PARAMETERS, MEASURES,
                           LHCSUMMARYFILENAME_FULL, CORCOEFFSOUTPUTFILE_FULL,
                           TIMEPOINTS = NULL, TIMEPOINTSCALE = NULL)
    }
  }
}
