% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_classification.R
\name{sits_classify}
\alias{sits_classify}
\alias{sits_classify.sits}
\alias{sits_classify.raster_cube}
\title{Classify time series or data cubes}
\usage{
sits_classify(
  data,
  ml_model,
  ...,
  filter_fn = NULL,
  multicores = 2,
  progress = TRUE
)

\method{sits_classify}{sits}(
  data,
  ml_model,
  ...,
  filter_fn = NULL,
  multicores = 2,
  progress = TRUE
)

\method{sits_classify}{raster_cube}(
  data,
  ml_model,
  ...,
  roi = NULL,
  filter_fn = NULL,
  impute_fn = sits_impute_linear(),
  start_date = NULL,
  end_date = NULL,
  memsize = 8,
  multicores = 2,
  output_dir,
  version = "v1",
  verbose = FALSE,
  progress = TRUE
)
}
\arguments{
\item{data}{Data cube.}

\item{ml_model}{R model trained by \code{\link[sits]{sits_train}}.}

\item{...}{Other parameters for specific functions.}

\item{filter_fn}{Smoothing filter to be applied (if desired).}

\item{multicores}{Number of cores to be used for classification.}

\item{progress}{Show progress bar?}

\item{roi}{Region of interest (see below)}

\item{impute_fn}{Impute function to replace NA.}

\item{start_date}{Start date for the classification.}

\item{end_date}{End date for the classification.}

\item{memsize}{Memory available for classification (in GB).}

\item{output_dir}{Directory for output file.}

\item{version}{Version of the output (for multiple
classifications).}

\item{verbose}{Print information about processing time?}
}
\value{
Predicted data (classified time series)
                          or a data cube with probabilities for each class.
}
\description{
This function classifies a set of time series or data cube given
a trained model prediction model created by \code{\link[sits]{sits_train}}.

SITS supports the following models:
\itemize{
 \item{support vector machines: } {see \code{\link[sits]{sits_svm}}}
 \item{random forests: }          {see \code{\link[sits]{sits_rfor}}}
 \item{extreme gradient boosting: } {see \code{\link[sits]{sits_xgboost}}}
 \item{multi-layer perceptrons: } {see \code{\link[sits]{sits_mlp}}}
 \item{1D CNN: } {see \code{\link[sits]{sits_tempcnn}}}
 \item{deep residual networks:}{see \code{\link[sits]{sits_resnet}}}
 \item{self-attention encoders:}{see \code{\link[sits]{sits_lighttae}}}
 }
}
\note{
The "roi" parameter defines a region of interest. It can be
   an sf_object, a shapefile, or a bounding box vector with
   named XY values ("xmin", "xmax", "ymin", "ymax") or
   named lat/long values ("lon_min", "lat_min", "lon_max", "lat_max")

   The "filter_fn" parameter specifies a smoothing filter to be applied to
   time series for reducing noise. Currently, options include
   Savitzky-Golay (see \code{\link[sits]{sits_sgolay}}) and Whittaker
   (see \code{\link[sits]{sits_whittaker}}).

   The "impute_fn" function is used to remove invalid or cloudy pixels
   from time series. The default is a linear interpolator, available
   in \code{\link[sits]{sits_impute_linear}}. Users can add their custom
   functions.

   The "memsize" and "multicores" parameters are used for multiprocessing.
   The "multicores" parameter defines the number of cores used for
   processing. The "memsize" parameter  controls the amount of memory
   available for classification. We recommend using a 4:1 relation between
   "memsize" and "multicores".

Please refer to the sits documentation available in
<https://e-sensing.github.io/sitsbook/> for detailed examples.
}
\examples{
if (sits_run_examples()) {
    # Example of classification of a time series
    # Retrieve the samples for Mato Grosso
    # train a random forest model
    rf_model <- sits_train(samples_modis_ndvi, ml_method = sits_rfor)

    # classify the point
    point_ndvi <- sits_select(point_mt_6bands, bands = c("NDVI"))
    point_class <- sits_classify(
        data = point_ndvi, ml_model = rf_model
    )
    plot(point_class)

    # Example of classification of a data cube
    # create a data cube from local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6",
        data_dir = data_dir
    )
    # classify a data cube
    probs_cube <- sits_classify(
        data = cube, ml_model = rf_model, output_dir = tempdir()
    )
    # label the probability cube
    label_cube <- sits_label_classification(
        probs_cube, output_dir = tempdir()
    )
    # plot the classified image
    plot(label_cube)
}

}
\author{
Rolf Simoes, \email{rolf.simoes@inpe.br}

Gilberto Camara, \email{gilberto.camara@inpe.br}
}
