% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-ou-fixed.R
\name{SimSSMOUFixed}
\alias{SimSSMOUFixed}
\title{Simulate Data from an Ornstein–Uhlenbeck Model
using a State Space Model Parameterization
for n > 1 Individuals (Fixed Parameters)}
\usage{
SimSSMOUFixed(
  n,
  mu0,
  sigma0,
  mu,
  phi,
  sigma,
  nu,
  lambda,
  theta,
  gamma_y = NULL,
  gamma_eta = NULL,
  x = NULL,
  type = 0,
  delta_t,
  time,
  burn_in = 0
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0}{Numeric matrix.
The covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{mu}{Numeric vector.
The long-term mean or equilibrium level
(\eqn{\boldsymbol{\mu}}).}

\item{phi}{Numeric matrix.
The rate of mean reversion,
determining how quickly the variable returns to its mean
(\eqn{\boldsymbol{\Phi}}).}

\item{sigma}{Numeric matrix.
The matrix of volatility
or randomness in the process
(\eqn{\boldsymbol{\Sigma}}).}

\item{nu}{Numeric vector.
Vector of intercepts for the measurement model
(\eqn{\boldsymbol{\nu}}).}

\item{lambda}{Numeric matrix.
Factor loading matrix linking the latent variables
to the observed variables
(\eqn{\boldsymbol{\Lambda}}).}

\item{theta}{Numeric matrix.
The measurement error covariance matrix
(\eqn{\boldsymbol{\Theta}}).}

\item{gamma_y}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the observed variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\mathbf{y}}}).}

\item{gamma_eta}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the latent variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}}).}

\item{x}{Numeric matrix.
The matrix of observed covariates in \code{type = 1} or \code{type = 2}.
The number of rows should be equal to \code{time + burn_in}.}

\item{type}{Integer.
State space model type.
See Details for more information.}

\item{delta_t}{Numeric.
Time interval (\eqn{\delta_t}).}

\item{time}{Positive integer.
Number of time points to simulate.}

\item{burn_in}{Positive integer.
Number of burn-in points to exclude before returning the results.}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers of length \code{t}.
\item \code{time}: A vector time points of length \code{t}.
\item \code{y}: A \code{t} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{t} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{t} by \code{j} matrix of values for the covariates.
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from an Ornstein–Uhlenbeck model
using a state space model parameterization
for \code{n > 1} individuals.
In this model,
the parameters are invariant across individuals.
}
\details{
\subsection{Type 0}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t}
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\mathbf{y}_{i, t}},
\eqn{\boldsymbol{\eta}_{i, t}},
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
are random variables
and
\eqn{\boldsymbol{\nu}},
\eqn{\boldsymbol{\Lambda}},
and
\eqn{\boldsymbol{\Theta}}
are model parameters.
\eqn{\mathbf{y}_{i, t}}
is a vector of observed random variables,
\eqn{\boldsymbol{\eta}_{i, t}}
is a vector of latent random variables,
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
is a vector of random measurement errors,
at time \eqn{t} and individual \eqn{i}.
\eqn{\boldsymbol{\nu}}
is a vector of intercepts,
\eqn{\boldsymbol{\Lambda}}
is a matrix of factor loadings,
and
\eqn{\boldsymbol{\Theta}}
is the covariance matrix of
\eqn{\boldsymbol{\varepsilon}}.

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\mu}
    -
    \boldsymbol{\eta}_{i, t}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t}
  }
where
\eqn{\boldsymbol{\mu}}
is the long-term mean or equilibrium level,
\eqn{\boldsymbol{\Phi}}
is the rate of mean reversion,
determining how quickly the variable returns to its mean,
\eqn{\boldsymbol{\Sigma}}
is the matrix of volatility
or randomness in the process, and
\eqn{\mathrm{d}\boldsymbol{W}}
is a Wiener process or Brownian motion,
which represents random fluctuations.
}

\subsection{Type 1}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t}
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right) .
  }

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\mu}
    -
    \boldsymbol{\eta}_{i, t}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Gamma}_{\boldsymbol{\eta}}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t}
  }
where
\eqn{\mathbf{x}_{i, t}} is a vector of covariates
at time \eqn{t} and individual \eqn{i},
and \eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}} is the coefficient matrix
linking the covariates to the latent variables.
}

\subsection{Type 2}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\Gamma}_{\mathbf{y}}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t}
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\boldsymbol{\Gamma}_{\mathbf{y}}} is the coefficient matrix
linking the covariates to the observed variables.

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\mu}
    -
    \boldsymbol{\eta}_{i, t}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Gamma}_{\boldsymbol{\eta}}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t} .
  }
}
}
\examples{
# prepare parameters
set.seed(42)
p <- k <- 2
iden <- diag(p)
n <- 5
mu0 <- c(-3.0, 1.5)
sigma0 <- iden
mu <- c(5.76, 5.18)
phi <- matrix(data = c(0.10, -0.05, -0.05, 0.10), nrow = p)
sigma <- matrix(
  data = c(2.79, 0.06, 0.06, 3.27),
  nrow = p
)
nu <- rep(x = 0, times = k)
lambda <- diag(k)
theta <- diag(x = 0.50, nrow = k)
delta_t <- 0.10
time <- 50
burn_in <- 0
gamma_y <- gamma_eta <- 0.10 * diag(k)
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    return(
      matrix(
        data = rnorm(n = k * (time + burn_in)),
        ncol = k
      )
    )
  }
)

# Type 0
ssm <- SimSSMOUFixed(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  type = 0,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# Type 1
ssm <- SimSSMOUFixed(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  gamma_eta = gamma_eta,
  x = x,
  type = 1,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# Type 2
ssm <- SimSSMOUFixed(
  n = n,
  mu0 = mu0,
  sigma0 = sigma0,
  mu = mu,
  phi = phi,
  sigma = sigma,
  nu = nu,
  lambda = lambda,
  theta = theta,
  gamma_y = gamma_y,
  gamma_eta = gamma_eta,
  x = x,
  type = 2,
  delta_t = delta_t,
  time = time,
  burn_in = burn_in
)

plot(ssm)

}
\references{
Chow, S.-M., Losardo, D., Park, J., & Molenaar, P. C. M. (2023).
Continuous-time dynamic models: Connections to structural equation models
and other discrete-time models.
In R. H. Hoyle (Ed.), Handbook of structural equation modeling (2nd ed.).
The Guilford Press.

Uhlenbeck, G. E., & Ornstein, L. S. (1930).
On the theory of the brownian motion.
\emph{Physical Review}, \emph{36}(5), 823–841.
\doi{10.1103/physrev.36.823}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{OU2SSM}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMOU}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{SimSSMVAR}()},
\code{\link{SimSSM}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{ou}
\keyword{sim}
\keyword{simStateSpace}
