% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sicure.R
\name{sicure.v}
\alias{sicure.v}
\title{Estimation of the vector of parameters in a single-index mixture cure model
with a vector of covariates}
\usage{
sicure.v(x_cov, time, delta, randomsearch = FALSE)
}
\arguments{
\item{x_cov}{A matrix or data frame giving the covariate values. Each row represents an individual and each column corresponds to a variable.}

\item{time}{A numeric vector giving the observed times.}

\item{delta}{A numeric vector giving the values of the uncensoring indicator, where 1 indicates that the event of interest has been observed and 0 indicates that the observation is censored.}

\item{randomsearch}{A logical value, \code{FALSE} by default, specifying whether a random search of the initial iterant is considered.}
}
\value{
A list with the following components:
\item{par}{A numeric vector of the estimated parameters. The last four correspond to the logarithms of the bandwidths.}
\item{value}{The value of the objective function (negative log-likelihood) at the estimated parameters.}
\item{si}{The estimated single-index variable.}
}
\description{
This function provides the estimation of the vector of parameters (\eqn{\boldsymbol{\theta_0}})  in a single-index mixture cure model with a vector of covariates (see Piñeiro-Lamas, 2024, Section 3.1, pages 37-38).
}
\details{
The vector of parameters, \eqn{\boldsymbol{\theta_0}}, is estimated by maximum likelihood, with the likelihood function also depending on four bandwidths.
Since the parameters cannot be obtained directly from the likelihood, the estimation is performed using numerical optimization with the Nelder-Mead method.
To begin the optimization, an initial iterant is required. To select a starting point for \eqn{\boldsymbol{\theta_0}}, a logistic regression model is fitted using the uncensoring indicator \code{delta} as the response variable and the covariates as predictors.
Regarding the initial bandwidths, for \eqn{h_3} the rule of thumb bandwidth selector used to estimate the density of the time variable is considered. For \eqn{h_1}, \eqn{h_2} and \eqn{h_4}, the rule of thumb bandwidth used to estimate the density of \eqn{\boldsymbol{\theta_{0,ini}}' \boldsymbol{X}}, where \eqn{\boldsymbol{\theta_{0,ini}}} is the initial estimate of \eqn{\boldsymbol{\theta_0}} and \eqn{\boldsymbol{X}} is the vector of covariates.
If \code{randomsearch = TRUE}, nine additional starting points are generated by adding a Gaussian noise (mean 0, standard deviation 0.25) to the logistic regression-based iterant.
Each of these ten starting points is used to estimate the vector of parameters of the single-index mixture cure model, and the one that gives place to the lowest negative log-likelihood value is selected.
}
\examples{
# Some artificial data
set.seed(123)
n <- 50
mix1a<-rnorm(n,mean=0,sd=1); mix1b<-rnorm(n,mean=0.25,sd=sqrt(2)); alf1<-rbinom(n,1,0.2)
mix2a<-rnorm(n,mean=0,sd=1); mix2b<-rnorm(n,mean=0.25,sd=sqrt(2)); alf2<-rbinom(n,1,0.2)
mix1<-alf1*mix1a+(1-alf1)*mix1b; mix2<-alf2*mix2a+(1-alf2)*mix2b
x_cov<-array(c(mix1,mix2),dim=c(n,2)) # Matrix of covariate values
theta<-c(1,1.2)
Z<-colSums(theta*t(x_cov))
y<-Z+rnorm(n,sd=sqrt(abs(Z))) # True lifetimes
# Probability of being susceptible
p_fun <- function(x){ 0.55 * exp(1.5*x+1.5)/(1+exp(1.5*x+1.5)) + 0.001 }
for (i in 1:n){
   w <- runif(1)
   if (w > p_fun(Z[i])) y[i] <- Inf
}
c<-rexp(n,rate=0.98) # Censoring values
t<-pmin(y,c) # Observed times
d = 1 * (y<=c) # Uncensoring indicator
\donttest{
suppressWarnings(sicure.v(x_cov, t, d))
}
}
\references{
Piñeiro-Lamas, B. (2024). High dimensional single-index mixture cure models [PhD thesis]. Universidade da Coruña. Available at \url{https://ruc.udc.es/dspace/handle/2183/37035}
}
\seealso{
\code{\link[sicure]{sicure.f}}, \code{\link[sicure]{sicure.vf}}
}
