% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pred_functions.R
\name{gen_marginal_samples}
\alias{gen_marginal_samples}
\title{Generate Marginal Samples of Predictive Distribution}
\usage{
gen_marginal_samples(
  mod,
  to_eval,
  nsamp = 200,
  fixed_x,
  n_eval_points,
  eval_range,
  display_progress = TRUE
)
}
\arguments{
\item{mod}{A \code{shrinkGPR} or \code{shrinkTPR} object representing the fitted Gaussian/t process regression model.}

\item{to_eval}{A character vector specifying the names of the covariates to evaluate. Can be one or two variables.}

\item{nsamp}{Positive integer specifying the number of posterior samples to generate. Default is 200.}

\item{fixed_x}{\emph{optional} data frame specifying a fixed covariate configuration. If provided, this configuration is used for
all nonswept covariates. If omitted, covariates are sampled from the training data.}

\item{n_eval_points}{Positive integer specifying the number of evaluation points along each axis. If missing, defaults to 100
for 1D and 30 for 2D evaluations.}

\item{eval_range}{\emph{optional} numeric vector (1D) or list of two numeric vectors (2D) specifying the range over which to evaluate
the covariates in \code{to_eval}. If omitted, the range is set to the range of the swept covariate(s) in the training data.}

\item{display_progress}{logical value indicating whether to display progress bars and messages during training. The default is \code{TRUE}.}
}
\value{
A list containing posterior predictive summaries over the evaluation grid:
\itemize{
\item \code{mean_pred}: A matrix (1D case) or array (2D case) of predicted means for each evaluation point and posterior sample.
\item \code{grid}: The evaluation grid used to generate predictions. A numeric vector (1D) or a named list of two vectors (\code{grid1}, \code{grid2}) for 2D evaluations.
}
}
\description{
\code{gen_marginal_samples()} generates model predictions over a grid of values for one or two specified covariates,
while filling in the remaining covariates either by drawing from the training data (if \code{fixed_x} is not provided)
or by using a fixed values for the remaining covariates (if \code{fixed_x} is provided). The result is a set of conditional
predictions that can be used to visualize the marginal effect of the selected covariates under varying input configurations.
}
\details{
This function generates conditional predictive surfaces by evaluating the fitted model across a grid of values for one or two selected covariates.
For each of the \code{nsamp} draws, the remaining covariates are either held fixed (if \code{fixed_x} is provided) or filled in by sampling a single row from the training data.
The selected covariates in \code{to_eval} are then varied across a regular grid defined by \code{n_eval_points} and \code{eval_range}, and model predictions are computed using \code{\link{calc_pred_moments}}.

The resulting samples represent conditional predictions across different covariate contexts, and can be used to visualize marginal effects, interaction surfaces, or predictive uncertainty.

Note that computational and memory requirements increase rapidly with grid size. In particular, for two-dimensional evaluations, the
kernel matrix scales quadratically with the number of evaluation points per axis. Large values of \code{n_eval_points} may lead to high
memory usage during prediction, especially when using a GPU. If memory constraints arise, consider reducing \code{n_eval_points}.
}
\examples{
\donttest{
if (torch::torch_is_installed()) {
  # Simulate data
  set.seed(123)
  torch::torch_manual_seed(123)
  n <- 100
  x <- matrix(runif(n * 2), n, 2)
  y <- sin(2 * pi * x[, 1]) + rnorm(n, sd = 0.1)
  data <- data.frame(y = y, x1 = x[, 1], x2 = x[, 2])

  # Fit GPR model
  res <- shrinkGPR(y ~ x1 + x2, data = data)

  # Generate posterior samples
  samps <- gen_posterior_samples(res, nsamp = 1000)

  # Plot the posterior samples
  boxplot(samps$thetas)
  }
}
}
