% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_ahels.R
\name{sample_ahels}
\alias{sample_ahels}
\title{Adapted Hypercube Evaluation of a Legacy Sample (ahels)}
\usage{
sample_ahels(
  mraster,
  existing,
  nQuant = 10,
  nSamp = NULL,
  threshold = 0.9,
  tolerance = 0,
  matrices = NULL,
  plot = FALSE,
  details = FALSE,
  filename = NULL,
  overwrite = FALSE
)
}
\arguments{
\item{mraster}{spatRaster. ALS metrics raster.}

\item{existing}{sf 'POINT'.  Existing plot network.}

\item{nQuant}{Numeric. Number of quantiles to divide covariates and samples into. Quantiles that do not
cover at least 1 percent of the area of interest will be excluded and be returned as \code{NA}.}

\item{nSamp}{Numeric. Maximum number of new samples to allocate.}

\item{threshold}{Numeric. Sample quantile ratio threshold. After the threshold \code{default = 0.9} is reached,
no additional samples will be added. Values close to 1 can cause the algorithm to continually loop.}

\item{tolerance}{Numeric. Allowable tolerance (<= 0.1 (10%)) around quantile density of 1. If \code{nSamp} is used samples will be
added until the \code{1 - tolerance} density is reached. If \code{threshold} is used, samples will be added until the
\code{threshold - tolerance} value is reached. This parameter allows the user to define a buffer around desired quantile densities
to permit the algorithm to not add additional samples if quantile density is very close to 1, or user-defined \code{threshold}.}

\item{matrices}{List. Quantile and covariance matrices generated from \code{calculate_pop(mraster = mraster, nQuant = nQuant)}.
Both \code{mraster} & \code{nQuant} inputs must be the same to supply the covariance matrix. Supplying the matrix allows users
with very large \code{mrasters} to pre-process the covariance matrix to avoid longer sampling processing times.}

\item{plot}{Logical. Plots samples of type \code{existing} (if provided; crosses) and \code{new} (circles) along with \code{mraster}.}

\item{details}{Logical. If \code{FALSE} (default) output is sf object of
systematic samples. If \code{TRUE} returns a list of sf objects where \code{tessellation}
is the tessellation grid for sampling, and \code{samples} are the systematic samples.}

\item{filename}{Character. Path to write output samples.}

\item{overwrite}{Logical. Choice to overwrite existing \code{filename} if it exists.}
}
\value{
Returns sf point object with existing samples and supplemental samples added by the ahels algorithm.
}
\description{
Perform the adapted Hypercube Evaluation of a Legacy Sample (ahels) algorithm using
existing site data and raster metrics. New samples are allocated based on quantile ratios between
the existing sample and covariate dataset.
}
\note{
Messages in the algorithm will state that samples have been added to under-represented quantiles. The number between
square brackets that follow represent the matrix row and column respectively that can be printed using \code{details = TRUE}.

In some cases, generally when a single metric is used as \code{mraster}, sampling ratios all be >= 1 before the
\code{nSamp} number of samples are allocated. The algorithm will stop in this scenario.

Special thanks to Dr. Brendan Malone for the original implementation of this algorithm.
}
\examples{
\dontrun{
#--- Load raster and existing plots---#
r <- system.file("extdata", "mraster.tif", package = "sgsR")
mr <- terra::rast(r)

e <- system.file("extdata", "existing.shp", package = "sgsR")
e <- sf::st_read(e)

sample_ahels(
  mraster = mr,
  existing = e,
  plot = TRUE
)

#--- supply quantile and covariance matrices ---#
mat <- calculate_pop(mraster = mr)

sample_ahels(
  mraster = mr,
  existing = e,
  matrices = mat,
  nSamp = 300
)
}
}
\references{
Malone BP, Minansy B, Brungard C. 2019. Some methods to improve the utility of conditioned Latin hypercube sampling. PeerJ 7:e6451 DOI 10.7717/peerj.6451
}
\seealso{
Other sample functions: 
\code{\link{sample_balanced}()},
\code{\link{sample_clhs}()},
\code{\link{sample_existing}()},
\code{\link{sample_nc}()},
\code{\link{sample_srs}()},
\code{\link{sample_strat}()},
\code{\link{sample_sys_strat}()},
\code{\link{sample_systematic}()}
}
\author{
Tristan R.H. Goodbody
}
\concept{sample functions}
