% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genthin.R
\name{thin_base}
\alias{thin_base}
\title{Base binomial thinning function.}
\usage{
thin_base(mat, designmat, coefmat, relative = TRUE, type = c("thin", "mult"))
}
\arguments{
\item{mat}{A numeric matrix of RNA-seq counts. The rows index the genes and
the columns index the samples.}

\item{designmat}{A design matrix. The rows index the samples and the columns
index the variables. The intercept should \emph{not} be included.}

\item{coefmat}{A matrix of coefficients. The rows index the genes and the
columns index the samples.}

\item{relative}{A logical. Should we apply relative thinning (\code{TRUE})
or absolute thinning (\code{FALSE}). Only experts should change
the default.}

\item{type}{Should we apply binomial thinning (\code{type = "thin"}) or
just naive multiplication of the counts (\code{type = "mult"}).
You should always have this set to \code{"thin"}.}
}
\value{
A matrix of new RNA-seq read-counts. This matrix has the signal
    added from \code{designmat} and \code{coefmat}.
}
\description{
Given a matrix of counts (\eqn{Y}) where \eqn{log_2(E[Y]) = Q},
a design matrix (\eqn{X}), and a matrix of coefficients (\eqn{B}),
\code{thin_diff} will generate a new matrix of counts such that
\eqn{log_2(E[Y]) = BX' + u1' + Q}, where \eqn{u} is some vector
of intercept coefficients. This function is used by all other
thinning functions. The method is
described in detail in Gerard (2020).
}
\examples{
## Simulate data from given matrix of counts
## In practice, you would obtain Y from a real dataset, not simulate it.
set.seed(1)
nsamp <- 10
ngene <- 1000
Y <- matrix(stats::rpois(nsamp * ngene, lambda = 100), nrow = ngene)
X <- matrix(rep(c(0, 1), length.out = nsamp))
B <- matrix(seq(3, 0, length.out = ngene))
Ynew <- thin_base(mat = Y, designmat = X, coefmat = B)

## Demonstrate how the log2 effect size is B
Bhat <- coefficients(lm(t(log2(Ynew)) ~ X))["X", ]
plot(B, Bhat, xlab = "Coefficients", ylab = "Coefficient Estimates")
abline(0, 1, col = 2, lwd = 2)

}
\references{
\itemize{
  \item{Gerard, D (2020). "Data-based RNA-seq simulations by binomial thinning." \emph{BMC Bioinformatics}. 21(1), 206. doi: \href{https://doi.org/10.1186/s12859-020-3450-9}{10.1186/s12859-020-3450-9}.}
}
}
\seealso{
\describe{
  \item{\code{\link{select_counts}}}{For subsampling the rows and columns
      of your real RNA-seq count matrix prior to applying binomial thinning.}
  \item{\code{\link{thin_diff}}}{For the function most users should
      be using for general-purpose binomial thinning.}
  \item{\code{\link{thin_2group}}}{For the specific application of
      thinning in the two-group model.}
  \item{\code{\link{thin_lib}}}{For the specific application of
      library size thinning.}
  \item{\code{\link{thin_gene}}}{For the specific application of
      total gene expression thinning.}
  \item{\code{\link{thin_all}}}{For the specific application of
      thinning all counts uniformly.}
}
}
\author{
David Gerard
}
