% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/safe_getMetadata.R
\name{safe_getMetadata}
\alias{safe_getMetadata}
\alias{rm_invalid_safe}
\alias{safe_isvalid}
\title{Get information from S2 file name or metadata}
\usage{
safe_getMetadata(
  s2,
  info = "all",
  format = "default",
  simplify = TRUE,
  abort = TRUE,
  allow_oldnames = FALSE
)

rm_invalid_safe(s2, req_res = c("10m", "20m", "60m"), allow_oldnames = FALSE)

safe_isvalid(
  s2,
  allow_oldnames = FALSE,
  check_file = TRUE,
  req_res = c("10m", "20m", "60m")
)
}
\arguments{
\item{s2}{Sentinel-2 products, which can be:
\itemize{
\item a list of products in the format \code{safelist} (see \linkS4class{safelist});
\item a vector of SAFE paths;
\item a vector of paths of xml product files with metadata.
If the product does not exist locally, the function can run only with
option \code{info = "nameinfo"} (see below).
}}

\item{info}{(optional) A character vector with the list of the metadata
which should be provided.
Accepted values are:
\itemize{
\item \code{"all"} (default): all the retrievable metadata are provided;
\item \code{"fileinfo"}: only the metadata obtained by scanning the file name
and product structure (without opening it with GDAL) are provided.
\item \code{"nameinfo"}: only the metadata obtained by scanning the file name
are provided (it is faster and there is no need to have downloaded
yet the file).
\item a vector of single specific information (one or more from the
followings):
\itemize{
\item \code{"name"} (SAFE name - this is always returned);
\item \code{"validname"} (TRUE or FALSE);
\item \code{"exists"} (TRUE or FALSE);
\item \code{"prod_type"} ('singlegranule' or 'product');
\item \code{"version"} (\code{'old'} or \code{'compact'});
\item \code{"tiles"} (vector with the tiles ID available in the product);
\item \code{"utm"} (vector with the UTM zones used in the product);
\item \code{"xml_main"} (name of the main XML file with metadata);
\item \code{"xml_granules"} (names of the XML with granule metadata);
\item \code{"level"} (\code{'1C'} or \code{'2A'});
\item \code{"creation_datetime"}, \code{"id_tile"}, \code{"mission"}, \code{"centre"},
\code{"file_class"}, \code{"id_orbit"}, \code{"orbit_number"},
\code{"sensing_datetime"}, \code{"id_baseline"}: metadata specific of
the product type and version (they are returned only if
obtainable for the specified input);
\item \code{"clouds"}, \code{"direction"}, \code{"orbit_n"}, \code{"preview_url"},
\code{"proc_baseline"}, \code{"level"}, \code{"sensing_datetime"},
\code{"nodata_value"}, \code{"saturated_value"}:
information retrieved from the metadata stored in the XML file;
\item \code{"res"}: resolutions with all the output products available;
\item \code{"jp2list"} (data.frame with the list of the JP2 band files -
asking for this info will cause \code{format} to be coerced to \code{"list"}).
}

Notice that the required info are returned only if available;
i.e., if some info requiring existing files are asked by the user, but
input SAFE do not exist, only info retrievable by the SAFE name are
returned.
}}

\item{format}{Output format, being one of the followings:
\itemize{
\item \code{"data.table"} and \code{"data.frame"}: a table with one row per \code{s2}
input and one column per required \code{info};
\item \code{"list"}: a list (one element per \code{s2} input) in which each element is
a list of the required \code{info};
\item \code{"vector"}: a list (one element per \code{info}) in which each element is
a named vector (with \code{s2} length and names) with the required \code{info};
\item \code{"default"} (default): \code{"vector"} if \code{info} is of length 1;
\code{"data.table"} otherwise.
}}

\item{simplify}{Logical parameter, which applies in case \code{s2} is of length 1:
in this case, if TRUE (default) and \code{format} is \code{"list"} or \code{"vector"},
a single \code{info} list or vector is returned;
if FALSE, a list of length 1 (containing the list or vector of the required
\code{s2} product) is returned.}

\item{abort}{Logical parameter: if TRUE (default), the function aborts
in case some inputs are not recognised, or if some files do not exists
(in case some \code{info} elements require the files to be present);
if FALSE, a warning is shown.}

\item{allow_oldnames}{Logical parameter: if TRUE, old (long) name products
are managed (metadata are returned, and they are considered valid;
if FALSE (default), they are considered as non-supported files.
Note that, from sen2r version 1.1.0, oldname products are no more supported
within processing chains, so this function is deprecated and no more
supported; moreover, it will be removed in next releases.}

\item{req_res}{Character: vector of variable length (0 to 3)
containing the names of the spatial resolution to be checked
(one or more among \code{"10m"}, \code{"20m"} and \code{"60m"}).
In case of level 2A-products, the existence of the JP2 files with the
required resolutions necessary for sen2r processing chains (spectral bands
and \code{SCL}) is checked, determining the result of the check.
Default is \code{c("10m","20m","60m")}, since Sen2Cor by default produces
all of these resolutions.
\code{NULL} can be used not to scan for JP2 content.
In case of level-1C products, in which each layer band is available in a
specific resolution, any of the previous values causes all JP2 layers to be
checked, while \code{NULL} causes no scan to be performed (as in the case of L2A).
In \code{safe_isvalid()}, this argument is ignored if \code{check_file = FALSE}.}

\item{check_file}{Logical: if TRUE (default), the content of the provided
paths is checked; if FALSE, only the validity of SAFE names is tested.}
}
\value{
\code{safe_getMetadata()} returns a data.table, a data.frame or a list
(depending on argument  \code{format}) with the output metadata;

\code{rm_invalid_safe()} returns a named vector (with the length of \code{s2}) with
TRUE if the \code{s2} product was removed, FALSE elsewhere.

\code{safe_isvalid()} returns a named vector (with the length of \code{s2}) with
TRUE if the product is a valid SAFE, FALSE if not.
}
\description{
The function \code{safe_getMetadata()} scans a Sentinel2 product
(main path or granule xml file) to retrieve information about the product.

The accessory function \code{rm_invalid_safe()} remove a SAFE archive in the case
it is not recognised by \code{safe_getMetadata()}.

The accessory function \code{safe_isvalid()} scan the SAFE name to understand
if it is a valid SAFE.
}
\note{
License: GPL 3.0
}
\examples{
# Define product name
s2_examplenames <- c(
  "S2A_MSIL1C_20190723T101031_N0208_R022_T32TNS_20190723T121220.SAFE",
  "S2A_MSIL1C_20190723T101031_N0208_R022_T32TNR_20190723T121220.SAFE"
)

# Return the information retrievable from the file names (files are not scanned)
safe_getMetadata(s2_examplenames, info="nameinfo")

# Return some specific information without scanning files
safe_getMetadata(s2_examplenames, info=c("level", "id_tile"))

# Return a single information without scanning files
# (in this case, the default output is a vector instead than a data.table)
safe_getMetadata(s2_examplenames, info="level")

# Check if the products are valid existing SAFE archives
safe_isvalid(s2_examplenames)

# Check if the product names are valid SAFE names
safe_isvalid(s2_examplenames, check_file = FALSE)
safe_isvalid("invalid_safe_name.SAFE", check_file = FALSE)

\dontrun{
# Download a sample SAFE archive (this can take a while)
s2_exampleurl <- c(
  "S2A_MSIL1C_20190723T101031_N0208_R022_T32TNS_20190723T121220.SAFE" =
    paste0("https://scihub.copernicus.eu/apihub/odata/v1/",
           "Products('19bbde60-992b-423d-8dea-a5e0ac7715fc')/$value")
)
s2_download(s2_exampleurl, outdir=tempdir())
s2_examplepath <- file.path(tempdir(), names(s2_exampleurl))

# Return all the available information
safe_getMetadata(s2_examplepath)

# Return some specific information
safe_getMetadata(s2_examplepath, info=c("clouds", "direction"))

# Return a single information
safe_getMetadata(s2_examplepath, info="nodata_value")

# Check if the downloaded SAFE is valid
safe_isvalid(s2_examplepath)

# Delete it if it is not recognised
rm_invalid_safe(s2_examplepath)

}
}
\references{
L. Ranghetti, M. Boschetti, F. Nutini, L. Busetto (2020).
"sen2r": An R toolbox for automatically downloading and preprocessing
Sentinel-2 satellite data. \emph{Computers & Geosciences}, 139, 104473. DOI:
\href{https://doi.org/10.1016/j.cageo.2020.104473}{10.1016/j.cageo.2020.104473},
URL: \url{http://sen2r.ranghetti.info/}.
}
\author{
Luigi Ranghetti, phD (2019) \email{luigi@ranghetti.info}
}
