# subfunctions that search output files for some information
# used by: seas

detect_error <- function(err, htmlmode = getOption("htmlmode")){
  # error parsing from .err or .err.html character vector
  #
  # err  character vector, content of output file
  #
  # returns an object of class x13messages which can be printed
  
  if (htmlmode == 1){
    ParseInfo <- function(openl, x){
      # find next closing tag
      clt <- grep("</p>", x)
      closel <- clt[clt >= openl][1]

      # extract info between tags
      z <- paste(x[openl:closel], collapse = "")

      # clean info
      z <- gsub("<p>.*</strong>", "", z) # remove trailing tag
      z <- gsub("</p>", "", z)           # remove closing tag 
      z <- gsub("&nbsp;", "", z)  
      z <- gsub("\\s+", " ", z)          # remove multiple space
      z <- gsub("^\\s", "", z)           # remove trailing space
      z <- gsub("<.+?>", "", z)           # remove inside HTML tags
      z
    }
  } else {
    ParseInfo <- function(openl, x){
      clt <- which(x == "  " | x == "" | grepl("^ [A-Z]{4}", x) | grepl("  \\*\\*", x)) 
      closel <- clt[clt > openl][1] - 1

      if (is.na(closel)){
        closel <- length(x)
      }

      z <- paste(x[openl:closel], collapse = "")
      z <- gsub("\\s+", " ", z)          # remove multiple space
      z <- gsub("^.*: ", "", z)    # remove trailing tag
      z <- gsub("^\\s", "", z)     # remove trailing space
      z
    }
  }

  z <- list()
  class(z) <- "x13messages"
  z$error <- sapply(grep("ERROR:", err), ParseInfo, x = err)
  z$warning <- sapply(grep("WARNING:", err), ParseInfo, x = err)
  # do not show this meaningless warning 
  # (its caused by default activation of spectrum)
  z$warning <- z$warning[!grepl("Spectrums are only generated for monthly series.", z$warning)]
  z$note <- sapply(grep("note:", err), ParseInfo, x = err)
  z
}

print.x13messages <- function(x, main = "\nMessages generated by X-13:\n"){
  if (length(x$error) == 0 & length(x$warning) == 0 & length(x$note) == 0 ){
    return(NULL)
  }
  cat(main)
  if (length(x$error) > 0){
    cat("Errors:\n")
    cat((paste(strwrap(paste("-", x$error), width = getOption("width") - 15, exdent = 2), 
               collapse = "\n")))
    cat("\n\n")
  }
  if (length(x$warning) > 0){
    cat("Warnings:\n")
    cat((paste(strwrap(paste("-", x$warning), width = getOption("width") - 15, exdent = 2), 
               collapse = "\n")))
    cat("\n\n")
  }
  if (length(x$note) > 0){
    cat("Notes:\n")
    cat((paste(strwrap(paste("-", x$note), width = getOption("width") - 15, exdent = 2), 
               collapse = "\n")))
    cat("\n")
  }
}



read_err <- function(basename){
  if (getOption("htmlmode") == 1){
    errtxt <- readLines(paste0(basename, "_err.html"))
  } else {
    errtxt <- readLines(paste0(basename, ".err"))
  }
  detect_error(errtxt)
}



drop_x13messages <- function(x, msg = "X-13 run failed\n\n", msgfun = stop, ontype = "error"){
  if (ontype != "all"){
    if (length(x[[ontype]]) == 0){
      return("")
    }
  }

  em <- paste0(capture.output(print(x, main = "")), collapse = "\n")
  if (em == "NULL") em <- ""
  msgfun(paste0(msg, em), call. = FALSE)
}


